﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/detail/Authentication/ldn_AuthenticationClient.h>
#include <nn/ldn/detail/Authentication/ldn_AuthenticationServer.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_Thread.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    struct ChallengeResponseAuthenticationServerBuffer
    {
        char threadStack[4 * 1024];
        char receive[2 * 1024];
        char send[2 * 1024];
    };

    struct ChallengeResponseAuthenticationClientBuffer
    {
        char receive[2 * 1024];
        char send[2 * 1024];
    };

    enum AuthenticationState
    {
        AuthenticationState_Unregistered,
        AuthenticationState_Registered,
        AuthenticationState_Processed,
    };

}}}} // namespace nn::ldn::detail::impl

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         チャレンジレスポンスによる認証サーバの実装です。
     *
     * @details
     * StaticAES を利用して他プラットフォームの接続を制限します。
     */
    class ChallengeResponseAuthenticationServer : public IAuthenticationServer
    {
    public:

        /**
         * @brief       インスタンスの生成に必要なバッファサイズを取得します。
         * @return      インスタンスの生成に必要なバッファサイズです。
         */
        static size_t GetRequiredBufferSize() NN_NOEXCEPT;

        /**
         * @brief       コンストラクタです。
         * @param[in]   buffer          認証に使用するバッファです。
         * @param[in]   bufferSize      buffer のバイトサイズです。
         */
        ChallengeResponseAuthenticationServer(void* buffer, size_t bufferSize) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~ChallengeResponseAuthenticationServer() NN_NOEXCEPT;

        virtual void BeginServer(
            const NetworkId& networkId, const Bit8 (&serverRandom)[RandomSize],
            IFrameReceiver* pReceiver, IFrameSender* pSender) NN_NOEXCEPT NN_OVERRIDE;

        virtual void EndServer() NN_NOEXCEPT NN_OVERRIDE;

        virtual int Register(MacAddress macAddress, bool accept) NN_NOEXCEPT NN_OVERRIDE;

        virtual void Unregister(int id) NN_NOEXCEPT NN_OVERRIDE;

        virtual void SetData(const void* data, size_t dataSize) NN_NOEXCEPT NN_OVERRIDE;

        virtual nn::os::Event& GetAuthenticationEvent() NN_NOEXCEPT NN_OVERRIDE;

        virtual bool GetAuthenticationHistory(
            AuthenticationHistroy* pOutHistory) NN_NOEXCEPT NN_OVERRIDE;

    private:

        struct AuthenticationClient
        {
            int32_t id;
            Bit8 status;
            bool accept;
            MacAddress macAddress;
            int64_t registeredAt;
        };

        void Expire() NN_NOEXCEPT;
        AuthenticationResult Judge(const void* data, size_t dataSize) const NN_NOEXCEPT;
        void Reply(const void* data, size_t dataSize) NN_NOEXCEPT;
        static void AuthThread(void* arg) NN_NOEXCEPT;

        nn::os::ThreadType m_Thread;
        nn::os::Event m_AuthEvent;
        nn::os::Event m_GetAuthEvent;
        nn::os::Event m_ReceivedEvent;
        mutable nn::os::Mutex m_ClientMutex;
        mutable nn::os::Mutex m_DataMutex;

        impl::ChallengeResponseAuthenticationServerBuffer* m_Buffer;
        IFrameReceiver* m_pReceiver;
        IFrameSender* m_pSender;
        FrameQueue m_ReceiveQueue;
        NetworkId m_NetworkId;
        AuthenticationClient m_Clients[StationCountMax];
        AuthenticationHistroy m_History;
        int32_t m_Counter;
        bool m_IsRunning;
        bool m_HasHistory;
    };

    /**
     * @brief         チャレンジレスポンスによる認証クライアントの実装です。
     */
    class ChallengeResponseAuthenticationClient : public IAuthenticationClient
    {
    public:

        /**
         * @brief       インスタンスの生成に必要なバッファサイズを取得します。
         * @return      インスタンスの生成に必要なバッファサイズです。
         */
        static size_t GetRequiredBufferSize() NN_NOEXCEPT;

        /**
         * @brief       コンストラクタです。
         * @param[in]   buffer          認証に使用するバッファです。
         * @param[in]   bufferSize      buffer のバイトサイズです。
         */
        ChallengeResponseAuthenticationClient(void* buffer, size_t bufferSize) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~ChallengeResponseAuthenticationClient() NN_NOEXCEPT;

        virtual void BeginClient(
            const Bit8 (&clientRandom)[RandomSize],
            IFrameReceiver* pReceiver, IFrameSender* pSender) NN_NOEXCEPT NN_OVERRIDE;

        virtual void EndClient() NN_NOEXCEPT NN_OVERRIDE;

        virtual void SetData(const void* data, size_t dataSize) NN_NOEXCEPT NN_OVERRIDE;

        virtual AuthenticationResult Authenticate(
            void* buffer, size_t* pOutSize, size_t bufferSize,
            MacAddress serverAddress, const NetworkId& networkId,
            const Bit8 (&serverRandom)[RandomSize]) NN_NOEXCEPT NN_OVERRIDE;

    private:

        nn::os::Event m_ReceivedEvent;

        impl::ChallengeResponseAuthenticationClientBuffer* m_Buffer;
        IFrameReceiver* m_pReceiver;
        IFrameSender* m_pSender;
        FrameQueue m_ReceiveQueue;
        bool m_IsRunning;
        bool m_IsAuthenticated;
    };

}}} // namespace nn::ldn::detail
