﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ldn/ldn_NetworkId.h>
#include <nn/ldn/detail/ldn_Version.h>

namespace nn { namespace ldn { namespace detail
{

    /**
     * @brief         Advertise の先頭に付加されるヘッダです。
     */
    struct AdvertiseHeader
    {
        //! ネットワーク識別子です。
        NetworkId       networkId;

        //! プロトコルバージョンです。
        Version         version;

        //! @ref AdvertiseFormat です。
        Bit8            format;

        //! ヘッダを除いたデータ本体のメッセージ長です。
        uint16_t        length;

        //! ペイロードの更新毎にインクリメントされるカウンタです。
        uint32_t        counter;
    };
    NN_STATIC_ASSERT(sizeof(AdvertiseHeader) == 40);

    /**
     * @brief         Advertise に付加されるヘッダのサイズです。
     */
    const size_t AdvertiseHeaderSize = sizeof(AdvertiseHeader);

    /**
     * @brief         Advertise に付加されるメッセージダイジェストのサイズです。
     */
    const size_t AdvertiseDigestSize = 32;

    /**
     * @brief         Advertise で配信できるペイロードの最大サイズです。
     */
    const size_t AdvertiseBodySizeMax = 1280;

    /**
     * @brief         カウンタの値から異常な Advertise とみなす閾値です。
     */
    const uint32_t AdvertiseRejectCounterThreshold = 256;

    /**
     * @brief         アクセスポイントから配信される Advertise のデータ構造です。
     */
    struct Advertise
    {
        //! ヘッダです。
        AdvertiseHeader header;

        //! メッセージダイジェストです。
        Bit8            digest[AdvertiseDigestSize];

        //! ペイロードです。
        Bit8            body[AdvertiseBodySizeMax];
    };
    NN_STATIC_ASSERT(sizeof(Advertise) == 1352);

    /**
     * @brief         Advertise の最小サイズです。
     */
    const size_t AdvertiseSizeMin = AdvertiseHeaderSize + AdvertiseDigestSize;

    /**
     * @brief         Advertise で配信できる最大サイズです。
     */
    const size_t AdvertiseSizeMax = sizeof(Advertise);

    /**
     * @brief         Advertise のうち、暗号化された部分の最大サイズです。
     */
    const size_t AdvertiseCipherTextSizeMax = AdvertiseDigestSize + AdvertiseBodySizeMax;

}}} // namespace nn::ldn::detail
