﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

 /**
 * @file
 * @brief       ローカルコンテンツ配信ライブラリの型定義と定数定義
 */

#include <nn/nn_Common.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace lcs
{
    /**
    * @brief       LCS ライブラリの初期化に必要なバッファサイズです。
    */
    const size_t RequiredBufferSize = 512 * 1024;

    /**
    * @brief       LCS ライブラリの初期化に使用するバッファに要求されるアライメントです。
    */
    const size_t RequiredBufferAlignment = 4 * 1024;

    /**
    * @brief       アプリケーションのディスプレイバージョンの最大サイズです。
    */
    const size_t DisplayVersionSizeMax = 16;

    /**
    * @brief       1 つのセッションで配信できるコンテンツの最大数です。
    */
    const int SharableContentsCountMax = 1;

    /**
    * @brief       1 つのセッションで受信するコンテンツの最大数です。
    */
    const int DownloadableContentsCountMax = SharableContentsCountMax + 1;

    /**
    * @brief       1 つのセッションに参加できる端末の最大数です。ホストを含みます。
    *
    * @details
    * セッションに参加できる端末の最大数です。この数字にはホストが含まれるため、
    * セッションには最大で @ref NodeCountMax - 1 台のクライアントが存在することになります。
    * なお、クライアントの最大数は @ref ClientCountMax として定義されています。
    *
    * ホストはセッションに接続できるクライアントの最大数を、必要に応じてさらに少なく制限できます。
    * そのためには、 OpenSession() に渡す SessionSettings 構造体の
    * nodeCountMax に @ref NodeCountMax よりも小さい値を指定します。
    */
    const int NodeCountMax = 8;

    /**
    * @brief       セッションに参加できるクライアントの最大数です。ホストを含みません。
    *
    * @details
    * セッションに参加できるクライアントの最大数です。この数字にはホストが含まれないため、
    * セッションには最大で ClientCountMax + 1 台の端末が存在することになります。
    * なお、ホストを含む端末の最大数は @ref NodeCountMax として定義されています。
    *
    * ホストはセッションに接続できるクライアントの最大数を、必要に応じてさらに少なく制限できます。
    * そのためには、 OpenSession() に渡す SessionSettings 構造体の
    * nodeCountMax に @ref NodeCountMax よりも小さい値を指定します。
    */
    const int ClientCountMax = NodeCountMax - 1;

    /**
    * @brief       スキャンで取得できるセッションの最大数です。
    */
    const int ScanResultCountMax = 16;

    /**
    * @brief       ユーザ名の最大バイト数です。終端文字 \0 を含まないサイズです。
    */
    const size_t UserNameBytesMax = 128;

    /**
    * @brief       通信品質の最大値で、通信品質が非常に良い状態です。
    */
    const size_t LinkLevelMax = 3;

    /**
    * @brief       通信品質の最小値で、通信品質が非常に悪い状態です。
    */
    const size_t LinkLevelMin = 0;


    /**
    * @brief       LCS ライブラリの状態を表す型です。
    *
    * @details
    * LCS ライブラリの状態を表す型です。
    *
    * 最初、LCS ライブラリは State_None 状態になっています。
    * この状態で Initialize() を実行すると State_Initialized 状態に遷移します。
    *
    * ホストは OpenSession() に成功すると State_Opened 状態に遷移します。
    * この状態では新しいクライアントの参加を募ることができます。
    * クライアントは JoinSession() でホストに接続すると State_Joined 状態に遷移します。
    *
    * ホストは参加者の募集が完了した後、
    * StartContentsShare() を実行することで State_Transferring 状態に遷移します。
    * クライアントもホストの状態遷移に同期して State_Transferring 状態に自動的に遷移します。
    *
    * その後は自動的に配信が実行されますが、
    * なんらかの理由で一時的に配信が中断されると State_Suspended 状態に遷移します。
    * State_Suspended 状態に遷移した場合は GetSuspendedReason() を実行して理由を確認し、
    * 問題を解決してから ResumeContentsShare() を実行すると State_Transferring 状態に戻ります。
    * ただし、 SuspendedReason によっては一旦 LCS ライブラリの使用を終了する必要があります。
    * その場合、 GetSessionContext() で取得した SessionContext を記憶しておいて、
    * 次回 LCS ライブラリを初期化した後で ResumeSession() を実行してください。
    * ResumeContentsShare() を実行した場合と同じく State_Transferring 状態に戻ります。
    *
    * コンテンツ配信に成功すると State_Completed 状態に遷移します。
    *
    * State_Opened, State_Joined, State_Transferring, State_Suspended, State_Completed
    * の各状態では LeaveSession() を使用することでセッションから離脱し、
    * State_Initialized 状態に戻ることができます。
    * 他の端末での操作や通信の失敗によって意図せずセッションから離脱した場合にも自動的に
    * State_Initialized 状態に遷移することがありますので、
    * LCS ライブラリの各関数では ResultInvalidState のハンドリングが必須です。
    * その場合、 GetContentsShareFailureReason() でコンテンツ配信に失敗した理由を取得できます。
    *
    * また、 State_None 以外の全ての状態から Finalize() を実行し、
    * State_None 状態に戻ることができます。
    */
    enum State
    {
        //! LCS ライブラリが初期化されていない状態です。
        State_None,

        //! LCS ライブラリが初期化された状態です。
        State_Initialized,

        //! ホストとしてセッションを構築した状態です。
        State_Opened,

        //! クライアントとしてホストが構築したセッションに参加した状態です。
        State_Joined,

        //! コンテンツを送信・受信・待機している状態です。
        State_Transferring,

        //! 問題が発生したため、セッションから切断されたり、コンテンツの配信に失敗した状態です。
        State_ContentsShareFailed,

        //! 解決しなければならない問題があるため、配信が中断されている状態です。
        State_Suspended,

        //! すべてのコンテンツの配信が完了した状態です。
        State_Completed,
    };

    /**
    * @brief       配信するコンテンツのバージョンを選択するポリシーです。
    *
    * @details
    * 配信するコンテンツのバージョンを選択するポリシーです。
    *
    * ContentsShareVersionPolicy_Latest を選択した場合、
    * セッションに参加する全ての端末の中で、最新のバージョンのコンテンツを配信します。
    * 参加者全員のコンテンツのバージョンが一致することが保証されます。
    *
    * ContentsShareVersionPolicy_Host を選択した場合、
    * ホストが所有するバージョンのコンテンツを配信します。
    * 最初からホストより新しいバージョンのコンテンツを所有するクライアントは受信者にならないため、
    * 配信が完了しても参加者全員のコンテンツのバージョンが一致することは保証されず、
    * ホストが所有するバージョン、あるいはそれよりも新しいコンテンツをもつことになります。
    *
    * なお、コンテンツの配信や起動のためにシステムの更新が必要と判断された場合、
    * 指定されたコンテンツだけではなくシステムも併せて配信されることがあります。
    * 配信するシステムのバージョンについてはこのポリシーで指定することができず、
    * LCS ライブラリが決定します。
    */
    enum ContentsShareVersionPolicy
    {
        //! 最新バージョンのコンテンツを配信します。
        ContentsShareVersionPolicy_Latest,

        //! ホストと同じバージョンのコンテンツを配信します。
        ContentsShareVersionPolicy_Host,
    };

    /**
    * @brief       ホストにおけるクライアント受付のポリシーです。
    *
    * @details
    * ホストにおけるクライアント受付のポリシーです。
    *
    * OpenSession() でセッションを構築すると、
    * デフォルトでは AcceptPolicy_AlwaysAccept が設定されています。
    * つまり、クライアント数の上限に達していない限り、
    * クライアントからの参加要求を全て自動的に受け入れます。
    *
    * クライアントの参加を打ち切るには SetClientAcceptPolicy() に
    * AcceptPolicy_AlwaysReject を指定してください。
    * 新しいクライアントの参加が自動的に拒否されるようになります。
    * なお、 SetClientAcceptPolicy() を実行しなくとも、
    * StartContentsShare() で配信を開始すると新しいクライアントは参加できなくなります。
    * ResumeSession() や ResumeContentsShare() で再合流することは可能です。
    */
    enum AcceptPolicy
    {
        //! 参加要求を全て受け入れます。
        AcceptPolicy_AlwaysAccept,

        //! 参加要求を全て拒否します。
        AcceptPolicy_AlwaysReject,
    };

    /**
    * @brief       コンテンツの種類を示すフラグです。
    */
    struct ContentsType
    {
        //! システムです。
        typedef nn::util::BitFlagSet<8, ContentsType>::Flag<0> System;

        //! アプリケーションの本体です。
        typedef nn::util::BitFlagSet<8, ContentsType>::Flag<1> Application;

        //! アプリケーションのパッチです。
        typedef nn::util::BitFlagSet<8, ContentsType>::Flag<2> Patch;
    };

    /**
    * @brief       コンテンツ配信時の状態です。
    *
    * @details
    * コンテンツ配信時の状態です。
    *
    * コンテンツの配信が始まったあとのセッションの状態です。
    * この状態はセッション全体の状態を示しており、
    * 送信・受信している端末の状態ではないことに注意してください。
    * 送信・受信している端末の状態については、Progress を参照してください。
    * 事前の準備が完了した後、コンテンツの配信を行うためにシステムの更新が必要な場合は、
    * SessionState_None から SessionState_ForceLup に遷移し、必要な端末でシステムの更新が完了後、
    * SessionState_ShareContents に遷移します。
    * コンテンツの配信を行うためのシステムの更新が必要ない場合は、
    * SessionState_None から SessionState_ShareContents に遷移します。
    * SessionState_ShareContents に遷移後は、端末でユーザの操作が必要になった場合に、
    * SessionState_Waiting に遷移し、必要な操作が行われた後は、
    * 再び SessionState_ShareContents に遷移します。
    */
    enum SessionState
    {
        //! 事前の準備をしている段階で、コンテンツの配信が行われていません。
        SessionState_None,

        //! コンテンツの情報を共有するために必須となるシステムの更新を行っている状態です。
        SessionState_ForceLup,

        //! コンテンツの配信や、コンテンツの起動に必要なシステムの更新を行っている状態です。
        SessionState_ShareContents,

        //! コンテンツの配信を一時停止している状態です。
        SessionState_Waiting,
    };

    /**
    * @brief       コンテンツ配信の役割です。
    *
    * @details
    * コンテンツ配信の役割です。
    *
    * 自身がそのコンテンツを送信中であれば TransferRole_Sender に、
    * 受信中であれば TransferRole_Receiver になります。
    * 送信も受信もしていないものの、他の端末の送受信完了を待機する必要がある場合、
    * あるいはまだそのコンテンツの配信を開始していない場合は TransferRole_None に設定されます。
    *
    * LCS ライブラリではホスト・クライアントの役割と送信者・受信者が対応しません。
    * そのため、ホストが受信者になることもあればクライアントが送信者になることもあります。
    */
    enum TransferRole
    {
        //! 他の端末の配信が完了するのを待機しています。
        TransferRole_None,

        //! コンテンツを送信しています。
        TransferRole_Sender,

        //! コンテンツを受信しています。
        TransferRole_Receiver,
    };

    /**
    * @brief       コンテンツの配信に失敗した理由です。
    *
    * @details
    * コンテンツの配信に失敗した理由です。
    * コンテンツの配信に失敗したとき、 GetContentsShareFailureReason() で取得できます。
    */
    enum ContentsShareFailureReason
    {
        ContentsShareFailureReason_None,                        //!< 配信に失敗していないため、理由を取得できません。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_FlightMode,                  //!< 自身のフライトモードによって通信が途切れてセッションから離脱しました。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_Sleep,                       //!< 自身のスリープによって通信が途切れてセッションから離脱しました。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_CommunicationError,          //!< 通信エラーが原因でコンテンツの配信に失敗しました。
                                                                //!< 他の端末で通信エラーが発生してもこの理由は返らず、
                                                                //!< 他の端末の通信エラーによって連鎖的に自身も
                                                                //!< コンテンツの配信を継続できなくなった場合は
                                                                //!< ContentsShareFailureReason_NodeLeaved が返ります。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_NodeLeaved,                  //!< 自分以外の端末が離脱したことにより、セッションを維持できなくなりました。
                                                                //!< 自分以外のある端末が何らかの理由でセッションから離脱したとしても、
                                                                //!< 他の端末は可能な限りセッションを維持してコンテンツ配信を継続しますが、
                                                                //!< その端末がいなければコンテンツの配信を継続できないと判断された場合、
                                                                //!< 端末の離脱と同時に自動的にセッションが破棄されることがあります。
                                                                //!< このとき、最初に離脱した端末以外の端末で取得できる理由がこれに相当します。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_CannotUpdateSystem,          //!< システムのバリエーションが異なり、システムの更新ができないことで
                                                                //!< コンテンツの配信ができないときに発生します。
                                                                //!< この場合、インターネットからなどの方法でシステムを更新する必要があります。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_ContentsCorrupted,           //!< 受信したコンテンツが破損を検知したときや、改竄されていることを検知したときに発生します。
                                                                //!< この理由による失敗のときはリトライすることで正しいコンテンツを受信できる可能性があります。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_NotExistDownloadContents,    //!< コンテンツが受信できない時に発生します。
                                                                //!< 具体的には以前新しいパッチを保持しており、起動必須バージョンに満たないパッチが配信される場合や、
                                                                //!< 子機が配信される際に親機のパッチも必要なのに送信者が親機のパッチを持っていない場合などがあります。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_BuiltInStorageError,         //!< 内蔵ストレージへのアクセスに失敗したため、コンテンツの配信に失敗したときに発生します。
                                                                //!< 他の端末でアクセスエラーが発生してもこの理由は返らず、
                                                                //!< 他の端末のアクセスエラーによって連鎖的に自身もコンテンツの配信を継続できなくなった場合は
                                                                //!< ContentsShareFailureReason_NodeLeaved が返ります。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_GameCardError,               //!< ゲームカードへのアクセスに失敗したため、コンテンツの配信に失敗したときに発生します。
                                                                //!< 他の端末でアクセスエラーが発生してもこの理由は返らず、
                                                                //!< 他の端末のアクセスエラーによって連鎖的に自身もコンテンツの配信を継続できなくなった場合は
                                                                //!< ContentsShareFailureReason_NodeLeaved が返ります。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_SdCardError,                 //!< SD カードへのアクセスに失敗したため、コンテンツの配信に失敗したときに発生します。
                                                                //!< 他の端末でアクセスエラーが発生してもこの理由は返らず、
                                                                //!< 他の端末のアクセスエラーによって連鎖的に自身もコンテンツの配信を継続できなくなった場合は
                                                                //!< ContentsShareFailureReason_NodeLeaved が返ります。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_StorageError,                //!< NAND / ゲームカード / SD カードいずれかへのアクセスに失敗したため、
                                                                //!< コンテンツの配信に失敗したときに発生します。
                                                                //!< 他の端末でアクセスエラーが発生してもこの理由は返らず、
                                                                //!< 他の端末のアクセスエラーによって連鎖的に自身もコンテンツの配信を継続できなくなった場合は
                                                                //!< ContentsShareFailureReason_NodeLeaved が返ります。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_UnknownError,                //!< システムで想定外のエラーが発生したため、コンテンツの配信に失敗したときに発生します。
                                                                //!< 通常このエラーは発生しません。
                                                                //!< このエラーが発生した場合は LCS ライブラリの不具合となります。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_UnshareableContents,         //!< LCS では送受信できないコンテンツが検出されたときに発生します。
                                                                //!< 5.0.0 以上のシステムでこの失敗は発生します。
                                                                //!< 5.0.0 未満のシステムでは送受信できないコンテンツが検出されたときは、
                                                                //!< ContentsShareFailureReason_NotExistDownloadContents が発生します。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。

        ContentsShareFailureReason_NotEnoughStorageSpace,       //!< LCS で容量が不足し、コンテンツを受信できなくなったときに発生します。
                                                                //!< 5.0.0 以上のシステムでこの失敗は発生します。
                                                                //!< 5.0.0 未満のシステムで受信できなくなったときは、
                                                                //!< ContentsShareFailureReason_NotExistDownloadContents が発生します。
                                                                //!< この失敗理由は、ホスト・クライアント双方で発生します。
    };

    /**
    * @brief       コンテンツ配信が中断された理由です。
    *
    * @details
    * コンテンツ配信が中断された理由です。
    *
    * @ref State_Suspended 状態に遷移したとき、
    * GetSuspendedReason() で SuspendedReason を取得できます。
    * 問題を解決した後、 ResumeContentsShare() や ResumeSession() で配信を再開できます。
    */
    enum SuspendedReason
    {
        SuspendedReason_None,                       //!< 配信が中断されていません。

        SuspendedReason_IncompatibleContentsInfo,           //!< コンテンツの情報に互換性がなく、本体の更新が必要です。
                                                            //!< この本体更新後、実際にコンテンツが配信できるかの判断を行いますので、
                                                            //!< この本体更新を行ってもパッチや子機などのコンテンツを受信できるとはかぎりません。
                                                            //!< 本体更新を行っても良い場合は ResumeContentsShare() で配信を再開してください。
                                                            //!< この中断は本体更新データを受信前に発生します。

        SuspendedReason_NeedTerminateApplication,           //!< アプリケーションのパッチを受信するので、アプリケーションを終了してください。
                                                            //!< 終了するアプリケーションは Progress で受信しているコンテンツの
                                                            //!< アプリケーション ID から判断できます。
                                                            //!< 該当するアプリケーションを終了した後、配信を再開してください。
                                                            //!< 該当するアプリケーションが起動されていない場合はそのまま再開してください。
                                                            //!< パッチの受信が開始されたときに発生します。

        SuspendedReason_EulaRequired,                       //!< EULA の同意が必要です。
                                                            //!< 必要な手続きを完了してから配信を再開してください。
                                                            //!< 再開をされると本体更新が適用され、再起動を要求されますので、
                                                            //!< アプリケーションが起動している場合はこの中断時にユーザに終了確認をしてください。
                                                            //!< この中断は本体更新データを受信した後の本体更新適用前に発生します。

        SuspendedReason_RebootRequired,                     //!< 本体の再起動が必要です。
                                                            //!< GetSessionContext() で再合流のための情報を取得してから
                                                            //!< SuspendSession() でセッションを抜け、
                                                            //!< 本体を再起動してから ResumeSession() でセッションに再合流してください。
                                                            //!< この中断は本体更新適用後に発生します。

        SuspendedReason_StorageSpaceNotEnough,              //!< 受信するコンテンツをインストールするための空き容量が不足しています。
                                                            //!< 不要なコンテンツを削除するなどして十分な空き容量を確保した後、
                                                            //!< ResumeContentsShare() で再開してください。
                                                            //!< コンテンツを削除するために一時的に LCS ライブラリを終了するときは、
                                                            //!< GetSessionContext() で再合流のための情報を取得してから
                                                            //!< SuspendSession() でセッションを抜け、データの整理を行ってから
                                                            //!< ResumeSession() でセッションに再合流してください。
                                                            //!< 必要な空き容量は GetRequiredStorageSize() で取得できます。
                                                            //!< この中断はパッチや子機などの受信前に発生します。

        SuspendedReason_StorageSpaceNotEnoughOnRequredNode, //!< 受信するコンテンツをインストールするための空き容量が不足しています。
                                                            //!< 不要なコンテンツを削除するなどして十分な空き容量を確保した後、配信を再開してください。
                                                            //!< 必要な空き容量は GetRequiredStorageSize() で取得できます。
                                                            //!< この中断はパッチや子機などの受信前に発生します。
                                                            //!< この中断が発生した端末はセッションの維持に必須の端末なので、
                                                            //!< 一時的に LCS ライブラリを終了してデータの整理を行うことができません。
                                                            //!< SuspendSession() を使用してもセッションが破棄されるので注意してください。
    };

    /**
    * @brief       ホストが構築するセッションの設定です。
    *
    * @details
    * ホストが構築するセッションの設定です。OpenSession() の引数として使用します。
    *
    * @a applications には配信したいアプリケーションのアプリケーション ID を、
    * @a applicationCount には配信したいアプリケーションの数を指定します。
    * 1 つのセッションで配信できるアプリケーションの最大数は @ref SharableContentsCountMax です。
    * ここで指定したアプリケーションだけではなく、そのパッチや、
    * アプリケーションの起動に必要なシステムや依存コンテンツも配信対象となります。
    * アプリケーション本体の配信が禁止されている場合はパッチだけが配信対象になります。
    *
    * @a nodeCountMax にはセッションに参加させる端末の最大数を
    * 1 以上 NodeCountMax 以下の整数で指定してください。
    *
    * @a contentsShareVersionPolicy は配信するコンテンツのバージョンを選択するポリシーです。
    * 詳細については ContentsShareVersionPolicy のリファレンスを参照してください。
    */
    struct SessionSettings
    {
        //! 配信するアプリケーションのリストです。
        nn::ncm::ApplicationId applications[SharableContentsCountMax];

        //! 配信するアプリケーションの個数です。
        int8_t applicationCount;

        //! セッションに参加できる端末の最大数です。
        int8_t nodeCountMax;

        //! 配信するコンテンツのバージョンを選択するポリシーです。
        Bit8 contentsShareVersionPolicy;

        //  予約領域です。
        char _reserved[5];
    };

    /**
    * @brief       配信されるコンテンツの情報です。
    *
    * @details
    * 配信されるコンテンツの情報です。
    *
    * type には ContentType が格納されており、このコンテンツが、
    * システム・アプリケーション・パッチのいずれに対応しているかがわかります。
    * アプリケーションとパッチの両方を配信できる場合、両方のフラグがセットされます。
    * システムの場合には後述の displayVersion, attributeFlag, applicationId
    * には正しい値が格納されませんので注意してください。
    *
    * displayVersion には、配信が予定されているバージョンが入ります。
    * 配信される予定のコンテンツがない場合は NULL 文字が入ります。
    *
    * attributeFlag はアプリケーションの属性です。
    * 体験版や子機などの属性がビットフラグで格納されています。
    * 詳細については nn::ns::AttributeFlag を参照してください。
    *
    * applicationId にはアプリケーション ID が格納されています。
    */
    struct ContentsInfo
    {
        //! コンテンツの種類です。
        nn::util::BitFlagSet<8, ContentsType> contentsFlag;

        //! アプリケーションの属性です。
        Bit32 attributeFlag;

        //! アプリケーションのディスプレイバージョンです。
        char displayVersion[DisplayVersionSizeMax];

        //! アプリケーション ID です。
        nn::ncm::ApplicationId applicationId;
    };

    /**
    * @brief       セッションの情報です。
    *
    * @details
    * セッションの情報です。セッションに参加する前は Scan() で、
    * セッションに参加した後は GetSessionInfo() で取得できます。
    *
    * hostUserName にはホストのユーザー名が格納されています。
    * これは、ホストが LCS ライブラリを初期化する際に Config::SetName() で設定した名前です。
    *
    * contents はそのセッションで配信されるコンテンツのリストです。
    * コンテンツの数は contentsCount に格納されています。
    * 既にクライアントが所有しておりダウンロードが不要なコンテンツも除外されません。
    * 取得できる情報の詳細については ContentsInfo を参照してください。
    *
    * nodeCountMax はセッションに参加できる端末の最大数で、
    * nodeCount はその時点でセッションに参加している端末の数です。この数はホストを含みます。
    * 最大数にはついては NodeCountMax や ClientCountMax のリファレンスも参照してください。
    *
    * linkLevel は通信品質を表します。
    * この値は Scan() で得られた SessionInfo のみで参照するようにしてください。
    * セッションに参加した後に GetSessionInfo() で得られる SessionInfo では、
    * linkLevel は不定値が入ります。
    * linkLevel の最大値は LinkLevelMax となり、最小値は LinkLevelMin となります。
    *
    * isContentShareSucceeded はこのセッションでコンテンツの配信が 1 回でも成功したかを表します。
    * ここでのコンテンツにはシステムは含まれず、アプリケーション本体・パッチの配信が
    * 1 回でも成功した場合に true となります。ここでの成功はアプリケーション本体・パッチを
    * 必要とする端末全てに行きわたったことを示すのではないことに注意してください。
    * アプリケーション本体・パッチの配信を行っていない場合や、
    * システムの配信のみが成功した場合は false となります。
    *
    * この型の変数は大きなメモリ領域を必要としますのでスタック上に確保することは推奨されません。
    * スタック上に確保する場合には実行時にスタックが溢れないように注意してください。
    */
    struct SessionInfo
    {
        //! ホストの名前です。
        char hostUserName[UserNameBytesMax + 1];

        //  予約領域です。
        char _reserved1[7];

        //! 配信されるコンテンツの情報です。
        ContentsInfo contents[SharableContentsCountMax];

        //! 配信されるアプリケーションの個数です。
        int8_t contentsCount;

        //! セッションに参加できる最大端末数です。
        int8_t nodeCountMax;

        //! セッションに参加している端末数です。
        int8_t nodeCount;

        //! 通信品質です。
        int8_t linkLevel;

        //! このセッションでのコンテンツ配信が成功したかを示すフラグです。
        bool isContentShareSucceeded;

        //  予約領域です。
        char _reserved2[3];

        char _internal[64];
    };

    /**
    * @brief       端末の情報です。
    *
    * @details
    * 取得できる端末の情報です。GetNodes() で取得します。
    *
    * index には参加したセッションで一意に与えられるインデックスが格納されています。
    * インデックスは 0 以外の値となります。
    *
    * userName には端末のユーザー名が格納されています。これは、
    * 端末が LCS ライブラリを初期化する際に Config::SetName() で設定した名前です。
    *
    *
    */
    struct NodeInfo
    {
        //! 端末のインデックスです。
        uint32_t index;

        //  予約領域です。
        char _reserved1[4];

        //! 端末の名前です。
        char userName[UserNameBytesMax + 1];

        //  予約領域です。
        char _reserved2[7];
    };

    /**
    * @brief       現在配信しているコンテンツの進捗情報です。
    *
    * @details
    * 現在配信しているコンテンツの進捗情報です。
    *
    * info を確認することで現在配信しているコンテンツがわかります。
    * ホストが SessionSettings に指定したコンテンツ以外に、システムを配信することがあります。
    * 取得できる情報の詳細については ContentsInfo のリファレンスを参照してください。
    *
    * size には現在送受信しているコンテンツのサイズが、
    * downloadedSize には送信あるいは受信が完了したサイズがバイト単位で格納されます。
    * transferRole == TransferRole_Sender かつ、size == downloadedSize となった場合、
    * 相手の受信者が State_Suspended 状態にあり、ResumeContentsShare() を待っている状態です。
    * つまり、コンテンツの送信自体は size == downloadedSize - 1 の時点で完了しています。
    * 受信者がコンテンツの受信中に ResumeContentsShare() を行っていた場合は、
    * size == downloadedSize になることなく、transferRole == TransferRole_None に遷移します。
    *
    * transferRole には TransferRole の値が格納されています。
    * 詳細については TransferRole のリファレンスを参照してください。
    *
    * LCS ライブラリでは複数のコンテンツを配信したり、1 つのコンテンツであっても複数回送信したり、
    * 自身の受信が完了してから他の端末を相手に送信を開始したりすることがあるため、
    * info, size, transferrole は可変であり downloadedSize は単調増加ではありません。
    * 具体的には一度送受信が完了してから info, size, transferRole が変化し、
    * downloadedSize が 0 に戻ることがあります。
    * 配信の完了を検知するには State を監視してください。
    */
    struct Progress
    {
        //! コンテンツの情報です。
        ContentsInfo info;

        //! 送受信を行うコンテンツのサイズです。
        uint64_t size;

        //! 送受信が完了したサイズです。
        uint64_t downloadedSize;

        //! コンテンツの配信におけるこの端末の役割です。
        Bit8 transferRole;

        //  予約領域です。
        char _reserved[3];

        //! 送信・受信を行っているときの相手のインデックです。
        //! transferRole が TransferRole_None の場合は無効です。
        //! transferRole が TransferRole_Sender の場合、インデックスには受信者のものが入ります。
        //! transferRole が TransferRole_Receiver の場合、インデックスには送信者者のものが入ります。
        uint32_t transferIndex;
    };

    /**
    * @brief       参加している端末のコンテンツの受信進捗情報です。
    *
    * @details
    * 参加している端末のコンテンツの受信進捗情報です。GetNodeProgress() で取得します。
    *
    * contentCount には受信する予定コンテンツの数が、
    * downloadedContentCount には受信が完了したコンテンツの数が格納されます。
    * contentCount の最大値は DownloadableContentsCountMax となります。
    */
    struct NodeProgress
    {
        //! 端末が受信する予定のコンテンツの数です。
        int8_t contentCount;

        //! 端末が受信したコンテンツの数です。
        int8_t downloadedContentCount;
    };

    /**
    * @brief       セッションへの再合流に必要な情報です。
    *
    * @details
    * セッションへの再合流に必要な情報です。
    * GetSessionContext() で取得し、 ResumeSession() の引数として使用します。
    * アプリケーションはこの構造体の内容を参照あるいは変更してはなりません。
    */
    struct SessionContext
    {
        char _context[1024];
    };

}} // end of namespace nn::lcs

