﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief API declaration for Hand Analysis
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/irsensor/irsensor_Result.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>
#include <nn/irsensor/irsensor_HandAnalysisConstants.h>
#include <nn/irsensor/irsensor_HandAnalysisTypes.h>
#include <nn/irsensor/irsensor_HandAnalysisTypesPrivate.h>

namespace nn { namespace irsensor {

/**
 * @name Hand Analysis
 * @{
 */

/**
 *  @brief                           Retrieve latest processed data by silhouette mode with full frame shape of Hand Analysis
 *
 *  @param[out] pOutValueArray       User allocated array that will receive the processed data
 *  @param[out] pReturnCount         Number of elements written to pOutValueArray (starting at index 0)
 *  @param[in]  maxCount             Number of elements of pOutValueArray
 *  @param[in]  infSamplingNumber    Lowest sampling number of elements you want to retrieve
 *  @param[in]  handle               Handle of IR sensor
 *
 * @details                          This function retrieves silhouette states send by the IR sensor. When user
 *                                   finger covers entire IR sensor, additional full shape information is appended to data.
 *                                   You can retrieve several states at the same time. The function will return all
 *                                   the states available where infSamplingNumber <= state.samplingNumber. If your
 *                                   array is not large enough it will return the more recent silhouette
 *                                   states, i.e the higher samplingNumber.
 *                                   Usually you want infSamplingNumber == 0 for the first call and having
 *                                   infSamplingNumber == lastSamplingNumber + 1 for the next one.
 *
 *                                   Compared with getting other irsensor state like in Moment or Clustering mode,
 *                                   getting Hand Analysis state will not only copying value coming from irsensor
 *                                   but also decoding it which is not negligible. When using infSamplingNumber
 *                                   the library knows which states you already have, and won't decode them neither
 *                                   provide it to you a second time.
 *
 *                                   When IR sensor is in trouble and unable to use, internal system displays the error.
 *                                   While displaying, this API blocks the thread which calls it.
 *                                   After user handles the error display, it returns nn::irsensor::ResultIrsensorDeviceError.
 *
 *                                   When the controller firmware is needed to update,
 *                                   internal system calls controller support applet and updates it.
 *                                   While updating, this API blocks the thread which calls it.
 *                                   If it succeeds it returns nn::ResultSuccess,
 *                                   and if it fails it returns nn::irsensor::ResultIrsensorDeviceError.
 *
 *                                   IR sensor is not available when application uses NFP functions on at least one controller.
 *                                   The second IR sensor is not avilable only when the total number of connected controller is larger than 4.
 *                                   If these constraints are not satisfied, this API returns nn::irsensor::ResultIrsensorNotReady.
 *
 *  @retresult
 *   @handleresult{nn::irsensor::ResultIrsensorNotReady}
 *   @handleresult{nn::irsensor::ResultIrsensorUnavailable}
 *   @handleresult{nn::irsensor::ResultIrsensorDeviceError}
 *   @handleresult{nn::irsensor::ResultHandAnalysisModeIncorrect}
 *  @endretresult
 *  @pre
 *                         - Handle is already initialized via Initialize API.
 *                         - For given handle, IR camera is running as silhouette mode of Hand Analysis
 *                         - pOutValue != nullptr
 *                         - pReturnCount != nullptr
 *  @post
 *                         - As for retrieved data pOutValue
 *                             - 0 <= pOutValue->shapeCount && pOutValue->shapeCount <= IrHandAnalysisShapeCountMax
 */
::nn::Result GetHandAnalysisSilhouetteStateWithFullFrameShape(HandAnalysisSilhouetteStateWithFullFrameShape* pOutValueArray, int *pReturnCount, int maxCount, int64_t infSamplingNumber, const IrCameraHandle& handle) NN_NOEXCEPT;

/**
 * @}
 */

} } // namespace nn::irsensor
