﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief API declaration for AdaptiveClusteringProcessor
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>
#include <nn/irsensor/irsensor_AdaptiveClusteringProcessorTypes.h>

namespace nn { namespace irsensor {

/**
 * @name AdaptiveClusteringProcessor
 * @{
 */

/**
 *  @brief              Start AdaptiveClustering mode of IR sensor
 *  @param[in] handle   Handle of IR camera
 *  @param[in] config   Setting for AdaptiveClustering processor
 *  @pre
 *                      - Handle is already initialized via Initialize API
 *                      - For given handle, IR camera is not running yet
 *  @details            User call Run**() API and IR sensor starts processing
 *                      User keeps getting data via Get**() API while AdaptiveClustering is running
 *                      When IR sensor is already running another mode, it stops running mode and starts AdaptiveClustering mode.
 */
void RunAdaptiveClusteringProcessor(const IrCameraHandle& handle, const AdaptiveClusteringProcessorConfig& config) NN_NOEXCEPT;

/**
 *  @brief                           Retrieve latest AdaptiveClustering mode data
 *
 *  @param[out] pOutStates           User allocated array that will receive the processed data
 *  @param[out] pOutCount            Number of elements written to pOutStates
 *  @param[in]  countMax             Maximum number of elements that can be written to pOutStates
 *  @param[in]  infSamplingNumber    Lowest sampling number of elements you want to retrieve
 *  @param[in]  handle               Handle of IR sensor
 *
 * @details                          This function retrieves AdaptiveClustering states sent by the IR sensor. You can
 *                                   retrieve several states at the same time. The function will return all
 *                                   the states available where infSamplingNumber <= state.samplingNumber. If your
 *                                   array is not large enough it will return the more recent silhouette
 *                                   states, i.e. the highest sampling numbers.
 *                                   Usually you want infSamplingNumber == 0 for the first call and having
 *                                   infSamplingNumber == lastSamplingNumber + 1 for the next one.
 *
 *                                   Unlike other irsensor modes (like clustering or moment modes), getting states
 *                                   in the AdaptiveClustering mode will copy and decode data, which is time consuming.
 *                                   The benefit of using infSamplingNumber is to avoid extra computation for states
 *                                   you already received.
 *
 *                                   When the IR sensor is not functioning properly, the system will display a dialog
 *                                   with an error message.
 *                                   This function blocks the calling thread while the message is displayed.
 *                                   After the user closes the error dialog, the function returns nn::irsensor::ResultIrsensorDeviceError.
 *
 *                                   When the controller firmware needs an update,
 *                                   the system calls the controller support applet to update it.
 *                                   While updating, this function blocks the calling thread.
 *                                   If the update succeeds this function returns nn::irsensor::ResultIrsensorNotReady,
 *                                   and otherwise returns nn::irsensor::ResultIrsensorDeviceError.
 *
 *                                   IR sensor is not available when the application uses NFC functions on at least one controller.
 *                                   When there are more than 4 connected controllers, only the first IR sensor is available.
 *                                   If these constraints are not satisfied, this function returns nn::irsensor::ResultIrsensorNotReady.
 *
 *  @retresult
 *   @handleresult{nn::irsensor::ResultIrsensorNotReady}
 *   @handleresult{nn::irsensor::ResultIrsensorUnavailable}
 *   @handleresult{nn::irsensor::ResultIrsensorDeviceError}
 *  @endretresult
 *  @pre
 *                         - Handle is already initialized via Initialize API.
 *                         - For a given handle, IR camera is running as AdaptiveClustering mode
 *                         - pOutStates != nullptr
 *                         - pOutCount != nullptr
 *  @post
 *                         - For all pOutValue in pOutStates .. pOutStates + pOutCount - 1:
 *                             - 0 <= pOutValue->objectCount && pOutValue->objectCount <= AdaptiveClusteringProcessorObjectCountMax
 */
::nn::Result GetAdaptiveClusteringProcessorStates(AdaptiveClusteringProcessorState* pOutStates, int* pOutCount, int countMax, int64_t infSamplingNumber, const IrCameraHandle& handle) NN_NOEXCEPT;

/**
 * @}
 */

}} // namespace nn::irsensor
