﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラーの拡張バスに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/hid/hid_NpadCommonTypes.h>

namespace nn { namespace hidbus {

/**
 * @brief 拡張バスのハンドルです。
 */
struct BusHandle
{
    uint64_t _storage;
};

/**
* @brief 拡張バスの種類です。
*/
enum BusType
{
    BusType_LeftJoyRail  = 0,  //!< 左 Joy-Con のレール
    BusType_RightJoyRail = 1,  //!< 右 Joy-Con のレール
};

/**
* @brief Joy-Con で定期的に通信を行うモードを有効化する際に渡すワークバッファのサイズです。
*/
const size_t JoyPollingModeWorkBufferSize = nn::os::MemoryPageSize;

/**
* @brief Joy-Con で定期的に通信を行うモードの受信データの最大値 (Byte) です。
*/
const size_t JoyPollingModeMaxReceiveDataSize = 38;

/**
* @brief Joy-Con で定期的に通信を行うモードの種類です。
*/
enum JoyPollingMode
{
    JoyPollingMode_SixAxisSensorDisable,   //!< Joy-Con の六軸センサを使用しないモードです。
    JoyPollingMode_SixAxisSensorEnable,    //!< Joy-Con の六軸センサと同時に使うモードです。
};

/**
* @brief Joy-Con で定期的に通信を行うモードの受信データです。
*/
struct JoyPollingReceivedData
{
    uint8_t data[JoyPollingModeMaxReceiveDataSize];         //!< 受信データを保存するバッファです。
    size_t  outSize;                                        //!< 受信データのサイズです。
    int64_t samplingNumber;                                 //!< 受信データのサンプリングナンバーです。
};

//! @name コントローラーの拡張バス関連 API
//! @{

/**
* @brief       Npad に搭載されている拡張バスを操作するためのハンドルを取得します。
*
* @details     Npad ID で指定された Npad に搭載されている拡張バスを操作するためのハンドルを取得します。
*              拡張バスに対するハンドルは BusType を指定して取得します。
*              取得された handle は物理的なコントローラの拡張バスと 1 対 1 対応します。
*              同じコントローラであっても切断後に再接続された場合は切断前の handle が無効化されます。
*              拡張バス機能が存在しないコントローラーが割り当てられた Npad を指定した場合、そのコントローラの handle は取得できません。
*
* @param[out]  pOutHandle      コントローラーの拡張バスのハンドルの格納先
* @param[in]   id              Npad ID
* @param[in]   busType         取得する Bus の Type
*
* @return      ハンドルが取得できたかどうか
*
*/
bool GetBusHandle( nn::hidbus::BusHandle* pOutHandle, const nn::hid::NpadIdType &id, BusType busType ) NN_NOEXCEPT;

/**
 * @brief       コントローラーの拡張バス機能を初期化します。
 *
 * @details     コントローラーの拡張バス機能を初期化します。
 *              本 API の呼び出しでは拡張バスに接続されたデバイスに対する操作は発生しません。
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                          処理に成功しました。}
 *      @handleresult{nn::hidbus::ResultInvalidHandle,            無効な handle です。}
 * @endretresult
 *
 * @param[in]   handle          拡張バスのハンドル
 */
nn::Result Initialize(const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief       コントローラーの拡張バス機能を終了します。
*
* @details     コントローラーの拡張バス機能を終了します。
*              本 API 呼び出し後、依存がある一部の機能を有効化できるようになります。
*              本 API の呼び出しでは拡張バスに接続されたデバイスに対する操作は発生しません。
*
* @param[in]   handle          拡張バスのハンドル
*/
void Finalize(const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
 * @brief       コントローラーの拡張バスに接続されたデバイスの有効化・無効化を切り替えます。
 *
 * @details     コントローラーの拡張バスに接続されたデバイスの有効化・無効化を切り替えます。
 *              コントローラーの拡張バスにデバイスが接続された状態で本関数で有効化すると、拡張バス機能が有効化されます。@n
 *              有効化されるとコントローラによっては本機能と依存した一部の機能が使用できなくなります。@n
 *              拡張バスからデバイスが取り外された場合や有効化したアプリケーションがインフォーカス状態でなくなると自動で無効化されます。
 *              インフォーカス状態以外で本 API を呼ぶと、ResultNotInForcus が返り有効化できません。
 *              インフォーカス状態への遷移時に再度有効化してください。@n
 *              拡張バス機能を使用するために対応コントローラーのファームウェアに更新が必要な場合に本関数を呼び出すと、
 *              ファームウェア更新を行う UI を表示し、ファームウェアを更新した上で有効化処理を行います。@n
 *              ファームウェア更新中にコントローラーが再起動した場合には更新完了後、 ResultInvalidHandle() が返ります。@n
 *              本機能はファームウェア更新中や、拡張バスへのデバイス接続時にデバイスとの通信準備を行う際にブロッキングしますのでご注意ください。
 *
 * @param[in]   isEnabled       有効化・無効化
 * @param[in]   deviceId        デバイス ID
 * @param[in]   handle          拡張バスのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *      @handleresult{nn::hidbus::ResultFWUpdateFailed,             拡張バス機能を使用するためのコントローラのアップデートに失敗しました。}
 *      @handleresult{nn::hidbus::ResultInvalidHandle,              無効な handle です。}
 *      @handleresult{nn::hidbus::ResultNotInForcus,                インフォーカス状態でないため有効化できません。インフォーカス状態になった後、再度本 API を呼び出してください。}
 *      @handleresult{nn::hidbus::ResultActivateFailureNpadBusy,    拡張バス機能と排他で制御されるべき他の機能が有効なため有効化できませんでした。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceNotAttached,  拡張バスにデバイスが接続されていません。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceReadyTimeout, 拡張バスに接続されたデバイスとの通信準備のタイムアウトです。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,      拡張バスに接続されたデバイスとの通信のタイムアウトです。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceInvalidId,    指定された デバイス ID と異なるデバイスが拡張バスに接続されています。}
 * @endretresult
 *
 * @pre
 *              Initialize() の呼び出しが完了している
 */
nn::Result EnableExternalDevice(bool isEnabled, uint32_t deviceId, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief       コントローラーの拡張バスへ接続されたデバイスへコマンドを送信し、返信をブロッキングして受け取ります。
*
* @details     コントローラーの拡張バスへ接続されたデバイスへコマンドを送信し、返信をブロッキングして受け取ります。
*              本関数は拡張デバイスからの返信が来るまでブロッキングします。
*
* @param[out]   pOutSize                デバイスから受信したデータをコピーしたサイズ
* @param[out]   pOutBuffer              デバイスから受信したデータをコピーするバッファへのポインタ
* @param[in]    outBufferSize            デバイスから受信したデータをコピーするバッファのサイズ
* @param[in]    handle                   コマンドを送信する拡張バスへのハンドル
* @param[in]    pInBuffer                デバイスへ送信するデータが入ったバッファへのポインタ
* @param[in]    inBufferSize             デバイスへ送信するデータが入ったバッファのサイズ
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している
*  - handle 取得時に BusType に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result SendAndReceive(size_t* pOutSize,
                          void* pOutBuffer, size_t outBufferSize,
                          const nn::hidbus::BusHandle& handle,
                          const void* pInBuffer, size_t inBufferSize
                          ) NN_NOEXCEPT;

/**
* @brief       Joy-Con レールに接続された拡張デバイスと定期的にデータ通信を行うモードを有効化します。
*
* @details     Joy-Con レールに接続された拡張デバイスと定期的にデータ通信を行うモードを有効化します。
*              本モードを有効化すると、定期的にデバイスとの送受信を行います。
*              hidbus 機能自体が無効化されると、本モード設定も無効化されます。
*              Joy-Con には六軸センサを同時に使うモードと同時に使わないモードがあります。
*              六軸センサと同時に使うモードの場合、定期受信できるサイズの最大値は 8 byte、読み出し可能な入力状態の最大数は 10 です。
*              六軸センサを使用しないモードの場合、定期受信できるサイズの最大値は 38 byte、読み出し可能な入力状態の最大数は 10 です。
*              各ハンドルごとにワークバッファを渡してください。
*              ワークバッファには JoyPollingModeWorkBufferSize 以上のサイズが必要です。
*              本関数に渡したワークバッファは DisableJoyPollingReceiveMode() の呼び出し、 hidbus 機能の無効化、 handle の無効化時に再利用可能となります。
*              再利用可能になるまでは、ワークバッファへアクセスしないでください。
*
* @param[in]   handle                    コマンドを送信する拡張バスへのハンドル
* @param[in]   pInCommand                デバイスへ定期送信するデータが入ったバッファへのポインタ
* @param[in]   inCommandSize             デバイスへ定期送信するデータが入ったバッファのサイズ
* @param[in]   workBuffer                ワークバッファ
* @param[in]   workBufferSize            ワークバッファのサイズ
* @param[in]   mode                      ポーリングのモード
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している
*  - EnableExternalDevice() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*  - workBuffer != nullptr
*  - workBuffer が nn::os::MemoryPageSize でアラインされている
*/
nn::Result EnableJoyPollingReceiveMode(const nn::hidbus::BusHandle& handle,
                                       const void* pInCommand, size_t inCommandSize,
                                       void* workBuffer, size_t workBufferSize,
                                       JoyPollingMode mode) NN_NOEXCEPT;

/**
* @brief       Joy-Con レールに接続された拡張デバイスと定期的にデータ通信を行うモードを無効化します。
*
* @details     Joy-Con レールに接続された拡張デバイスと定期的にデータ通信を行うモードを無効化します。
*
* @param[in]   handle                    コマンドを送信する拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している
*  - EnableExternalDevice() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result DisableJoyPollingReceiveMode(const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief       Joy-Con レールに接続された拡張デバイスと定期的に通信を行うモードの最新の受信データを取得します。
*
* @details     指定の handle と対応する拡張バスに接続されたデバイスから GetJoyPollingReceivedData() で 1 つの受信データを読み出した時と同じ値が返ります。
*
* @param[in]   pOutData                　最新の受信データ
* @param[in]   handle                    受信データをコピーする拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している。
*  - EnableExternalDevice() の呼び出しが完了している
*  - EnablePollingReceiveMode() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetJoyPollingReceivedData(JoyPollingReceivedData* pOutData, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief        Joy-Con レールに接続された拡張デバイスと定期的に通信を行うモードの受信データを過去に遡って読み出します。
*
* @details     最新のものから過去に遡って利用可能な数だけ順に、指定の handle と対応する拡張バスに接続されたデバイスから受信データを読み出します。
*              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
*              Joy-Con で六軸センサと同時に使うモードの場合、定期受信できるサイズの最大値は 8 byte、読み出し可能な入力状態の最大数は 10 です。
*              Joy-Con で六軸センサを使用しないモードの場合、定期受信できるサイズの最大値は 38 byte、読み出し可能な入力状態の最大数は 10 です。
*              利用可能な入力状態には読み出し済みのものも含まれます。
*              差分だけを利用したい場合は JoyPollingReceivedData::samplingNumber を参照してください。
*              内部で接続されたデバイスとの通信のタイムアウトが発生した場合、データ更新されません。
*
* @param[out]  pOutData                　受信データを受け取るバッファへのポインタ
* @param[in]   count                　   受信データを受け取る数
* @param[in]   handle                    受信データをコピーする拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している。
*  - EnableExternalDevice() の呼び出しが完了している
*  - EnablePollingReceiveMode() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetJoyPollingReceivedData(JoyPollingReceivedData* pOutData, int count, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

//! @}

}} // namespace nn::hidbus
