﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Npad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/hid/system/hid_CommonTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_Color.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       プレイレポート用のデバイスタイプの定義です
 */
typedef uint8_t PlayReportDeviceType;

const PlayReportDeviceType PlayReportDeviceType_Unknown              = 0;       //!< 未知のデバイス
const PlayReportDeviceType PlayReportDeviceType_JoyConLeft           = 1;       //!< 左コン
const PlayReportDeviceType PlayReportDeviceType_JoyConRight          = 2;       //!< 右コン
const PlayReportDeviceType PlayReportDeviceType_SwitchProController  = 3;       //!< Switch Pro Controller
const PlayReportDeviceType PlayReportDeviceType_Palma                = 4;       //!< Palma
const PlayReportDeviceType PlayReportDeviceType_GcController         = 5;       //!< Gc Controller
const PlayReportDeviceType PlayReportDeviceType_UsbController        = 11;      //!< USB ゲームパッド

/**
 * @brief       プレイレポート用の操作形態の定義です
 */
typedef uint8_t PlayReportPlayStyle;

const PlayReportPlayStyle PlayReportPlayStyle_Unknown               = 0;       //!< 未知のデバイス
const PlayReportPlayStyle PlayReportPlayStyle_SwitchProController   = 1;       //!< Switch Pro Controller 相当の操作
const PlayReportPlayStyle PlayReportPlayStyle_Handheld              = 2;       //!< 携帯持ち相当の操作
const PlayReportPlayStyle PlayReportPlayStyle_JoyConDual            = 3;       //!< Joy-Con 2本持ち
const PlayReportPlayStyle PlayReportPlayStyle_JoyConLeftHorizontal  = 4;       //!< 左コン 横持ち
const PlayReportPlayStyle PlayReportPlayStyle_JoyConLeftVertical    = 5;       //!< 左コン 片手持ち
const PlayReportPlayStyle PlayReportPlayStyle_JoyConRightHorizontal = 6;       //!< 右コン 横持ち
const PlayReportPlayStyle PlayReportPlayStyle_JoyConRightVertical   = 7;       //!< 右コン 片手持ち
const PlayReportPlayStyle PlayReportPlayStyle_Palmal                = 8;       //!< Palma

static const size_t PlayReportControllerUsageCountMax = 16;    //!< コントローラーの利用状態の最大数です

/**
 * @brief      PlayReport で送信するコントローラーの利用状態です
 */
struct PlayReportControllerUsage
{
    PlayReportDeviceType deviceType;      //!< デバイスの種類。 (1バイトに変換)
    PlayReportPlayStyle style;            //!< ゲームの操作方法。フルコン相当やジョイコン横持ち相当など 1バイト
    uint8_t controllerNumber;             //!< コントローラー番号。コントローラー番号に相当するものがない場合は 0。 1バイト
    NN_PADDING5;
    union
    {
        uint8_t raw[8];                   //!< デバイス情報を補足する情報として領域を確保。 8バイト
        struct
        {
            uint8_t subType;
            InterfaceType interfaceType;
            uint8_t version[4];           //!< Major-Minor-Micor-Revision 4 バイト
            ::nn::Bit8 reserved[2];
        } nxControllerInfo;
        struct
        {
            uint8_t vidHigh;
            uint8_t vidLow;
            uint8_t pidHigh;
            uint8_t pidLow;
            ::nn::Bit8 reserved[4];
        } usbDevice;
    };
    NN_PADDING4;
};

//! @name コントローラーのプレイレポート用の関数
//! @{

/**
 * @brief       プレイレポート用のコントローラーの利用状態の情報が更新通知をバインドします。
 *
 * @details     コントローラーの利用状態が更新を示す通知を指定のイベントオブジェクトにバインドします。
 *              コントローラーの接続や操作形態の変更などが生じた際に登録されたイベントオブジェクトに対して通知されます。
 *              通知は最低1分以上の間隔をおいて行われます。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindPlayReportControllerUsageUpdateEvent(::nn::os::SystemEventType* pEvent,
                                              ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       ユーザーが使用中のコントローラーの情報を取得します
 *
 * @details     利用状態にあるコントローラーの情報を取得します。
 *              コントローラーの情報はデバイス単位で、 PlayReportControllerUsageCountMax 個まで取得できます。
 *              ひとつもコントローラーが操作状態にない場合は、 0 が返ります。
 *
 * @param[out]  pOutValues                   コントローラーの利用状態を読み出すバッファ（配列）
 * @param[in]   count                       pOutValues の数
 *
 * @return      利用状態にあるコントローラーの数を返します。
 *
 * @pre
 *              - pOutValues != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 */
int GetPlayReportControllerUsages(PlayReportControllerUsage* pOutValues, int count) NN_NOEXCEPT;

/**
 * @brief       プレイレポート用のコントローラーの登録情報が更新通知をバインドします。
 *
 * @details     コントローラーの登録情報が更新を示す通知を指定のイベントオブジェクトにバインドします。
 *              本通知は新たなコントローラーが登録されるたびに通知が行われます。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindPlayReportRegisteredDeviceUpdateEvent(::nn::os::SystemEventType* pEvent,
                                               ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;


/**
 * @brief       DeviceType 型を プレイレポート用の PlayReportDeviceType 型に変換します
 *
 * @param[in]   deviceType                      変換元の DeviceTypeSet 型の値
 *
 * @pre
 *              - deviceType.CountPopulation() <= 1
 */
PlayReportDeviceType GetPlayReportDeviceType(DeviceTypeSet& deviceType) NN_NOEXCEPT;


//! @}

}}} // namespace nn::hid::system
