﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Npad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid { namespace system {

typedef ::nn::util::BitFlagSet<32, NpadStyleTag>::Flag<29> NpadStyleSystemExt;       //!< 本体機能の拡張操作を表す Npad の種類です。

/**
 * @brief       Npad の操作の入力状態を表す構造体です。
 *
 * @details     本体機能向けにボタン、スティックの状態を提供します。
 *              アナログスティックの入力状態などがユーザーの持ち方に合わせて適切に補正されます。
 *              取得できる入力は、十字, A, B, X, Y, L, R, ZL, ZR, +, - ボタンと左右2つのアナログスティックです。
 */
struct NpadSystemExtState
{
    int64_t samplingNumber;             //!< Npad の入力状態が更新される度に増加する値です。
    NpadButtonSet buttons;              //!< Npad のデジタルボタンの状態です。
    AnalogStickState analogStickL;      //!< Npad の左アナログスティックの状態です。
    AnalogStickState analogStickR;      //!< Npad の右アナログスティックの状態です。
    NpadAttributesSet attributes;       //!< Npad の状態を表す属性値です。
};

/**
 * @brief       Npad の本体機能操作の操作形態の主となる種類を表す型です
 */
enum NpadSystemExtMainStyle
{
    NpadSystemExtMainStyle_None = 0,                       //!< 操作形態なし
    NpadSystemExtMainStyle_HandheldNoJoyCon,               //!< 携帯機 Joy-Con 無し
    NpadSystemExtMainStyle_HandheldLeftJoyOnly,            //!< 携帯機 Joy-Con (L) のみ
    NpadSystemExtMainStyle_HandheldRightJoyOnly,           //!< 携帯機 Joy-Con (R) のみ
    NpadSystemExtMainStyle_HandheldJoyCon,                 //!< 携帯機両Joy-Con 有り
    NpadSystemExtMainStyle_JoyDual,                        //!< Joy-Con 2本持ち
    NpadSystemExtMainStyle_JoyDualLeftOnly,                //!< Joy-Con 2本持ち 左のみ
    NpadSystemExtMainStyle_JoyDualRightOnly,               //!< Joy-Con 2本持ち 右のみ
    NpadSystemExtMainStyle_JoyLeftHorizontal,              //!< Joy-Con (L) 横持ち
    NpadSystemExtMainStyle_JoyLeftVertical,                //!< Joy-Con (L) 縦持ち
    NpadSystemExtMainStyle_JoyRightHorizontal,             //!< Joy-Con (R) 横持ち
    NpadSystemExtMainStyle_JoyRightVertical,               //!< Joy-Con (R) 縦持ち
    NpadSystemExtMainStyle_SwitchProController,            //!< Switch Pro Ctonroller 縦持ち
    NpadSystemExtMainStyle_CompatibleProController,        //!< Pro Controller 互換の未知デバイス
    NpadSystemExtMainStyle_CompatibleJoyCon,               //!< Joy-Con 互換の未知デバイス
    NpadSystemExtMainStyle_Unknown = 0xff,                 //!< 未知のコントローラーによる操作
};

/**
 * @brief       Npad の本体機能操作の操作形態の補足する種類を表す型です
 */
enum NpadSystemExtSubStyle
{
    NpadSystemExtSubStyle_Default = 0,                     //!< デフォルトのサブ操作形態
};

//! @name Npad 関連 API
//! @{

/**
 * @brief       Npad の本体機能拡張操作の最新の入力状態を取得します。
 *
 * @details     指定の Npad ID と対応する Npad から GetNpadStates() で 1 つの入力状態を読み出した時と同じ値が返ります。
 *              最低 1 回の入力状態の更新が保証されるため、 SetSupportedNpadIdType() 呼び出し完了直後から利用可能です。
 *
 * @param[out]  pOutValue               入力状態を読み出すバッファ
 * @param[in]   id                      Npad ID
 *
 * @pre
 *              - 指定の Npad ID が SetSupportedNpadIdType() で有効化されている
 *              - pOutValue != nullptr
 * @post
 *              - !(pOutValue->buttons.Get<NpadButton::Left>() && pOutValue->buttons.Get<NpadButton::Right>())
 *              - !(pOutValue->buttons.Get<NpadButton::Up>() && pOutValue->buttons.Get<NpadButton::Down>())
 *              - アナログスティックの入力状態 outValue->analogStickL と outValue->analogStickR は下記の条件を満たす円の内側（境界含む）に位置
 *                  - 原点は (0, 0)
 *                  - 半径は AnalogStickMax
 */
void GetNpadState(NpadSystemExtState* pOutValue, const NpadIdType& id
                  ) NN_NOEXCEPT;

/**
 * @brief       Npad の本体機能拡張操作の入力状態を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に、指定の Npad ID と対応する Npad から入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *
 *              読み出し可能な入力状態の最大数は NpadStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は NpadSystemExtState::samplingNumber を参照してください。
 *
 *              指定の Npad ID が本体機能拡張操作が有効になっていない場合、入力状態は無入力（デジタルボタンの押下は無く、アナログスティックはニュートラル位置）となります。
 *
 * @param[out]  pOutValues              入力状態を読み出すバッファ（配列）
 * @param[in]   count                   読み出す入力状態の数
 * @param[in]   id                      Npad ID
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - 指定の Npad ID が SetSupportedNpadIdType() で有効化されている
 *              - pOutValues != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 *              - 読み出した入力状態 pOutValues[i] について
 *                  - !(pOutValues[i].buttons.Get<NpadButton::Left>() && pOutValues[i].buttons.Get<NpadButton::Right>())
 *                  - !(pOutValues[i].buttons.Get<NpadButton::Up>() && pOutValues[i].buttons.Get<NpadButton::Down>())
 *                  - アナログスティックの入力状態 pOutValues[i].analogStickL と pOutValues[i].analogStickR は下記の条件を満たす円の内側（境界含む）に位置
 *                      - 原点は (0, 0)
 *                      - 半径は AnalogStickMax
 */
int GetNpadStates(NpadSystemExtState* pOutValues,
                  int count,
                  const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       Npad の本体機能拡張操作の操作形態を取得します
 *
 * @details     Npad の本体機能操作の操作形態を取得します。
 *              操作形態はデバイスの種類や状態によって定義されます。
 *              操作形態はメインとサブの 2種類の値で構成されます
 *
 * @param[out]  pOutMainStyle           主となる操作形態
 * @param[out]  pOutSubStyle            主となる操作形態を分類するためのサブタイプ
 * @param[in]   id                      Npad ID
 *
 * @pre
 *              - pOutMainStyle != nullptr
 *              - pOutSubStyle != nullptr
 */
void GetNpadSystemExtStyle(NpadSystemExtMainStyle* pOutMainStyle,
                           NpadSystemExtSubStyle* pOutSubStyle,
                           const NpadIdType& id) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid
