﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Npad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_Color.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       Hid で共通で仕様するデバイスの種類です
 */
struct DeviceType
{
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<0>  SwitchProController;      //!< Switch Proコントローラー
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<0>  FullKeyController;        //!< 互換性維持
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<1>  Handheld;                 //!< 携帯機コントローラー
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<2>  HandheldJoyLeft;          //!< Joy-Con (L) 本体装着
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<3>  HandheldJoyRight;         //!< Joy-Con (R) 本体装着
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<4>  JoyConLeft;               //!< Joy-Con (L)
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<5>  JoyConRight;              //!< Joy-Con (R)
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<6>  Palma;                    //!< Palma
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<31> Unknown;                 //!< 未知のデバイスまたは種別が判定できないデバイス
};

/**
 * @brief       デバイスの種類の集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, DeviceType> DeviceTypeSet;


/**
 * @brief       接続インタフェースの定義です
 */
typedef uint8_t InterfaceType;

const InterfaceType InterfaceType_Unknown              = 0;       //!< 未知のデバイス
const InterfaceType InterfaceType_Bluetooth            = 1;       //!< Bluetooth
const InterfaceType InterfaceType_Rail                 = 2;       //!< Rail
const InterfaceType InterfaceType_Usb                  = 3;       //!< Usb

/**
 * @brief       電池残量を表す型です。
 */
typedef int32_t BatteryLevel;

const BatteryLevel BatteryLevel_None        = 0;       //!< 電池がない
const BatteryLevel BatteryLevel_CriticalLow = 1;       //!< 電池残量 Critical Low
const BatteryLevel BatteryLevel_Low         = 2;       //!< 電池残量 Low
const BatteryLevel BatteryLevel_Medium      = 3;       //!< 電池残量 Midium
const BatteryLevel BatteryLevel_High        = 4;       //!< 電池残量 High

/**
 * @brief      電源状態を表す型です。
 */
struct PowerInfo
{
    bool isPowered;                             //!< コントローラーは給電中かどうかを表します。
    bool isCharging;                            //!< コントローラーは充電中かどうかを表します。
    NN_PADDING6;
    BatteryLevel batteryLevel;                  //!< コントローラーの電池残量を表します。
};

/**
 * @brief      コントローラーの色情報を表す構造体です。
 */
struct ControllerColor
{
    nn::util::Color4u8Type main;                               //!< コントローラーの主となる色 (本体部分)
    nn::util::Color4u8Type sub;                                //!< コントローラーのサブ部分の色 (ボタン部)
};


}}} // namespace nn::hid::system
