﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       アナログスティックのキャリブレーション処理に関する API の宣言
 */

#pragma once

#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/system/hid_UniquePad.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       アナログスティックを識別するための定義です。
 */
enum AnalogStickPosition
{
    AnalogStickPosition_Right,    //!< 右アナログスティック
    AnalogStickPosition_Left,     //!< 左アナログスティック
};

/**
 * @brief       アナログスティックのキャリブレーションシーケンスの経過を示す定義です。
 */
enum AnalogStickManualCalibrationStage
{
    AnalogStickManualCalibrationStage_ReleaseFromRight,      //!< 右方向からの原点補正シーケンス
    AnalogStickManualCalibrationStage_ReleaseFromBottom,     //!< 下方向からの原点補正シーケンス
    AnalogStickManualCalibrationStage_ReleaseFromLeft,       //!< 左方向からの原点補正シーケンス
    AnalogStickManualCalibrationStage_ReleaseFromTop,        //!< 上方向からの原点補正シーケンス
    AnalogStickManualCalibrationStage_Rotate,                //!< 外周補正シーケンス
    AnalogStickManualCalibrationStage_Update,                //!< キャリブレーション値書き込みシーケンス
    AnalogStickManualCalibrationStage_Completed,             //!< キャリブレーションシーケンス完了
    AnalogStickManualCalibrationStage_Clear,                 //!< キャリブレーション値の初期化中
    AnalogStickManualCalibrationStage_ClearCompleted,        //!< キャリブレーション値の初期化完了
};

//! @name アナログスティックのキャリブレーション関連 API
//! @{

/**
 * @brief       アナログスティックの値を取得します。
 *
 * @details     任意の UniquePad のアナログスティックの値を取得します。
 *              キャリブレーション処理中のアナログスティックの値は UI で表示するのに適した値の形に変換されて出力されます。
 *
 *              キャリブレーションを行う対象のアナログスティックを UniquePadId と AnalogStickPositionで指定します。
 *              AnalogStick が 1つしかないデバイスの場合は AnalogStickPosition_Right を指定してください。
 *
 *              指定のコントローラーが存在しないときは原点無入力(0,0) 時の値が返されます。
 *
 * @param[out]  pOutValue               アナログスティックの状態を読み出すバッファ
 * @param[in]   id                      キャリブレーションを行うコントローラーの UniquePadId
 * @param[in]   position                キャリブレーションを行うアナログスティックの位置
 *
 */
void GetAnalogStickState(AnalogStickState* pOutValue, UniquePadId id, AnalogStickPosition position) NN_NOEXCEPT;

/**
 * @brief       アナログスティックの押し込みボタンの押下状態を取得します。
 *
 * @details     任意の UniquePad のアナログスティックの値を取得します。
 *
 *              キャリブレーションを行う対象のアナログスティックを UniquePadId と AnalogStickPositionで指定します。
 *              AnalogStick が 1つしかないデバイスの場合は AnalogStickPosition_Right を指定してください。
 *
 *              指定のコントローラーが存在しないときは false が返されます。
 *
 * @param[in]   id                      キャリブレーションを行うコントローラーの UniquePadId
 * @param[in]   position                キャリブレーションを行うアナログスティックの位置
 *
 * @return      アナログスティックの押下状態
 *
 */
bool IsAnalogStickButtonPressed(UniquePadId id, AnalogStickPosition position) NN_NOEXCEPT;

/**
 * @brief       アナログスティックの手動キャリブレーション処理を開始します。
 *
 * @details     BasicXpad のアナログスティックを指定してキャリブレーション処理を開始します。
 *              アナログスティックのキャリブレーション処理は複数同時に行うことはできません。
 *              進行中のキャリブレーション処理がある場合は、CancelAnalogStickManualCalibration() で終了する必要があります。
 *
 *              キャリブレーションを行う対象のアナログスティックを UniquePadId と AnalogStickPositionで指定します。
 *              AnalogStick が 1つしかないデバイスの場合は AnalogStickPosition_Right を指定してください。
 *
 *              キャリブレーション処理はいくつかのステージに分かれて進行します
 *              現在のステージは GetAnalogStickManualCalibrationValue() で取得できます。
 *              各ステージはUIに連動したものとなっています。
 *
 *              キャリブレーション処理中にUIに表示するアナログスティックの値は GetAnalogStickStateOnManualCalibration() で取得できます。
 *
 * @param[in]   id                      キャリブレーションを行うコントローラーの UniquePadId
 * @param[in]   position                キャリブレーションを行うアナログスティックの位置
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultAnalogStickDeviceNotConnected,  コントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultAnalogStickManualCalibrationNotSupported,  指定のコントローラーでは手動補正は対応していません。}
 * @endretresult
 *
 * @pre
 *              - 進行中のキャリブレーション処理が存在しない
 *
 */
Result StartAnalogStickManualCalibration(UniquePadId id, AnalogStickPosition position) NN_NOEXCEPT;

/**
 * @brief       アナログスティックのキャリブレーション処理の経過状況を取得します。
 *
 * @details     アナログスティックのキャリブレーション処理はいくつかのステージに分かれて行われます。
 *
 *              キャリブレーションを行う対象のアナログスティックを UniquePadId と AnalogStickPositionで指定します。
 *              AnalogStick が 1つしかないデバイスの場合は AnalogStickPosition_Right を指定してください。

 *
 *              アナログスティックのキャリブレーション処理は以下のステージで順に進行します。
 *              ユーザーが適切な操作を行うことで自動的に次のステージに進みます。
 *              1. AnalogStickManualCalibrationStage_ReleaseFromRight
 *              2. AnalogStickManualCalibrationStage_ReleaseFromBottom
 *              3. AnalogStickManualCalibrationStage_ReleaseFromLeft
 *              4. AnalogStickManualCalibrationStage_ReleaseFromTop
 *              5. AnalogStickManualCalibrationStage_Rotate
 *              6. AnalogStickManualCalibrationStage_Update
 *              7. AnalogStickManualCalibrationStage_Completed
 *
 *              アナログスティックのキャリブレーション処理が問題なく進行している間は、nn::ResultSuccess が返されます。
 *              キャリブレーション中に何か問題があった場合は、失敗の理由が返されます。
 *              処理に失敗した場合は、RetryCurrentAnalogStickManualCalibrarionStage() で直前の手順から処理をやり直すか
 *              CancelAnalogStickManualCalibration() でキャリブレーション処理を終了してください。
 *
 *              キャリブレーション処理が完了すると AnalogStickManualCalibrationStage_Completedが返ります。
 *
 * @param[out]  pOutValue               アナログスティックのキャリブレーションステージ
 * @param[in]   id                      キャリブレーションを行うコントローラーの UniquePadId
 * @param[in]   position                キャリブレーションを行うアナログスティックの位置
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                    処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultAnalogStickCalibrationInvalidRelease,  ゼロ点キャリブレーション時にアナログスティックが正しく原点に戻りませんでした。}
 *   @handleresult{nn::hid::system::ResultAnalogStickDeviceNotConnected,  コントローラーが接続されていません。}
 * @endretresult
 *
 *
 */
nn::Result GetAnalogStickManualCalibrationStage(AnalogStickManualCalibrationStage* pOutValue, UniquePadId id, AnalogStickPosition position) NN_NOEXCEPT;

/**
 * @brief       現在のアナログスティックのキャリブレーション処理をやり直します。
 *
 * @details     キャリブレーション処理中に失敗した場合などに、現在のキャリブレーション処理をやり直します。
 *              本関数では現在処理中のステージからやり直しを行います。
 *              ユーザー操作などによって正しくキャリブレーションができなかった場合に、ユーザーに注意を表示してやり直す際に使用します。
 *
 *              キャリブレーションを行う対象のアナログスティックを UniquePadId と AnalogStickPositionで指定します。
 *              AnalogStick が 1つしかないデバイスの場合は AnalogStickPosition_Right を指定してください。
 *
 *              キャリブレーション処理を一番最初からやり直す場合は、CancelAnalogStickManualCalibration() で処理を終了した後に、
 *              StartAnalogStickManualCalibration() で再度処理を開始してください。
 *
 *              StartAnalogStickManualCalibration() が呼ばれておらず進行中のキャリブレーション処理が存在しない場合は本関数は何も作用しません。
 *
 * @param[in]   id                      キャリブレーションを行うコントローラーの UniquePadId
 * @param[in]   position                キャリブレーションを行うアナログスティックの位置
 *
 */
void RetryCurrentAnalogStickManualCalibrationStage(UniquePadId id, AnalogStickPosition position) NN_NOEXCEPT;

/**
 * @brief       進行中アナログスティックのキャリブレーション処理をキャンセルします。
 *
 * @details     進行中のアナログスティックのキャリブレーション処理をキャンセルします。
 *
 *              キャリブレーションを行う対象のアナログスティックを UniquePadId と AnalogStickPositionで指定します。
 *              AnalogStick が 1つしかないデバイスの場合は AnalogStickPosition_Right を指定してください。
 *
 *              キャリブレーション処理が以下のいずれかの処理中はコントローラーに保存されているキャリブレーション値を更新中のためキャンセルができません。
 *              書き込みが完了したあとに、キャリブレーション処理は正常終了します。
 *              　- AnalogStickManualCalibrationStage_Update
 *                - AnalogStickManualCalibrationStage_Clear
 *
 *              StartAnalogStickManualCalibration() が呼ばれておらず進行中のキャリブレーション処理が存在しない場合は本関数は何も作用しません。
 *
 * @param[in]   id                      キャリブレーションを行うコントローラーの UniquePadId
 * @param[in]   position                キャリブレーションを行うアナログスティックの位置
 *
 */
void CancelAnalogStickManualCalibration(UniquePadId id, AnalogStickPosition position) NN_NOEXCEPT;

/**
 * @brief       アナログスティックのキャリブレーション値を初期値に戻します。
 *
 * @details     アナログスティックのキャリブレーション値を工程で検査時に書き込まれた初期値に戻します。
 *              コントローラーにはアナログスティックごとに工程での検査時に計測されたキャリブレーション値が書き込まれています。
 *              本関数では、ユーザーの手元で書き込まれたキャリブレーション値をクリアし初期値を使用するように戻します。
 *
 *              キャリブレーションを行う対象のアナログスティックを UniquePadId と AnalogStickPositionで指定します。
 *              AnalogStick が 1つしかないデバイスの場合は AnalogStickPosition_Right を指定してください。
 *
 *              変更処理には時間がかかるため、非同期で処理が行われます。
 *              初期化中は GetAnalogStickManualCalibrationStage() で取得される状態が AnalogStickManualCalibrationStage_Clear が返ります。
 *              初期化が完了すると、AnalogStickManualCalibrationStage_ClearCompleted が返ります。
 *
 *              初期値への書き戻しは StartAnalogStickManualCalibration() が呼ばれてキャリブレーション処理が終了するまでの間であればいつでも実施できます。
 *
 *              StartAnalogStickManualCalibration() が呼ばれておらず進行中のキャリブレーション処理が存在しない場合は本関数は何も作用しません。
 *
 * @param[in]   id                      キャリブレーションを行うコントローラーの UniquePadId
 * @param[in]   position                キャリブレーションを行うアナログスティックの位置
 *
 */
void ResetAnalogStickManualCalibration(UniquePadId id, AnalogStickPosition position) NN_NOEXCEPT;

/**
 * @brief       原点補正処理中に、アナログスティックがリリースポジション(スティックを各方向に倒しきった位置)にあるかどうかを取得します。
 *
 * @details     原点補正処理では、アナログスティックを各方向に倒してから指を離します。
 *              指を離す前にスティックが一定領域に入っていることをユーザーに UI で示します。
 *              本関数では該当の領域にスティックが入っているかどうかをチェックすることができます。
 *
 *              キャリブレーションを行う対象のアナログスティックを UniquePadId と AnalogStickPositionで指定します。
 *              AnalogStick が 1つしかないデバイスの場合は AnalogStickPosition_Right を指定してください。
 *
 *              StartAnalogStickManualCalibration() が呼ばれておらず進行中のキャリブレーション処理が存在しない場合は常に false を返します。
 *
 * @param[in]   id                      キャリブレーションを行うコントローラーの UniquePadId
 * @param[in]   position                キャリブレーションを行うアナログスティックの位置
 *
 * @return      アナログスティックがリリースポジション内にある場合は true, そうでない場合は false
 *
 */
bool IsAnalogStickInReleasePosition(UniquePadId id, AnalogStickPosition position) NN_NOEXCEPT;

/**
 * @brief       外周補正処理中に、アナログスティックが外周範囲内にあるかどうかを取得します。
 *
 * @details     外周補正処理中は、アナログスティックを外周に沿って数回転します。
 *              内部ではアナログスティックの値が一定閾値に入っていることを確認し、正しくスティックが外周に沿って回転していることをチェックします。
 *              本関数では、アナログスティックが外周に沿って倒されているかどうかを判定する機能を提供します。
 *
 *              キャリブレーションを行う対象のアナログスティックを UniquePadId と AnalogStickPositionで指定します。
 *              AnalogStick が 1つしかないデバイスの場合は AnalogStickPosition_Right を指定してください。
 *
 *              StartAnalogStickManualCalibration() が呼ばれておらず進行中のキャリブレーション処理が存在しない場合は常に false を返します。
 *
 * @param[in]   id                      キャリブレーションを行うコントローラーの UniquePadId
 * @param[in]   position                キャリブレーションを行うアナログスティックの位置
 *
 * @return      アナログスティックが外周範囲内にある場合は true, そうでない場合は false
 *
 */
bool IsAnalogStickInCircumference(UniquePadId id, AnalogStickPosition position) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
