﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Seven 向け SixAxisSensor に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_MathTypes.h>

namespace nn { namespace hid {

const int SevenSixAxisSensorStateCountMax = 32;      //!< 内部的に保持される SevenSixAxisSensorState の最大数

const int SevenSixAxisSensorWorkBufferSize = 512 * 1024; //!< SevenSixAxisSensor のワークバッファサイズです。

/**
 * @brief       Seven 向け SixAxisSensor から取得される状態を表す構造体です。
 */
struct SevenSixAxisSensorState
{
    ::nn::TimeSpanType               timeStamp;        //!< サンプリング処理開始から処理結果を取得した時点までの時間情報です。
    int64_t                          samplingNumber;   //!< Seven 向け SixAxisSensor の入力状態がサンプリングされる度に増加する値です。
    int64_t                          coordinateNumber; //!< 基準姿勢が変わる度に増加する値です。
    ::nn::util::Float3               acceleration;     //!< 加速度センサーの各方向ごとの加速度の値です。単位は G です。
    ::nn::util::Float3               angularVelocity;  //!< ジャイロセンサーの各方向ごとの角速度の値です。360dps を 1.0 とする値です。
    nn::util::Vector4fType           rotation;         //!< 基準姿勢からの回転を表す単位クォータニオンです。 nn::util::MatrixFromQuaternion() で回転行列に変換されます。
};

//! @name Seven 向け SixAxisSensor関連 API
//! @{

/**
 * @brief       Seven 向け SixAxisSensor を初期化します。
 *
 * @details     Seven 向け SixAxisSensor を初期化します。
 *              本関数が呼び出された際にワークバッファが設定されます。
 *              ワークバッファは、Seven 向け SixAxisSensor の計算用途に利用されます。
 *              再利用可能になるまではアクセスしないでください。
 *              引数指定されるバッファは、FinalizeSevenSixAxisSensor() の呼び出し時に再利用可能となります。
 *
 * @param[in]   workBuffer           Seven 向け SixAxisSensor が利用するワークバッファ
 * @param[in]   workBufferSize       Seven 向け SixAxisSensor が利用するワークバッファのサイズ
 *
 * @pre
 *              - Seven 向け SixAxisSensor が未初期化である
 *
 *              - workBuffer != nullptr
 *              - workBuffer が nn::os::MemoryPageSize でアラインされている
 *              - workBufferSize == SevenSixAxisSensorWorkBufferSize
 */
void InitializeSevenSixAxisSensor(void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

/**
 * @brief       Seven 向け SixAxisSensor を終了します。
 *
 * @details     Seven 向け SixAxisSensor を終了します。
 *              本 API 呼び出し後、サンプリング処理は停止します。
 *              InitializeSevenSixAxisSensor() で割り当てられたバッファは再利用可能になります。
 *              Seven 向け SixAxisSensor が終了済の場合は何もしません。
 *
 * @post
 *              Seven 向け SixAxisSensor が未初期化状態
 */
void FinalizeSevenSixAxisSensor() NN_NOEXCEPT;

/**
 * @brief       Seven 向け SixAxisSensor のサンプリングを開始します。
 *
 * @details     Seven 向け SixAxisSensor のサンプリングを開始します。
 *              本関数が呼び出された際に基準姿勢が設定されます。
 *              既にサンプリングが開始されている場合は何もしません。
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                処理に成功しました。}
 * @endretresult
 *
 * @pre
 *              InitializeSevenSixAxisSensor() の呼び出しが完了している
 */
Result StartSevenSixAxisSensor() NN_NOEXCEPT;

/**
 * @brief       Seven 向け SixAxisSensor のサンプリングを停止します。
 *
 * @details     Seven 向け SixAxisSensor のサンプリングを停止します。
 *              既にサンプリングが停止している場合は何もしません。
 *
 * @pre
 *              InitializeSevenSixAxisSensor() の呼び出しが完了している
 */
void StopSevenSixAxisSensor() NN_NOEXCEPT;

/**
 * @brief       Seven 向け SixAxisSensor の入力状態を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に、Seven 向け SixAxisSensor から入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              Seven 向け SixAxisSensor のサンプリング処理が開始されていない場合は、利用可能な入力状態がないためバッファには何も行わず 0 を返します。
 *              読み出し可能な入力状態の最大数は SevenSixAxisSensorStateCountMax 個です。
 *
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は SevenSixAxisSensorState::samplingNumber を参照してください。
 *
 *              サンプリングが開始された状態でアプリがインフォーカス状態ではない状態に遷移した場合、
 *              姿勢回転の追従やジャイロセンサーのゼロ点ドリフトの追従を目的として、低周波数でサンプリングを継続します。
 *              ヨー方向の姿勢回転は角速度の累積的な計算により導出するため、低周波数でのサンプリングにより精度が低下します。
 *
 *              サンプリングが開始された状態でスリープ状態に遷移した場合、サンプリング処理が停止します。
 *              スリープ状態から復帰後、再度サンプリングが開始されますがサンプリング状態が切り替わるため、
 *              SevenSixAxisSensorState::rotation の基準姿勢が変わります。
 *              基準姿勢の変化を検出したい場合は、 SevenSixAxisSensorState::coordinateNumber を参照してください。
 *
 * @param[out]  outStates                   入力状態を読み出すバッファ（配列）
 * @param[in]   count                       読み出す入力状態の数
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - outStates != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 *              - accelerometer の絶対値は AccelerometerMax 以下
 *              - angularVelocity の絶対値は AngularVelocityMax 以下
 */
int GetSevenSixAxisSensorStates(SevenSixAxisSensorState* outStates,
                             int count) NN_NOEXCEPT;

/**
 * @brief       Seven 向け SixAxisSensor が静止状態にあるかどうかを取得します。
 *
 * @details     Seven 向け SixAxisSensor が静止状態かどうかを取得します。
 *              以下の条件を満たした状態が一定時間継続したときに静止状態と判定します。
 *              - 各軸の加速度の変位が、ある閾値以内であること
 *              - 各軸の角速度の変位が、ある閾値以内であること
 *
 *              Seven 向け SixAxisSensor がサンプリングを開始していない場合は false を返します。
 *              サンプリングが開始された状態でアプリがインフォーカス状態ではない状態に遷移した場合は、
 *              インフォーカス状態と同様の静止判定を行った結果を返します。
 *
 *              センサー自身のハードウェア特性に基づくノイズが内部的に発生することが避けられないため、
 *              本関数では 6 軸センサーの種類に応じた最適な閾値を内部的に用いることで静止判定を行っています。
 *              コントローラーが静止状態にあるかどうかを判定する際は、本関数の利用を推奨します。
 *
 * @pre
 *              InitializeSevenSixAxisSensor() の呼び出しが完了している
 *
 * @return      Seven 向け SixAxisSensor が静止状態かどうかを返します。
 */
bool IsSevenSixAxisSensorAtRest() NN_NOEXCEPT;

//! @}

}} // namespace nn::hid
