﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Palma 固有の機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_Result.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/hid/hid_ResultPalma.h>

namespace nn { namespace hid {

/**
 * @brief Palma に対して接続単位でユニークに割り当てられるコネクションハンドルです。
 */
struct PalmaConnectionHandle
{
    uint64_t _storage;
};

/**
 * @brief Palma のアクティビティのエントリのインデックスを示す型です。
 */
typedef uint16_t PalmaActivityIndex;

/**
 * @brief Palma の RGBLED パターンのエントリのインデックスを示す型です。
 */
typedef uint16_t PalmaRgbLedPatternIndex;

/**
 * @brief Palma の音源データのエントリのインデックスを示す型です。
 */
typedef uint16_t PalmaWaveIndex;

/**
 * @brief Palma の Play Log のエントリのインデックスを示す型です。
 */
typedef uint16_t PalmaPlayLogFieldIndex;

/**
 * @brief       Palma の FrMode 時の動作種別です。
 */
enum PalmaFrModeType
{
    PalmaFrModeType_Off,                         //!< FrMode がオフになっています。
    PalmaFrModeType_B01,                         //!< B01 で動作します。
    PalmaFrModeType_B02,                         //!< B02 で動作します。
    PalmaFrModeType_B03,                         //!< B03 で動作します。
    PalmaFrModeType_Downloaded,                  //!< Downloaded で動作します。
};

/**
 * @brief Palma の Application Section のサイズ
 */
const size_t PalmaApplicationSectionSize = 0x1000;

/**
 * @brief Palma の Application Section の読み書きにおける 1回あたりの最大長です。
 */
const size_t PalmaApplicationSectionAccessUnitSizeMax = 256;

/**
 * @brief Palma の Application Section の読み書きの際に利用するバッファを定義した型です。
 */
struct PalmaApplicationSectionAccessBuffer
{
    Bit8 raw[PalmaApplicationSectionAccessUnitSizeMax];               //!< Application Section の読み書きに用いるバッファです。
};


/**
 * @brief Palma に対しての操作に対する実行結果の種類です
 */
enum PalmaOperationType
{
    PalmaOperationType_PlayActivity,                                  //!< Activity の再生
    PalmaOperationType_SetFrModeType,                                 //!< FrMode の動作種別の設定
    PalmaOperationType_ReadStep,                                      //!< Step 機能の値の読み出し
    PalmaOperationType_EnableStep,                                    //!< Step 機能の有効/無効切り替え
    PalmaOperationType_ResetStep,                                     //!< Step 機能の値のリセット
    PalmaOperationType_ReadApplicationSection,                        //!< Application Section の読み出し
    PalmaOperationType_WriteApplicationSection,                       //!< Application Section への書き込み
    PalmaOperationType_ReadUniqueCode,                                //!< UniqueCode の読み出し
    PalmaOperationType_SetUniqueCodeInvalid,                          //!< UniqueCode の無効化
    PalmaOperationType_WriteActivityEntry,                            //!< Activity Entry の書き込み
    PalmaOperationType_WriteRgbLedPatternEntry,                       //!< RGB LED Entry の書き込み
    PalmaOperationType_WriteWaveEntry,                                //!< Wave Entry の書き込み
    PalmaOperationType_ReadDataBaseIdentificationVersion,             //!< データベース識別バージョンの読み出し
    PalmaOperationType_WriteDataBaseIdentificationVersion,            //!< データベース識別バージョンの書き込み
    PalmaOperationType_SuspendFeature,                                //!< Feature Set の機能の一時的な停止/再開
    PalmaOperationType_ReadPlayLog,                                   //!< Play Log の値の読み出し
    PalmaOperationType_ResetPlayLog,                                  //!< Play Log の値のリセット
};

/**
 * @brief 音源データベースの種類
 */
enum PalmaWaveSet
{
    PalmaWaveSet_Small,    //!< 音源データベースの種類 (Small)
    PalmaWaveSet_Medium,   //!< 音源データベースの種類 (Medium)
    PalmaWaveSet_Large,    //!< 音源データベースの種類 (Large)
};

/**
 * @brief Palma ActivityDataBase の音源データベース Small の最大サイズです。
 */
const size_t PalmaWaveSetSmallSizeMax = 8 * 1024;

/**
 * @brief Palma ActivityDataBase の音源データベース Medium の最大サイズです。
 */
const size_t PalmaWaveSetMediumSizeMax = 16 * 1024;

/**
 * @brief Palma ActivityDataBase の音源データベース Large の最大サイズです。
 */
const size_t PalmaWaveSetLargeSizeMax = 32 * 1024;

/**
 * @brief Palma のアクティビティのエントリを表す型です。
 */
struct PalmaActivityEntry
{
    PalmaRgbLedPatternIndex rgbLedPatternIndex;     //!< 再生する RGB LED パターンのインデックス
    PalmaWaveSet waveSet;                           //!< 再生する音源データベースの種類
    PalmaWaveIndex waveIndex;                       //!< 再生する音源データのインデックス
};

/**
 * @brief Palma の ActivityDataBase が不正な状態であることを表すデータベース識別バージョンです。
 */
const int32_t DataBaseIdentificationVersionInvalid = 0xFFFFFFFF;

/**
 * @brief Palma の Feature Set の定義です。
 */
struct PalmaFeature
{
    typedef ::nn::util::BitFlagSet<32, PalmaFeature>::Flag<0> FrMode;         //!< Palma の FrMode 機能
    typedef ::nn::util::BitFlagSet<32, PalmaFeature>::Flag<1> RumbleFeedback; //!< Palma の接続状態を示す振動フィードバック機能
    typedef ::nn::util::BitFlagSet<32, PalmaFeature>::Flag<2> Step;           //!< Palma の Step 機能
    typedef ::nn::util::BitFlagSet<32, PalmaFeature>::Flag<3> MuteSwitch;     //!< Palma の Mute 切り替え機能
};

/**
 * @brief Palma の Feature Set を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, PalmaFeature> PalmaFeatureSet;


/**
 * @brief Palma に対しての操作に対する実行結果です。
 */
struct PalmaOperationInfo
{
    PalmaOperationType type;                            //!< 操作結の種類です。
    Result result;                                      //!< 実行結果を表す Result です。
    /**
      * @brief Palma に対する各操作固有の処理結果です
      */
    union IndividualInfo
    {
        /**
          * @brief Activity 再生に対する操作固有の処理結果です
          */
        struct InfoPlayActivity
        {
            PalmaActivityIndex index;                        //!< Activity のインデックス
        } playActivity;                                      //!< Activity 再生に対する操作固有の処理結果です
        /**
          * @brief FrMode の動作種別の設定操作固有の処理結果です
          */
        struct InfoSetFrModeType
        {
            PalmaFrModeType frModeType;                      //!< FrMode の動作種別
        } setFrModeType;                                     //!< FrMode の動作種別の設定操作固有の処理結果です
        /**
          * @brief Step 機能の値の読み出し操作固有の処理結果です
          */
        struct InfoReadStep
        {
            uint32_t step;                                    //!< Step 機能で取得される値
        } readStep;                                           //!< Step 機能の値の読み出し操作固有の処理結果です
        /**
          * @brief Step 機能の有効/無効切り替え操作固有の処理結果です
          */
        struct InfoEnablePalmaStep
        {
            bool isEnabled;                                   //!< 有効/無効の設定
        } enablePalmaStep;                                    //!< Step 機能の有効/無効切り替え操作固有の処理結果です
        /**
          * @brief Application Section の読み出し操作固有の処理結果です
          */
        struct InfoReadApplicationSection
        {
            int32_t address;                                  //!< 読み込みを行ったアドレスです。
            int32_t size;                                     //!< 読み出されたデータのサイズです。
            PalmaApplicationSectionAccessBuffer buffer;       //!< 読み出されたデータです。
        } readApplicationSection;                             //!< Application Section の読み出し操作固有の処理結果です
        /**
          * @brief Application Section への書き込み操作固有の処理結果です
          */
        struct InfoWriteApplicationSection
        {
            int32_t address;                                  //!< 書き込みを行ったアドレスです。
            int32_t size;                                     //!< 書き込みを行ったサイズです。
        } writeApplicationSection;                            //!< Application Section への書き込み操作固有の処理結果です
        /**
          * @brief UniqueCode 読み出し操作固有の処理結果です
          */
        struct InfoReadUniqueCode
        {
            Bit8 uniqueCode[16];                              //!< UniqueCode の内容
            Bit8 contentId[4];                                //!< UniqueCode の識別子
        } readUniqueCode;                                     //!< UniqueCode 読み出し操作固有の処理結果です
        /**
          * @brief ActivityEntry の書き込み操作固有の処理結果です
          */
        struct InfoWriteActivityEntry
        {
            PalmaActivityIndex index;                         //!< 書き込みを行ったインデックスです。
        } writeActivityEntry;                                 //!< ActivityEntry の書き込み操作固有の処理結果です
        /**
          * @brief RgbLedPatternEntry の書き込み操作固有の処理結果です
          */
        struct InfoWriteRgbLedPatternEntry
        {
            PalmaRgbLedPatternIndex index;                    //!< 書き込みを行ったインデックスです。
        } writeRgbLedPatternEntry;                            //!< RgbLedPatternEntry の書き込み操作固有の処理結果です
        /**
          * @brief WaveEntry の書き込み操作固有の処理結果です
          */
        struct InfoWriteWaveEntry
        {
            PalmaWaveSet waveSet;                             //!< 書き込みを行った音源データベースの種類です。
            PalmaWaveIndex index;                             //!< 書き込みを行ったインデックスです。
        } writeWaveEntry;                                     //!< WaveEntry の書き込み操作固有の処理結果です
        /**
          * @brief DataBaseIdentificationVersion 読み出し操作固有の処理結果です
          */
        struct InfoReadDataBaseIdentificationVersion
        {
            int32_t version;                                  //!< 読み出されたデータベース識別バージョン
        } readDataBaseIdentificationVersion;                  //!< データベース識別バージョン 読み出し操作固有の処理結果です
        /**
          * @brief DataBaseIdentificationVersion 書き込み操作固有の処理結果です
          */
        struct InfoWriteDataBaseIdentificationVersion
        {
            int32_t version;                                  //!< 書き込まれたデータベース識別バージョン
        } writeDataBaseIdentificationVersion;                 //!< データベース識別バージョン 書き込み操作固有の処理結果です
        /**
          * @brief Feature Set の機能の一時的な停止/再開に対する操作固有の処理結果です
          */
        struct InfoSuspendPalmaFeature
        {
            PalmaFeatureSet suspendFeatureSet;                //!< 一時的に停止された機能の Feature Set
        } suspendPalmaFeature;                                //!< Feature Set の機能の一時的な停止/再開に対する操作固有の処理結果です
        /**
          * @brief Play Log の読み出し操作固有の処理結果です
          */
        struct InfoReadPalmaPlayLog
        {
            PalmaPlayLogFieldIndex index;                     //!< 読み出しを行った Play Log のインデックス
            int32_t size;                                     //!< 読み出された Play Log の値のサイズ
            Bit8 raw[8];                                      //!< 読み出された Play Log の値
        } readPalmaPlayLog;                                   //!< Play Log の読み出し操作固有の処理結果です
        /**
          * @brief Play Log の値をリセットする操作固有の処理結果です
          */
        struct InfoResetPalmaPlayLog
        {
            PalmaPlayLogFieldIndex index;                     //!< 値をリセットした Play Log のインデックス
        } resetPalmaPlayLog;                                  //!< Play Log の値をリセットする操作固有の処理結果です
        Bit8 raw[320];
    } individual;  //!<  Palma に対する各操作固有の処理結果です
};

}} // namespace nn::hid
