﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       本体 6 軸センサーのデバッグ機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

namespace nn { namespace hid { namespace debug {

const int ConsoleSixAxisSensorCountStateCountMax = 16;      //!< 内部的に保持される ConsoleSixAxisSensorCountState の最大数

/**
 * @brief       3軸をもつセンサーの値 [count] を表す構造体です。
 */
struct SensorCountState
{
     int16_t x;  //!< センサーの x 軸の値
     int16_t y;  //!< センサーの y 軸の値
     int16_t z;  //!< センサーの z 軸の値
};

/**
 * @brief       6軸センサーの入力状態 [count] を表す構造体です。
 */
struct ConsoleSixAxisSensorCountState
{
    ::nn::TimeSpanType        deltaTime;        //!< 前回サンプリングされてから今回のサンプリングまでの経過時間です。
    int64_t                   samplingNumber;   //!< 6 軸センサーの入力状態がサンプリングされる度に増加する値です。
    int64_t                   tick;             //!< サンプリング時点でのシステムチック値です。
    SensorCountState          acceleration;     //!< 加速度のカウント値。
    SensorCountState          angularVelocity;  //!< 角速度のカウント値。
    NN_PADDING4;
};

//! @name 本体 6 軸センサー関連 API
//! @{

/**
 * @brief       本体 6 軸センサーの生値 [count] を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に、本体 6 軸センサーから入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              読み出し可能な入力状態の最大数は ConsoleSixAxisSensorCountStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は SixAxisSensorCountState::samplingNumber を参照してください。
 *              本 API は工程検査での利用に限定され、製品環境では利用できません。
 *
 * @param[out]  pOutStates                  入力状態を読み出すバッファ（配列）
 * @param[in]   count                       読み出す入力状態の数
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - InitializeSevenSixAxisSensor() の呼び出しが完了している
 *              - pOutStates != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 */
int GetConsoleSixAxisSensorCountStates(ConsoleSixAxisSensorCountState* pOutStates,
                                       int count) NN_NOEXCEPT;

/**
 * @brief       本体 6 軸センサーのサンプリング周波数を返します。
 *
 * @details     本体 6 軸センサーの内部サンプリング周波数を返します。
 *
 * @pre
 *              - ConsoleSixAxisSensor が初期化状態にある
 */
int64_t GetConsoleSixAxisSensorSamplingFrequency() NN_NOEXCEPT;

/**
 * @brief       ConsoleSixAxisSensor をファイナライズします。
 *
 * @details     事前に nn::hid::InitializeConsoleSixAxisSensor() を呼び出した回数と同じだけ
 *              呼びだされた際に ConsoleSixAxisSensor の初期化状態を解除します。
 *
 * @pre
 *              - ConsoleSixAxisSensor が初期化状態にある
 */
void FinalizeConsoleSixAxisSensor() NN_NOEXCEPT;

}}} // namespace nn::hid::debug
