﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx/gfx_Common.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
#include <nn/gfx/gfx_DeviceData-api.vk.1-os.win32.h>
#elif defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
#include <nn/gfx/gfx_DeviceData-api.vk.1-os.horizon.h>
#else
    #error
#endif

namespace nn {
namespace gfx {

class DeviceInfo;
class DisplayHandle;

// VK 専用インターフェース
/**
* @brief 確保するメモリーの用途です。
*/
enum AllocationScope
{
    AllocationScope_GfxManage = -1,         //!< Vulkanではなくgfx側の管理用メモリーです。
    AllocationScope_Command = 0,            //!< Vulkan Commandに紐付けられたメモリーです。
    AllocationScope_Object,                 //!< Vulkan Objectに紐付けられたメモリーです。
    AllocationScope_Cache,                  //!< Vulkan Cacheに紐付けられたメモリーです。
    AllocationScope_Device,                 //!< Vulkan Deviceに紐付けられたメモリーです。
    AllocationScope_Instance                //!< Vulkan Instanceに紐付けられたメモリーです。
};

/**
* @brief 確保するメモリーの種別です。
*/
enum InternalAllocationType
{
    InternalAllocationType_Executable = 0   //!< Vulkanのhostの実行に使用されるメモリーです。
};

/**
* @brief 各種メモリーアロケーターの関数型です。
*
* @details
* 各引数、戻り値の仕様は、Vulkanのユーザーアロケーターの仕様に準じます。
*/
typedef void* ( *VkAllocateFunction )( void* pUserData, size_t size, size_t alignment, AllocationScope allocationScope );

typedef void* ( *VkReallocateFunction )( void* pUserData, void* pOriginal, size_t size, size_t alignment, AllocationScope allocationScope );

typedef void ( *VkFreeFunction )( void* pUserData, void* pMemory );

typedef void ( *VkNotifyInternalAllocationFunction )( void* pUserData, size_t size, InternalAllocationType internalAllocationType, AllocationScope allocationScope );

typedef void ( *VkNotifyInternalFreeFunction )( void* pUserData, size_t size, InternalAllocationType internalAllocationType, AllocationScope allocationScope );

/**
* @brief 各種メモリーアロケーター関数のポインターを格納する構造体です。
*/
struct VkAllocator
{
    void*                               pUserData;                  //!< ユーザーデータです。
    VkAllocateFunction                  pAllocate;                  //!< メモリー確保関数の関数ポインターです。
    VkReallocateFunction                pReallocate;                //!< メモリー再確保関数の関数ポインターです。
    VkFreeFunction                      pFree;                      //!< メモリー解放関数の関数ポインターです。
    VkNotifyInternalAllocationFunction  pNotifyInternalAllocation;  //!< 内部メモリー確保通知関数の関数ポインターです。
    VkNotifyInternalFreeFunction        pNotifyInternalFree;        //!< 内部メモリー解放通知関数の関数ポインターです。
};

namespace detail {

template<>
class DeviceImpl< ApiVariationVk1 >
    : public DataContainer< DeviceImplData< ApiVariationVk1 > >
{
    NN_DISALLOW_COPY( DeviceImpl );

public:
    typedef ApiVariationVk1 Target;
    typedef DeviceInfo InfoType;

    static const bool IsWorkingMemoryRequired = false;

    DeviceImpl() NN_NOEXCEPT;

    ~DeviceImpl() NN_NOEXCEPT;

    void Initialize( const InfoType& info ) NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    /**
    * @brief メモリーアロケーター関数を設定します。
    *
    * @param[in] pVkAllocator メモリーアロケーター関数ポインター構造体へのポインター
    *
    * @post
    * - デバイスにメモリーアロケーターが設定されている
    *
    * @details
    * pVkAllocatorにNULLが指定された場合は、メモリーアロケーターが未設定の状態となり、内部で独自の方法でメモリーを確保します。
    */
    void Vk1SetAllocator( VkAllocator* pVkAllocator ) NN_NOEXCEPT;
};

}
}
}
