﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>
#include <nn/ens/detail/util/ens_DataHolder.h>
#include <nn/ens/detail/util/ens_JsonPath.h>

namespace nn { namespace ens { namespace detail { namespace util {

/**
 * @brief   レスポンスを読み込むクラス
 *
 * @details
 */
class ResponseReader
{
public:
    /**
     * @brief   データの読み込みコールバック
     *
     * @details
     */
    using ReadCallback = bool (*)(const JsonPath& jsonPath, const DataHolder& holder, void* pParam);

public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    ResponseReader() NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    virtual ~ResponseReader() NN_NOEXCEPT;

    /**
     * @brief   データの読み込みコールバックを設定します。
     *
     * @param[in]   pCallback   コールバック
     * @param[in]   pParam      コールバックパラメータ
     *
     * @details
     */
    virtual void SetCallback(ReadCallback pCallback, void* pParam) NN_NOEXCEPT = 0;

    /**
     * @brief   レスポンスを読み込みます。
     *
     * @param[in]   pData   レスポンス
     * @param[in]   size    レスポンスのサイズ
     *
     * @return  妥当なレスポンスを読み込んだかどうか
     *
     * @pre
     *  - pData != nullptr
     *
     * @details
     */
    virtual bool Read(const void* pData, size_t size) NN_NOEXCEPT = 0;

    /**
     * @brief   レスポンスをダンプします。
     *
     * @tparam      Reader  ResponseReader を継承したクラス
     *
     * @param[in]   pData   レスポンス
     * @param[in]   size    レスポンスのサイズ
     *
     * @return  妥当なレスポンスを読み込んだかどうか
     *
     * @pre
     *  - pData != nullptr
     *
     * @details
     *   本関数は NN_DETAIL_ENS_INFO を使用し、レスポンスをダンプします。
     */
    template <class Reader>
    static bool Dump(const void* pData, size_t size) NN_NOEXCEPT
    {
        Reader reader;

        reader.SetCallback(ResponseDumpCallback, nullptr);

        return reader.Read(pData, size);
    }

private:
    //
    static bool ResponseDumpCallback(const JsonPath& jsonPath, const DataHolder& holder, void* pParam) NN_NOEXCEPT;
};

}}}}
