﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>

namespace nn { namespace ens { namespace detail { namespace util {

/**
 * @brief   JSON パスオブジェクト
 *
 * @details
 *  データの特定の要素を指定するためのパスを表すオブジェクトです。@n
 *  データの連想配列のキーには文字列以外のオブジェクトも指定できますが、本クラスを使用する場合、キーは文字列のみに制限されます。
 *
 *  JSON パスは以下の文字列を使用して表現されます。
 *
 *  - $
 *      - ルート要素を表します。
 *  - .
 *      - オブジェクトの階層を表します。
 *  - []
 *      - 配列を表します。
 *  - キー名
 *      - キー名を表します。
 *  - *
 *      - キー名に対するワイルドカードを表します。
 *
 *  {"compact":true,"schema":0} の schema 要素を表現したい場合、JSON パス文字列は "$.schema" になります。
 */
class JsonPath
{
public:
    /**
     * @brief   ノードの最大深さです。
     *
     * @details
     *  ノードのネストは、ルート要素・オブジェクト・配列・キーを発見する度に発生します。
     */
    static const int NodeDepthMax = 32;

    /**
     * @brief   JSON パス文字列のバッファサイズです。
     *
     * @details
     */
    static const size_t PathSize = 256;

public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    JsonPath() NN_NOEXCEPT;

    /**
     * @brief   内部状態をクリアします。
     *
     * @details
     */
    void Clear() NN_NOEXCEPT;

    /**
     * @brief   JSON パス文字列の比較を行います。
     *
     * @param[in]   pPath   JSON パス文字列
     *
     * @return  JSON パス文字列が一致したかどうか
     *
     * @pre
     *  - pPath != nullptr
     *
     * @details
     *  本関数は、JSON パスの完全一致比較を行います。@n
     *  完全一致比較しか行わない場合、 Match() より高速に動作します。
     *
     *  {"array":[{"id":0,"name":"A"},{"id":1,"name":"B"}]}
     *
     *  上記の 2 番目の id 要素を取得したい場合、JSON パス文字列を以下のように記述します。
     *
     *  - "$.array[1].id"
     */
    bool Compare(const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   JSON パス文字列のパターンマッチング比較を行います。
     *
     * @param[in]   pPath   JSON パス文字列
     *
     * @return  JSON パス文字列がマッチしたかどうか
     *
     * @pre
     *  - pPath != nullptr
     *
     * @details
     *  本関数は、ワイルドカードを使用したパターンマッチングを行います。
     *
     *  実装を簡略化するため、以下の制約があります。
     *
     *  - キー名にワイルドカード文字が含まれる場合、パターンマッチングは正しく動作しません。
     *  - 1 階層に複数のワイルドカード文字は指定できません。
     *  - 複数階層を跨いだパターンマッチングはできません。
     *
     *  {"array":[{"id":0,"name":"A"},{"id":1,"name":"B"}]}
     *
     *  上記の id 要素をすべて取得したい場合は、JSON パス文字列を以下のように記述します。
     *
     *  - "$.array[*].id"
     */
    bool Match(const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   JSON パス文字列を取得します。
     *
     * @return  JSON パス文字列
     *
     * @details
     */
    const char* GetString() const NN_NOEXCEPT;

public:
    /**
     * @brief   オブジェクトノードを追加します。
     *
     * @return  追加できたかどうか
     *
     * @details
     *  本関数はパーサー専用関数です。@n
     *  ノードの深さが上限に達していた場合や、JSON パスのバッファが不足していた場合、失敗します。
     */
    bool PushObject() NN_NOEXCEPT;

    /**
     * @brief   配列ノードを追加します。
     *
     * @return  追加できたかどうか
     *
     * @details
     *  本関数はパーサー専用関数です。@n
     *  ノードの深さが上限に達していた場合や、JSON パスのバッファが不足していた場合、失敗します。
     */
    bool PushArray() NN_NOEXCEPT;

    /**
     * @brief   キーノードを追加します。

     * @param[in]   pKey    キー
     * @param[in]   length  キーの長さ
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  追加できたかどうか
     *
     * @details
     *  本関数はパーサー専用関数です。@n
     *  ノードの深さが上限に達していた場合や、JSON パスのバッファが不足していた場合、失敗します。
     */
    bool PushKey(const char* pKey, size_t length) NN_NOEXCEPT;

    /**
     * @brief   親ノードが配列ノードだった場合、親ノードの配列の要素数をインクリメントします。
     *
     * @return  成功したかどうか
     *
     * @details
     *  本関数はパーサー専用関数です。@n
     *  JSON パスのバッファが不足していた場合、失敗します。
     */
    bool IncrementArrayCountIfParentNodeIsArray() NN_NOEXCEPT;

    /**
     * @brief   ノードを一つ取り出します。
     */
    void Pop() NN_NOEXCEPT;

    /**
     * @brief   親ノードがキーノードだった場合、ノードを一つ取り出します。
     */
    void PopIfParentNodeIsKey() NN_NOEXCEPT;

private:
    //
    enum NodeType : int8_t
    {
        NodeType_Root,
        NodeType_Object,
        NodeType_Array,
        NodeType_Key,
    };

    //
    struct Node
    {
        NodeType type;
        uint16_t positionInPath;
        uint32_t arrayCount;
    };

private:
    //
    Node m_Nodes[NodeDepthMax];
    uint16_t m_Depth;
    //
    char m_Path[PathSize];
    uint16_t m_PathLength;

private:
    //
    size_t GetPathRemainSize() const NN_NOEXCEPT;
    size_t GetPathRemainSize(uint16_t depth) const NN_NOEXCEPT;
    //
    void WriteArrayBrackets() NN_NOEXCEPT;
};

}}}}
