﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>

namespace nn { namespace ens { namespace detail { namespace util {

/**
 * @brief   DataHolder クラスのデータ種別
 *
 * @details
 */
enum DataHolderType
{
    DataHolderType_Nil,        //!< 空
    DataHolderType_Boolean,    //!< 論理型
    DataHolderType_Integer,    //!< 整数型
    DataHolderType_Float,      //!< 浮動小数点数型
    DataHolderType_String,     //!< 文字列型
    DataHolderType_Binary,     //!< バイナリデータ型
    DataHolderType_Extension,  //!< 拡張データ型
    DataHolderType_BeginMap,   //!< 連想配列の開始
    DataHolderType_EndMap,     //!< 連想配列の終了
    DataHolderType_BeginArray, //!< 配列の開始
    DataHolderType_EndArray,   //!< 配列の終了
};

/**
 * @brief   出力変数の型種別
 *
 * @details
 */
enum ValueType
{
    ValueType_Boolean, //!< bool 型
    ValueType_S8,      //!< int8_t 型
    ValueType_S16,     //!< int16_t 型
    ValueType_S32,     //!< int32_t 型
    ValueType_S64,     //!< int64_t 型
    ValueType_U8,      //!< uint8_t 型
    ValueType_U16,     //!< uint16_t 型
    ValueType_U32,     //!< uint32_t 型
    ValueType_U64,     //!< uint64_t 型
    ValueType_Double,  //!< double 型
    ValueType_String,  //!< 文字列型
    ValueType_Binary,  //!< バイナリ型
};

/**
 * @brief   整数型を扱う構造体
 *
 * @details
 */
struct Integer
{
    union
    {
        int64_t s64;  //!< 符号付き整数
        uint64_t u64; //!< 符号なし整数
    };
    bool isSigned;    //!< 符号付き整数の範囲内かどうか
};

/**
 * @brief   文字列型を扱う構造体
 *
 * @details
 */
struct String
{
    const char* pValue; //!< 文字列
    size_t length;      //!< 文字列長
};

/**
 * @brief   バイナリデータ型を扱う構造体
 *
 * @details
 */
struct Binary
{
    const void* pValue; //!< データ
    size_t length;      //!< データサイズ
};

/**
 * @brief   拡張データ型を扱う構造体
 *
 * @details
 */
struct Extension
{
    int8_t type;        //!< データ種別
    const void* pValue; //!< データ
    size_t length;      //!< データサイズ
};

/**
 * @brief   任意のデータを格納する構造体
 *
 * @details
 */
struct DataHolder
{
    DataHolderType type;     //!< データ種別

    union
    {
        bool boolean;        //!< 論理値
        Integer integer;     //!< 整数
        double float64;      //!< 浮動小数点数
        String string;       //!< 文字列
        Binary binary;       //!< バイナリデータ
        Extension extension; //!< 拡張データ
    };

    /**
     * @brief   bool 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(bool* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   int8_t 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(int8_t* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   int16_t 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(int16_t* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   int32_t 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(int32_t* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   int64_t 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(int64_t* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   uint8_t 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(uint8_t* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   uint16_t 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(uint16_t* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   uint32_t 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(uint32_t* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   uint64_t 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(uint64_t* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   double 型として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(double* pOutValue) const NN_NOEXCEPT;

    /**
     * @brief   文字列として値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     * @param[in]   size        出力変数のサイズ
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - size > 0
     *
     * @details
     */
    bool Get(char* pOutValue, size_t size) const NN_NOEXCEPT;

    /**
     * @brief   バイナリとして値を取得します。
     *
     * @param[out]  pOutSize    バイナリサイズ
     * @param[out]  pOutValue   出力変数
     * @param[in]   size        出力変数のサイズ
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutSize != nullptr
     *  - pOutValue != nullptr
     *  - size > 0
     *
     * @details
     */
    bool Get(size_t* pOutSize, void* pOutValue, size_t size) const NN_NOEXCEPT;

    /**
     * @brief   バイナリとして値を取得します。
     *
     * @param[out]  pOutValue   出力変数
     *
     * @return  値が取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *
     * @details
     */
    bool Get(ReceiveBuffer* pOutValue) const NN_NOEXCEPT;
};

}}}}
