﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn{ namespace capsrv{

    //! @brief アルバムにあるファイルの種類を示す変数の型です。
    typedef uint8_t AlbumFileContentsType;

    //! @brief アルバムにあるファイルの種類を示す値です。
    //! @details
    //! この値は AlbumFileId に含まれます。
    //! 各列挙子は 0～63 の範囲であり、
    //! AlbumFileContentsFlag のビット番号としても使用されます。
    //!
    //! - GetAlbumFileList() で取得できます。
    //! - GetAlbumUsage() の集計種別です。
    enum AlbumFileContents
    {

        //! @brief 画面写真ファイルです。
        AlbumFileContents_ScreenShot = 0,

        //! @brief 動画ファイルです。
        AlbumFileContents_Movie = 1,

        //! @brief アプリが作成した Scoop 対象外の画面写真ファイルです。
        AlbumFileContents_ExtraScreenShot = 2,

        //! @brief アプリが作成した Scoop 対象外の動画ファイルです。
        AlbumFileContents_ExtraMovie = 3,
    };

    //! @brief アルバムのファイルの組合せを示すためのフラグです。
    typedef nn::util::BitFlagSet<64> AlbumFileContentsFlag;

    //! @brief SDK がサポートするファイルの種類の数です。
    //! @details
    //! この値は SDK のサポートするファイルの種類の数を示します。
    //! アプレットが対応しているファイルタイプの数は、アプレットの定数として定義してください。
    static const int AlbumFileContentsCount = 4;


    //! @brief 指定された AlbumFileContentsType が Extra コンテンツか否かを返す
    inline bool IsExtraAlbumFileContents(AlbumFileContentsType contents) NN_NOEXCEPT
    {
        return contents == AlbumFileContents_ExtraScreenShot || contents == AlbumFileContents_ExtraMovie;
    }

}}
