﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#ifdef HIDINPUTER_EXPORTS
#define HIDINPUTER_API extern "C" __declspec(dllexport)
#else
#define HIDINPUTER_API extern "C" __declspec(dllimport)
#endif

#include <autoTestAssistTool/HidInputerTypes.h>
#include <autoTestAssistTool/Common\autoTestAssistTool_LibTypes.h>

/**
*   @file
*   @brief  HID 入力操作に関する API
*/

/*---------------------------------------------------------------------
/ 初期化・終了処理
/--------------------------------------------------------------------*/
/**
* @brief        HidInputer ライブラリの初期化処理を行います。
* @param[in]    serialNumber    対象ターゲットのシリアル番号

* @return       初期化処理の成否が返ります。
* @retval       HidInputerResult_Success             初期化に成功したときの値です。
* @retval       HidInputerResult_Unexpected          不測のエラーが発生したときの値です。
* @retval       HidInputerResult_TargetNotFound      対象ターゲットが見つからなかったときの値です。

* @details      初期化が完了していない状態では、Hid 操作が正常に実行されません。
*/
HIDINPUTER_API HidInputerResult Initialize( const SerialNumberString& serialNumber );

/**
* @brief        NX-Win32、NX-x64 アプリケーションに対して HidInputer ライブラリの初期化処理を行います。

* @return       初期化処理の成否が返ります。
* @retval       HidInputerResult_Success             初期化に成功したときの値です。
* @retval       HidInputerResult_Unexpected          不測のエラーが発生したときの値です。

* @details      NX-Win32, NX-x64 アプリケーションに対して、HidInputer ライブラリの初期化を実行します。@n
                NX-Win32, NX-x64 アプリケーションが複数起動している場合、全てのアプリケーションに対して HidInputer の操作が反映されます。@n
                初期化が完了していない状態では、Hid 操作が正常に実行されません。
*/
HIDINPUTER_API HidInputerResult InitializeForLocal();

/**
* @brief    HidInputer ライブラリの終了処理を行います。

* @pre      Initialize 関数の呼び出しが完了している。
* @details  HidInputer ライブラリの終了処理を行います。 @n
            この関数の実行後は、Initialize 関数が実行されるまで Hid 操作が正常に行われなくなります。
*/
HIDINPUTER_API void Finalize();

/*---------------------------------------------------------------------
/ HID 入力情報の記録
/--------------------------------------------------------------------*/
/**
* @brief        ターゲットの Hid 入力の記録を開始します。

* @return       操作の成否が返ります。
* @retval       HidInputerResult_Success             操作に成功したときの値です。
* @retval       HidInputerResult_Unexpected          不測のエラーが発生したときの値です。
* @retval       HidInputerResult_NotInitialized      未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。
* @details      接続されているターゲットへの Hid 入力操作を記録します。@n
*               記録した操作は、ExportHidInputRecordsWithPythonScript() より、スクリプトとして出力することができます。@n
*               記録できる操作情報は、以下の入力情報です。@n
*
*               @li InputDirector 経由での開発機へのゲームパッド入力
*               @li 開発機に接続されたデバッグパッド入力
*               @li 開発機に接続された Joy-Con 入力
*               @li 開発機に接続された Switch Pro Controller 入力
*               @li 開発機へのタッチスクリーン入力
*
*               Hid 操作記録用の API（BeginHidInputRecording, EndHidInputRecording, ExportHidInputRecordsWithPythonScript）はスレッドセーフではありません。@n
*               複数スレッドから実行された場合の動作は保障されません。@n
*               呼び出し時に既に記録されていた Hid 入力記録はリセットされます。
*/
HIDINPUTER_API HidInputerResult BeginHidInputRecording();

/**
* @brief        ターゲットの Hid 入力の記録を中止します。

* @return       操作の成否が返ります。
* @retval       HidInputerResult_Success             操作に成功したときの値です。
* @retval       HidInputerResult_NotInitialized      未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。
* @details      初期化が完了していない状態では正常に実行されません。@n
*               接続されているターゲットへの Hid 入力操作を中止します。
*/
HIDINPUTER_API HidInputerResult EndHidInputRecording();

/**
* @brief        記録している Hid 入力操作をスクリプトとして出力します。

* @param[in]    exportFilename          出力先のファイル名

* @return       操作の成否が返ります。
* @retval       HidInputerResult_Success             操作に成功したときの値です。
* @retval       HidInputerResult_NotInitialized      未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。
* @details      記録している Hid 入力操作を python スクリプトとして出力します。
*               指定するファイル名は UTF-8 エンコードの文字列として指定してください。@n
*               出力されたスクリプトを実行するためには、パッケージサンプルに同梱されている HACDllDriver.py が同ディレクトリに配置されている必要があります。@n
*               Hid 入力を記録している状態では実行できません。
*/
HIDINPUTER_API HidInputerResult ExportHidInputRecordsWithPythonScript(const char* exportFilename);

/*---------------------------------------------------------------------
/ コントローラー接続解除
/--------------------------------------------------------------------*/
/**
* @brief        未接続のコントローラーを接続します。
* @param[out]    pOutControllerId                                    接続したコントローラーのIDの出力先

* @return       コントローラー接続の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_ConnectControllerFailed         コントローラー接続に失敗したときの値です。
* @retval       ControllerResult_ControllerCountOver             接続できるコントローラー数を超えたときの値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      未接続のコントローラーを探し、Bluetooth 接続の SwitchProController として接続します。 @n
                接続に成功すると、controllerId へ接続したコントローラーの ID を渡します。@n
                任意のコントローラー種別、接続インターフェース、色を指定する場合は、AddControllerWithDeviceInfo をご利用ください。@n
                コントローラーは、デバッグコントローラーを除いて最大 8 台までしか接続できません。
*/
HIDINPUTER_API ControllerResult AddController(int *pOutControllerId);

/**
* @deprecated   この関数は廃止予定です。AddController、もしくは AddControllerWithDeviceInfo を利用して接続するようにしてください。

* @brief        指定されたIDのコントローラーを接続します。

* @param[in]    controllerId        接続するコントローラーのID

* @return       コントローラー接続の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_ConnectControllerFailed         コントローラー接続に失敗したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      指定されたコントローラー ID として、Bluetooth 接続の SwitchProController を接続します。 @n
*               既に接続されたコントローラー ID が指定された場合は、接続に失敗します。@n
                0 <= controllerId <= 7 の範囲の controllerId を指定することができ、
                ControllerId_ControllerId01 を指定したとき、プレイヤー番号 0 (オリジン）のコントローラーとして接続され、
                controllerId 値の増加に対応したプレイヤー番号のコントローラーとして接続されます。@n
                コントローラーは、デバッグコントローラーを除いて最大 8 台までしか接続できません。@n
                任意のコントローラー種別、接続インターフェース、色を指定する場合は、AddControllerWithDeviceInfo をご利用ください。
*/
HIDINPUTER_API ControllerResult AddControllerForId(int controllerId);

/**
* @brief        コントローラー接続を解除します。

* @param[in]    controllerId                                        解除するコントローラーのIDです。

* @return       コントローラ解放の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
                指定されたcontrollerIdのコントローラーが事前に接続されてなければなりません。

* @details      指定されたコントローラー、デバッグコントローラーの接続を解除します。 @n
                未接続のコントローラーに対して実行された場合は失敗します。
*/
HIDINPUTER_API ControllerResult ReleaseController(int controllerId);

/**
* @brief        デバッグコントローラーを接続します。
* @param[out]    pOutControllerId                                    接続したコントローラーのIDの出力先

* @return       デバッグコントローラー接続の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_ConnectControllerFailed         コントローラー接続に失敗したときの値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      デバッグコントローラーを接続し ID を返します。 @n
                デバッグコントローラーは 2 台以上接続できません。
*/
HIDINPUTER_API ControllerResult AddDebugController(int *pOutControllerId);


/**
* @brief        デバイス情報を指定してコントローラー接続を行います。
* @param[out]   pOutControllerId                                接続したコントローラーのIDの出力先
* @param[in]    deviceInfo                                      接続するコントローラーデバイス情報

* @return       コントローラー接続の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_ConnectControllerFailed         コントローラー接続に失敗したときの値です。
* @retval       ControllerResult_InvalidDeviceInfo               不正なデバイス情報のコントローラー接続が実行されたときの値です。
* @retval       ControllerResult_ControllerCountOver             接続可能なコントローラー数を超えたときの値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      任意のデバイス種別やインターフェースタイプを指定したコントローラーを接続します。@n
                接続に成功すると、controllerId へ接続したコントローラーの ID を渡します。@n
                コントローラーは、デバッグコントローラーを除いて最大 8 台までしか接続できません。
*/
HIDINPUTER_API ControllerResult AddControllerWithDeviceInfo(int *pOutControllerId, const ControllerDeviceInfo& deviceInfo);

/**
* @brief        接続済みのコントローラーの数を取得します。

* @param[out]   pOutCount                                       接続済みのコントローラー数出力先

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                        操作に成功したときの値です。
* @retval       ControllerResult_NotInitialized                 未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      接続済みのコントローラー数を pOutValue へ渡します。
*/

HIDINPUTER_API ControllerResult GetConnectedControllerCount(int *pOutCount);

/**
* @brief        接続済みのコントローラー ID 一覧を取得します。

* @param[out]   pOutCount                                       取得した ID 数の出力先
* @param[out]   pOutIdsArray                                    ID 一覧の出力先配列
* @param[in]    arrayLength                                     pOutIdsArray の配列の要素数

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                        操作に成功したときの値です。
* @retval       ControllerResult_NotInitialized                 未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      接続済みのコントローラーの ID 一覧を pOutIdsArray へ渡します。
*/
HIDINPUTER_API ControllerResult GetConnectedControllerIds(int *pOutCount, int pOutIdsArray[], int arrayLength);

/**
* @brief        接続済みのコントローラーのデバイス情報を取得します。

* @param[out]   pOutDeviceInfo                                  取得したデバイス情報の出力先
* @param[out]   controllerId                                    対象コントローラーのID

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                        操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                     不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId            不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_NotInitialized                 未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      対象コントローラーのデバイス情報を pOutDeviceInfo へ渡します。@n
                デバッグコントローラーに対して実行した場合は、処理に失敗し、エラー値として ControllerResult_InvalidControllerId を返します。
*/
HIDINPUTER_API ControllerResult GetConnectedControllerDeviceInfo(ControllerDeviceInfo* pOutDeviceInfo, int controllerId);

/**
* @brief        接続済みのデバッグコントローラーの数を取得します。

* @param[out]   pOutCount                                       接続済みのデバッグコントローラー数出力先

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                        操作に成功したときの値です。
* @retval       ControllerResult_NotInitialized                 未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      接続済みのデバッグコントローラー数を pOutValue へ渡します。
*/
HIDINPUTER_API ControllerResult GetConnectedDebugControllerCount(int *pOutCount);

/**
* @brief        接続済みのデバッグコントローラー ID 一覧を取得します。

* @param[out]   pOutCount                                       取得した ID 数の出力先
* @param[out]   pOutIdsArray                                    ID 一覧の出力先配列
* @param[in]    arrayLength                                     pOutIdsArray の配列の要素数

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                        操作に成功したときの値です。
* @retval       ControllerResult_NotInitialized                 未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      接続済みのデバッグコントローラーの ID 一覧を pOutIdsArray へ渡します。
*/
HIDINPUTER_API ControllerResult GetConnectedDebugControllerIds(int *pOutCount, int pOutIdsArray[], int arrayLength);

/*---------------------------------------------------------------------
/ コントローラーボタン操作
/--------------------------------------------------------------------*/
/**
* @brief        指定されたコントローラーのボタンを押します。

* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    button                                              対象ボタン

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidButton                   不正なボタンに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの対象ボタンを押します。 @n
*/
HIDINPUTER_API ControllerResult PressButton(int controllerId, Button button);

/**
* @brief        指定されたコントローラーのボタンを離します。

* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    button                                              対象ボタン

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidButton                   不正なボタンに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの対象ボタンを離します。
*/
HIDINPUTER_API ControllerResult ReleaseButton(int controllerId, Button button);

/**
* @brief        指定されたコントローラーのボタンを押した後に離します。

* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    button                                              対象ボタン
* @param[in]    waitPress                                           押してから離すまでのスリープ時間 [ms]

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidButton                   不正なボタンに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの対象ボタンにプッシュ操作（押した後に離す）を実行します。 @n
                ボタンを押し、waitPress[ms] 間のスリープを実行した後に、ボタンが離されます。
*/
HIDINPUTER_API ControllerResult PushButton(int controllerId, Button button, int waitPress);

/**
* @brief        指定されたコントローラーの複数ボタンを同時に押します。

* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    buttonList                                          対象ボタン配列
* @param[in]    buttonCount                                         対象ボタン数

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidButton                   不正なボタンに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの複数ボタンを同時に押します。
*/
HIDINPUTER_API ControllerResult PressMultiButton(int controllerId, Button buttonList[], int buttonCount);

/**
* @brief        指定されたコントローラーの複数ボタンを同時に離します。
*
* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    buttonList                                          対象ボタン配列
* @param[in]    buttonCount                                         対象ボタン数

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidButton                   不正なボタンに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの複数ボタンを同時に離します。
*/
HIDINPUTER_API ControllerResult ReleaseMultiButton(int controllerId, Button buttonList[], int buttonCount);


/**
* @brief        指定されたコントローラーの複数ボタンを同時に押した後に離します。

* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    buttonList                                          対象ボタン配列
* @param[in]    buttonCount                                         対象ボタン数
* @param[in]    waitPress                                           押してから離すまでのスリープ時間 [ms]

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidButton                   不正なボタンに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの対象ボタンへ同時にプッシュ操作（押した後に離す）を実行します。 @n
                ボタンを押し、waitPress[ms] 間のスリープを実行した後に、ボタンが離されます。
*/
HIDINPUTER_API ControllerResult PushMultiButton(int controllerId, Button buttonList[], int buttonCount, int waitPress);

/**
* @brief        指定されたコントローラーで、ボタンを押した後に離す操作を対象ボタンへ順番に実行します。

* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    buttonList                                          対象ボタン配列
* @param[in]    buttonCount                                         対象ボタン数
* @param[in]    waitPress                                           押してから離すまでのスリープ時間 [ms]
* @param[in]    waitSpan                                            ボタン操作の間隔 [ms]

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidButton                   不正なボタンに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの対象ボタンにを順番にプッシュ操作（押した後に離す）を実行します。 @n
                ボタンを押し、waitSpan[ms] 間のスリープを実行した後に、ボタンが離す操作が順番に実行されます。
*/
HIDINPUTER_API ControllerResult PushMultiSeq(int controllerId, Button buttonList[], int buttonCount, int waitPress, int waitSpan);


/**
* @brief        指定されたコントローラーの入力状態を変更します。

* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    buttonList                                          押すボタン配列
* @param[in]    buttonCount                                         押すボタン数
* @param[in]    stickList                                           操作するアナログスティック種別の配列
* @param[in]    stickDegreeList                                     各アナログスティックの入力方向 [degree]
* @param[in]    stickPowerList                                      各アナログスティックの入力値(1.0 >= x >= -1.0)
* @param[in]    stickCount                                          操作するアナログスティックの数

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidStick                    不正なアナログスティックに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーのボタン、スティックの入力状態を変更します。@n
*               buttonList には押し込み状態にする全てのボタンを指定してください。 @n
*               buttonList に指定されていないボタンは、自動的に全て離した状態に変更されます。
*/
HIDINPUTER_API ControllerResult SetControllerState(int controllerId, Button buttonList[], int buttonCount, Stick stickList[], float stickDegreeList[], float stickPowerList[], int stickCount);




/**
* @brief        指定されたコントローラーのすべてのボタンを離し、スティックをデフォルト状態に戻します。

* @param[in]    controllerId           対象コントローラーのID

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーのすべてのボタンが全て離された状態になり、スティックを傾きの無い状態にします。
*/
HIDINPUTER_API ControllerResult ResetControllerState(int controllerId);

/*---------------------------------------------------------------------
/ コントローラーアナログスティック操作
/--------------------------------------------------------------------*/
/**
* @brief        指定されたコントローラーのアナログスティックの入力値を変更します。
*
* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    stick                                               対象アナログスティック
* @param[in]    degree                                              入力方向 [degree]
* @param[in]    power                                               入力値(1.0 >= x >= -1.0)

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidStick                    不正なアナログスティックに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの左アナログスティックに値を入力し、スティックを傾けた状態にします。 @n
*               入力方向は、基準値（0）が右方向となり、値が増えるにつれて時計回りに変化します。
*/
HIDINPUTER_API ControllerResult HoldAnalogStick(int controllerId, Stick stick, float degree, float power);

/**
* @brief        指定されたコントローラーのアナログスティックを傾きのない状態にします。
*
* @param[in]    controllerId                                        対象コントローラーの ID
* @param[in]    stick                                               対象アナログスティック

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidStick                    不正なアナログスティックに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの左アナログスティック入力値を 0 にし、傾き状態を解除します。
*/
HIDINPUTER_API ControllerResult ReleaseAnalogStick(int controllerId, Stick stick);

/**
* @brief        指定されたコントローラーの右アナログスティックに値を入力し、その後傾きのない状態に戻します。
*
* @param[in]    controllerId                                        対象コントローラーのID
* @param[in]    stick                                               対象アナログスティック
* @param[in]    degree                                              入力方向 [degree]
* @param[in]    power                                               入力値
* @param[in]    waitHold                                            入力してから、傾きのない状態に戻すまでの待機時間 [ms]

* @return       操作の成否が返ります。
* @retval       ControllerResult_Success                         操作に成功したときの値です。
* @retval       ControllerResult_Unexpected                      不測のエラーが発生したときの値です。
* @retval       ControllerResult_InvalidControllerId             不正なコントローラー ID が指定されていた時の値です。
* @retval       ControllerResult_InvalidStick                    不正なアナログスティックに対して操作が実行された時の値です。
* @retval       ControllerResult_NotInitialized                  未初期化状態で呼び出されたときの値です。

* @pre          controllerId のコントローラーが事前に接続されていなければなりません。

* @details      指定されたコントローラーの右アナログスティックに値を入力し、その後傾きのない状態に戻します。
*               入力方向は、基準値（0）が右方向となり、値が増えるにつれて時計回りに変化します。 @n
                傾きのない状態に戻す前に内部的に、waitHold[ms] 間のスリープを実行しています。
*/
HIDINPUTER_API ControllerResult TiltAnalogStick(int controllerId, Stick stick, float degree, float power, int waitHold);


/*---------------------------------------------------------------------
/ シングルタッチ操作
/--------------------------------------------------------------------*/
/**
* @brief        指定座標をタッチ状態にします。
*
* @param[in]    touchId                                             対象タッチ ID（0～9の固定値）
* @param[in]    x                                                   x 座標
* @param[in]    y                                                   y 座標

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ IDに対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      指定タッチ ID で、指定座標をタッチ状態にすることができます。
*/
HIDINPUTER_API TouchResult TouchPressDown(int touchId, int x, int y);


/**
* @brief        任意のfingerIdを指定して、指定座標をタッチ状態にします。
*
* @param[in]    touchId                                             対象タッチ ID（0～9の固定値）
* @param[in]    x                                                   x 座標
* @param[in]    y                                                   y 座標
* @param[in]    fingerId                                            タッチ箇所に指定するfingerId

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ IDに対して操作が指定された時の値です。
* @retval       TouchResult_InvalidFingerId                      既にタッチされている fingerId が指定されたときの値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      任意のfingerIdを指定して、指定タッチ ID で、指定座標をタッチ状態にすることができます。@n
                fingerId は、アプリケーションが受け取る各タッチの識別子です。この値は 0 から始まり、検出されたタッチごとにインクリメントされた値が付与されます。@n
                全てのタッチが解放されたとき、fingerId は 0 にリセットされます。 @n
                既にタッチされている箇所と同一の fingerId を指定した場合、操作は失敗し TouchResult_InvalidFingerId が返ります。
*/
HIDINPUTER_API TouchResult TouchPressDownWithFingerId(int touchId, int x, int y, int fingerId);

/**
* @brief        指定された ID のタッチを解除します。
*
* @param[in]    touchId                                             対象タッチ ID（0～9の固定値）

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      指定された ID のタッチを解除することが出来ます。
*/
HIDINPUTER_API TouchResult TouchPressUp(int touchId);

/**
* @brief        指定タッチ ID の座標を変更します。

* @param[in]    touchId                                             対象タッチ ID（0～9の固定値）
* @param[in]    x                                                   変更先 x 座標
* @param[in]    y                                                   変更先 y 座標

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      指定タッチ ID の座標を移動することができます。 @n
                このアクションでは、タッチ状態は変更されません。
*/
HIDINPUTER_API TouchResult TouchMove(int touchId, int x, int y);

/**
* @brief        指定座標をタップします。
*
* @param[in]    touchId                                             対象タッチ ID（0～9の固定値）
* @param[in]    x                                                   x 座標
* @param[in]    y                                                   y 座標
* @param[in]    waitPress                                           タップ時間 [ms]

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      指定座標を 1 度タップします。 @n
*               指定座標をタッチした後に、タッチ状態を解除するまでに waitPress[ms] のスリープを実行しています。
*/
HIDINPUTER_API TouchResult Tap(int touchId, int x, int y, int waitPress);

/**
* @brief        方向指定で指定タッチ ID でドラッグ操作を行います。
*
* @param[in]    touchId        対象タッチ ID（0～9の固定値）
* @param[in]    startX         ドラッグ開始 x 座標
* @param[in]    startY         ドラッグ開始 y 座標
* @param[in]    angle          ドラッグ方向 [degree]
* @param[in]    length         ドラッグ距離
* @param[in]    actionDuration 操作時間 [ms]
* @param[in]    waitSpan       操作間隔 [ms]
* @param[in]    releaseAfter   操作後タッチ解除フラグ（操作後にタッチを解除する場合は true, しない場合は false を指定)

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
*               actionDuration > waitSpan でなければなりません。


* @details      方向指定で指定タッチ ID でドラッグ操作を行います。 @n
*               actionDuration[ms] でドラッグ操作を完了させます。 @n
*               また、動作中は waitSpan[ms] 間隔で入力座標の更新が行われます。 @n
*               ドラッグ方向は、基準値（0）が右方向となり、値が増えるについて時計回りの方向となります。 @n
*               実行後は、ドラッグ処理が完了するまで処理が戻りません。
*/
HIDINPUTER_API TouchResult DragAngle(int touchId, int startX, int startY, float angle, float length, int actionDuration, int waitSpan, bool releaseAfter);

/**
* @brief        位置指定で指定タッチ ID でドラッグ操作を行います。
*
* @param[in]    touchId                                             対象タッチ ID（0～9の固定値）
* @param[in]    startX                                              ドラッグ開始 x 座標
* @param[in]    startY                                              ドラッグ開始 y 座標
* @param[in]    endX                                                ドラッグ終了 x 座標
* @param[in]    endY                                                ドラッグ終了 x 座標
* @param[in]    actionDuration                                      操作時間 [ms]
* @param[in]    waitSpan                                            操作間隔 [ms]
* @param[in]    releaseAfter                                        操作後タッチ解除フラグ（操作後にタッチを解除する場合は true, しない場合は false を指定)

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
*               actionDuration > waitSpan でなければなりません。


* @details      位置指定で指定タッチ ID でドラッグ操作を行います。 @n
*               actionDuration[ms] でドラッグ操作を完了させます。@n
*               また、動作中は waitSpan[ms] 間隔で入力座標の更新が行われます。 @n
*               実行後は、ドラッグ処理が完了するまで処理が戻りません。
*/
HIDINPUTER_API TouchResult DragPos(int touchId, int startX, int startY, int endX, int endY, int actionDuration, int waitSpan, bool releaseAfter);


/**
* @brief        タッチスクリーンの入力状態を変更します。
*
* @param[in]    touchIdList                                         対象タッチ ID（0～9の固定値）の配列
* @param[in]    fingerIdList                                        各タッチに割り当てる fingerId の配列
* @param[in]    posXList                                            各タッチの x 座標
* @param[in]    posYList                                            各タッチの y 座標
* @param[in]    touchCount                                          touchIdList のインデックス数

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_InvalidFingerId                      指定された fingerId が重複していた時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      タッチスクリーンの入力状態を変更します。@n
                touchIdList で指定されていない touchId は、自動的にタッチ状態が解除されます。
*/
HIDINPUTER_API TouchResult SetTouchState(int touchIdList[], int fingerIdList[], float posXList[], float posYList[], int touchCount);


/**
* @brief        タッチスクリーンへの全ての入力を解除します。

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      タッチスクリーンの全ての入力状態が解除されます。
*/
HIDINPUTER_API TouchResult ResetTouchState();

/*---------------------------------------------------------------------
/ マルチタッチ操作
/--------------------------------------------------------------------*/
/**
* @brief        複数個所を同時にタップします。
*
* @param[in]    touchIdList    対象タッチ ID（0～9の固定値）の配列
* @param[in]    touchCount     touchIdList のインデックス数
* @param[in]    touchPosXList  タッチの x 座標配列
* @param[in]    touchPosYList  タッチの y 座標配列
* @param[in]    waitSpan       タップ時間 [ms]

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。

* @details      指定座標に対して、同時にタップ処理を実行します。
*               指定座標をタッチした後に、タッチ状態を解除するまでに waitPress[ms] のスリープを実行しています。
*/
HIDINPUTER_API TouchResult MultiTap(int touchIdList[], int touchCount, float touchPosXList[], float touchPosYList[],
    int waitSpan);

/**
* @brief        複数個所を同時にタッチし、中心点座標を基に座標移動、拡大、回転動作を行います。
*
* @param[in]    touchIdList             対象タッチ ID（0～9の固定値）の配列
* @param[in]    touchCount              touchIdList のインデックス数
* @param[in]    startCenterX            中心点の開始x座標
* @param[in]    startCenterY            中心点の開始y座標
* @param[in]    centerMoveDeg           中心点の移動方向 [degree]
* @param[in]    centerMoveLength        中心点の移動距離
* @param[in]    touchRelativePosXList   中心点を基準とした、各タッチ ID の相対x座標
* @param[in]    touchRelativePosYList   中心点を基準とした、各タッチ ID の相対y座標
* @param[in]    startDeg                開始時の回転角度 [degree]
* @param[in]    rotateDeg               動作全体を通しての回転量 [degree]
* @param[in]    startScale              開始時のスケール
* @param[in]    increaseScale           動作全体を通してのスケール増加量
* @param[in]    actionDuration          動作を行う時間長 [ms]
* @param[in]    waitSpan                間隔 [ms]
* @param[in]    waitTouchBegin          全ての指をタッチしてから、動作を開始するまでの時間長 [ms]
* @param[in]    waitTouchEnd            動作が完了してから、タッチを解放するまでの時間長 [ms]
* @param[in]    releaseAfter            操作後タッチ解除フラグ（操作後にタッチを解除する場合は true, しない場合は false を指定)

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
*               actionDuration > waitSpan でなければなりません。


* @details      touchIdList のタッチを使用して、複数箇所を同時にタッチし、中心点を基準に回転、拡大、平行移動のタッチを動作を実行することができます。 @n
*               各タッチの初期座標は、中心点からの相対座標を表す touchRelativePosXList, touchRelativePosYList に startScale を掛けた値を、中心点から startDeg 度回転させた値です。 @n
*               動作中は、actionDuration[ms] の間に中心点を centerMoveDeg, centerMoveLength を基に移動、各タッチの相対座標を increaseScale 分増加、中心点を基準に rotateDeg 度回転させます。 @n
*               また、動作中は waitSpan[ms] 間隔で入力座標の更新が行われます。 @n
*               タッチ後に動作開始まで、もしくは動作終了後タッチ解除までに時間間隔を置きたい場合は、waitTouchBegin, waitTouchEnd に値を指定してください。 @n
*               回転角度は、基準値（0）が右方向となり、値が増えるについて時計回りの方向へ回転します。@n
*               実行後は、動作が完了するまで処理が戻りません。
*/
HIDINPUTER_API TouchResult MultiTouchAnimator(int touchIdList[], int touchCount,
    int startCenterX, int startCenterY,
    float centerMoveDeg, float centerMoveLength,
    float touchRelativePosXList[], float touchRelativePosYList[],
    float startDeg, float rotateDeg,
    float startScale, float increaseScale,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter);

/**
* @brief        複数同時にタッチし位置指定でドラッグ操作を行います。
*
* @param[in]    touchIdList            対象タッチ ID （0～9の固定値）の配列
* @param[in]    touchCount             touchIdList のインデックス数
* @param[in]    startCenterX           中心点の開始x座標
* @param[in]    startCenterY           中心点の開始y座標
* @param[in]    endCenterX             中心点の終了x座標
* @param[in]    endCenterY             中心点の終了y座標
* @param[in]    touchRelativePosXList  中心点を基準とした、各タッチ ID の相対x座標
* @param[in]    touchRelativePosYList  中心点を基準とした、各タッチ ID の相対y座標
* @param[in]    actionDuration         動作を行う時間長 [ms]
* @param[in]    waitSpan               間隔 [ms]
* @param[in]    waitTouchBegin         全ての指をタッチしてから、動作を開始するまでの時間長 [ms]
* @param[in]    waitTouchEnd           動作が完了してから、タッチを解放するまでの時間長 [ms]
* @param[in]    releaseAfter           操作後タッチ解除フラグ（操作後にタッチを解除する場合は true, しない場合は false を指定)

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
*               actionDuration > waitSpan でなければなりません。


* @details      touchIdList のタッチを使用して、複数箇所を同時にタッチし、指定位置へドラッグを行います。 @n
*               各タッチの初期座標は、中心点からの相対座標を表す touchRelativePosXList, touchRelativePosYList です。@n
*               動作中は、actionDuration[ms] の間に中心点を endCenterX, endCenterY へ移動させます。 @n
*               また、動作中は waitSpan[ms] 間隔で入力座標の更新が行われます。 @n
*               タッチ後に動作開始まで、もしくは動作終了後タッチ解除までに時間間隔を置きたい場合は、waitTouchBegin, waitTouchEnd に値を指定してください。 @n
*               実行後は、動作が完了するまで処理が戻りません。
*/
HIDINPUTER_API TouchResult MultiDragPos(int touchIdList[], int touchCount,
    int startCenterX, int startCenterY,
    int endCenterX, int endCenterY,
    float touchRelativePosXList[], float touchRelativePosYList[],
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter);


/**
* @brief        複数個所を同時にタッチし方向指定でドラッグ操作を行います。
*
* @param[in]    touchIdList                                         対象タッチ ID（0～9の固定値）の配列
* @param[in]    touchCount                                          touchIdList のインデックス数
* @param[in]    startCenterX                                        中心点の開始x座標
* @param[in]    startCenterY                                        中心点の開始y座標
* @param[in]    centerMoveDeg                                       中心点の移動方向 [degree]
* @param[in]    centerMoveLength                                    中心点の移動距離
* @param[in]    touchRelativePosXList                               中心点を基準とした、各タッチ ID の相対x座標
* @param[in]    touchRelativePosYList                               中心点を基準とした、各タッチ ID の相対y座標
* @param[in]    actionDuration                                      動作を行う時間長 [ms]
* @param[in]    waitSpan                                            間隔 [ms]
* @param[in]    waitTouchBegin                                      全ての指をタッチしてから、動作を開始するまでの時間長 [ms]
* @param[in]    waitTouchEnd                                        動作が完了してから、タッチを解放するまでの時間長 [ms]
* @param[in]    releaseAfter                                        操作後タッチ解除フラグ（操作後にタッチを解除する場合は true, しない場合は false を指定)

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
*               actionDuration > waitSpan でなければなりません。


* @details      touchIdList のタッチを使用して、複数箇所を同時にタッチし、指定方向へドラッグを行います。 @n
*               各タッチの初期座標は、中心点からの相対座標を表す touchRelativePosXList, touchRelativePosYList です。@n
*               動作中は、actionDuration[ms] の間に中心点を centerMoveDeg, centerMoveLength 方向へ移動させます。
*               また、動作中は waitSpan[ms] 間隔で入力座標の更新が行われます。 @n
*               タッチ後に動作開始まで、もしくは動作終了後タッチ解除までに時間間隔を置きたい場合は、waitTouchBegin, waitTouchEnd に値を指定してください。 @n
*               回転角度は、基準値（0）が右方向となり、値が増えるについて時計回りの方向へ回転します。 @n
*               実行後は、動作が完了するまで処理が戻りません。
*/
HIDINPUTER_API TouchResult  MultiDragAngle(int touchIdList[], int touchCount,
    int startCenterX, int startCenterY,
    float centerMoveDeg, float centerMoveLength,
    float touchRelativePosXList[], float touchRelativePosYList[],
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter);

/**
* @brief        複数指でピンチ操作を行います。
*
* @param[in]    touchIdList                                         対象タッチ ID （0～9の固定値）の配列
* @param[in]    touchCount                                          touchIdList のインデックス数
* @param[in]    centerX                                             中心点の x 座標
* @param[in]    centerY                                             中心点の y 座標
* @param[in]    touchDegList                                        各タッチの配置方向 [degree]
* @param[in]    startRadius                                         ピンチ操作開始時の半径（中心点と各タッチ箇所との距離）
* @param[in]    increaseRadius                                      動作全体を通して変化させる半径
* @param[in]    actionDuration                                      動作を行う時間長 [ms]
* @param[in]    waitSpan                                            間隔 [ms]
* @param[in]    waitTouchBegin                                      全ての指をタッチしてから、動作を開始するまでの時間長 [ms]
* @param[in]    waitTouchEnd                                        動作が完了してから、タッチを解放するまでの時間長 [ms]
* @param[in]    releaseAfter                                        操作後タッチ解除フラグ（操作後にタッチを解除する場合は true, しない場合は false を指定)

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
*               actionDuration > waitSpan でなければなりません。


* @details      touchIdList のタッチを使用して、複数箇所を同時にタッチし、ピンチ操作を行います。 @n
*               各タッチの初期座標は、中心点からの相対座標として、 touchDegList 度の方向へ startScale 移動した箇所です。@n
*               動作中は、actionDuration[ms] の間に各タッチの相対座標を、increaseScale 増加させます。 @n
*               また、動作中は waitSpan[ms] 間隔で入力座標の更新が行われます。 @n
*               タッチ後に動作開始まで、もしくは動作終了後タッチ解除までに時間間隔を置きたい場合は、waitTouchBegin, waitTouchEnd に値を指定してください。 @n
*               回転角度は、基準値（0）が右方向となり、値が増えるについて時計回りの方向へ回転します。@n
*               実行後は、動作が完了するまで処理が戻りません。
*/
HIDINPUTER_API TouchResult Pinch(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[],
    float startRadius, float increaseRadius,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter);

/**
* @brief        中心点をタッチした状態でピンチ操作を行います。
*
* @param[in]    touchIdList                                         対象タッチ ID（0～9の固定値）の配列
* @param[in]    touchCount                                          touchIdList のインデックス数
* @param[in]    centerX                                             中心点の x 座標
* @param[in]    centerY                                             中心点の y 座標
* @param[in]    touchDegList                                        各タッチの配置方向 [degree]
* @param[in]    startRadius                                         ピンチ操作開始時の半径（中心点と各タッチ箇所との距離）
* @param[in]    increaseRadius                                      動作全体を通して変化させる半径
* @param[in]    actionDuration                                      動作を行う時間長 [ms]
* @param[in]    waitSpan                                            間隔 [ms]
* @param[in]    waitTouchBegin                                      全ての指をタッチしてから、動作を開始するまでの時間長 [ms]
* @param[in]    waitTouchEnd                                        動作が完了してから、タッチを解放するまでの時間長 [ms]
* @param[in]    releaseAfter                                        操作後タッチ解除フラグ（操作後にタッチを解除する場合は true, しない場合は false を指定)

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
*               actionDuration > waitSpan でなければなりません。


* @details      touchIdList のタッチを使用して、中心点とその周囲複数箇所を同時にタッチし、ピンチ操作を行います。 @n
*               各タッチの初期座標は、中心点と、中心点からの相対座標として、 touchDegList 度の方向へ startScale 移動した箇所です。@n
*               動作中は、actionDuration[ms] の間に各タッチの相対座標を、increaseScale 増加させます。 @n
*               また、動作中は waitSpan[ms] 間隔で入力座標の更新が行われます。 @n
*               タッチ後に動作開始まで、もしくは動作終了後タッチ解除までに時間間隔を置きたい場合は、waitTouchBegin, waitTouchEnd に値を指定してください。 @n
*               回転角度は、基準値（0）が右方向となり、値が増えるについて時計回りの方向へ回転します。@n
*               touchIdList に指定した配列の先頭の touchId は、操作開始から完了まで中心点をタッチし続けます。 @n
*               touchDegList へは、中心点を除いたデータ数(touchCount - 1)の配列を指定してください。 @n
*               実行後は、動作が完了するまで処理が戻りません。
*/
HIDINPUTER_API TouchResult CenterTouchPinch(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[],
    float startRadius, float increaseRadius,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter);


/**
* @brief        中心点を基に複数タッチを回転移動させます。
*
* @param[in]    touchIdList                                         対象タッチ ID（0～9の固定値）の配列
* @param[in]    touchCount                                          touchIdList のインデックス数
* @param[in]    centerX                                             中心点の x 座標
* @param[in]    centerY                                             中心点の y 座標
* @param[in]    touchDegList                                        各タッチの配置方向 [degree]
* @param[in]    radius                                              各タッチの中心点からの距離
* @param[in]    startDeg                                            開始時の回転角度 [degree]
* @param[in]    rotateDeg                                           動作全体を通しての回転量 [degree]
* @param[in]    actionDuration                                      動作を行う時間長 [ms]
* @param[in]    waitSpan                                            間隔 [ms]
* @param[in]    waitTouchBegin                                      全ての指をタッチしてから、動作を開始するまでの時間長 [ms]
* @param[in]    waitTouchEnd                                        動作が完了してから、タッチを解放するまでの時間長 [ms]
* @param[in]    releaseAfter                                        操作後タッチ解除フラグ（操作後にタッチを解除する場合は true, しない場合は false を指定)

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
*               actionDuration > waitSpan でなければなりません。


* @details      touchIdList のタッチを使用して、中心点の周囲複数箇所を同時にタッチし、中心点を基準とした回転操作を行います。 @n
*               各タッチの初期座標は、中心点を基準に、 (touchDegList + startDeg) 度の方向へ radius 分移動した箇所です。 @n
*               操作は、actionDuration[ms] の間に実行され、最終的に各タッチの回転角度を rotateDeg 増加させます。 @n
*               また、動作中は waitSpan[ms] 間隔で入力座標の更新が行われます。 @n
*               タッチ後に動作開始まで、もしくは動作終了後タッチ解除までに時間間隔を置きたい場合は、waitTouchBegin, waitTouchEnd に値を指定してください。 @n
*               回転角度は、基準値（0）が右方向となり、値が増えるについて時計回りの方向へ回転します。@n
*               実行後は、動作が完了するまで処理が戻りません。
*/
HIDINPUTER_API TouchResult MultiRotate(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[], float radius,
    float startDeg, float rotateDeg,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter);

/**
* @brief        中心点をタッチした状態で、その座標を基に複数タッチを回転移動させます。
*
* @param[in]    touchIdList                                         対象タッチ ID（0～9の固定値）の配列
* @param[in]    touchCount                                          touchIdList のインデックス数
* @param[in]    centerX                                             中心点の x 座標
* @param[in]    centerY                                             中心点の y 座標
* @param[in]    touchDegList                                        各タッチの配置方向 [degree]
* @param[in]    radius                                              各タッチの中心点からの距離
* @param[in]    startDeg                                            開始時の回転角度 [degree]
* @param[in]    rotateDeg                                           動作全体を通しての回転量 [degree]
* @param[in]    actionDuration                                      動作を行う時間長 [ms]
* @param[in]    waitSpan                                            間隔 [ms]
* @param[in]    waitTouchBegin                                      全ての指をタッチしてから、動作を開始するまでの時間長 [ms]
* @param[in]    waitTouchEnd                                        動作が完了してから、タッチを解放するまでの時間長 [ms]
* @param[in]    releaseAfter                                        操作後タッチ解除フラグ（操作後にタッチを解除する場合は true, しない場合は false を指定)

* @return       操作の成否が返ります。
* @retval       TouchResult_Success                              操作に成功したときの値です。
* @retval       TouchResult_Unexpected                           不測のエラーが発生したときの値です。
* @retval       TouchResult_InvalidTouchId                       不正なタッチ ID に対して操作が指定された時の値です。
* @retval       TouchResult_NotInitialized                       未初期化状態で呼び出されたときの値です。

* @pre          Initialize() が呼び出されていなければなりません。 @n
*               actionDuration > waitSpan でなければなりません。

* @details      touchIdList のタッチを使用して、中心点とその周囲複数箇所を同時にタッチし、中心点を基準とした回転操作を行います。 @n
*               各タッチの初期座標は、中心点と、中心点を基準に (touchDegList + startDeg) 度の方向へ radius 分移動した箇所です。 @n
*               操作は、actionDuration[ms] の間に実行され、最終的に各タッチの回転角度を rotateDeg 増加させます。 @n
*               また、動作中は waitSpan[ms] 間隔で入力座標の更新が行われます。 @n
*               タッチ後に動作開始まで、もしくは動作終了後タッチ解除までに時間間隔を置きたい場合は、waitTouchBegin, waitTouchEnd に値を指定してください。 @n
*               回転角度は、基準値（0）が右方向となり、値が増えるについて時計回りの方向へ回転します。@n
*               touchIdList に指定した配列の先頭の touchId は、操作開始から完了まで中心点をタッチし続けます。 @n
*               touchDegList へは、中心点を除いたデータ数(touchCount - 1)の配列を指定してください。 @n
*               実行後は、動作が完了するまで処理が戻りません。
*/
HIDINPUTER_API TouchResult CenterTouchMultiRotate(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[], float radius,
    float startDeg, float rotateDeg,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter);



