﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <list>
#include <map>
#include <cerrno>

#include "CppHeaderWriter.h"
#include "BdfWriter.h"
#include "BuiltinDataDescrParser.h"
#include "util.h"


using namespace std;

const char                      *AppConstants::g_NintendoSdkEnvVarName = "NINTENDO_SDK_ROOT";
const char                      *AppConstants::g_DataIdHeaderName = "ssl_DataId.h";
const char                      *AppConstants::g_SslSrcBuiltin = "Programs/Eris/Sources/Resources/Ssl/Builtin";
const char                      *AppConstants::g_SslIncludeSubdir = "Programs/Eris/Include/nn/ssl";
const char                      *AppConstants::g_pErrInvalidArgs = "Invalid or missing arguments";
const char                      *AppConstants::g_DataEnumName = "BuiltinDataId";

void PrintUsage(int argc, char *argv[], const char *msg)
{
    if (msg != nullptr)
    {
        printf("ERROR: %s\n", msg);
    }

    printf("Usage: %s -n data_name -i descriptor_file -o builtin_data_file [-h header_file] [-d header_path]\n", argv[0]);
    printf("\t-i descriptor_file: The input descriptor XML file\n");
    printf("\t-o builtin_data_file: The output Built-in Data File for inclusion in the platform\n");
    printf("\t-n enum_name: The name of the built-in data enum to be created.  Defaults to %s.\n", AppConstants::g_DataEnumName);
    printf("\t-h header_file: The name of the header file to generate. Defaults to %s.\n", AppConstants::g_DataIdHeaderName);
    printf("\t-d header_path: The absolute path where to output the certificate IDs header.\n\t                Defaults to %s/%s.\n",
           AppConstants::g_NintendoSdkEnvVarName,
           AppConstants::g_SslIncludeSubdir);

    printf("Got arguments:\n");
    for (int i = 1; i < argc; i++)
    {
        printf("\t%d: %s\n", i, argv[i]);
    }
}

enum ArgParmType
{
    ArgParmType_ParmNone,
    ArgParmType_ParmString,
    ArgParmType_ParmInteger,
    ArgParmType_ParmLongInteger,
    ArgParmType_ParmFloat,
};


typedef struct _ArgDetails
{
    const char                  *pArgStr;
    bool                        isRequired;
    bool                        hasParm;
    bool                        isPresent;
    ArgParmType                 parmType;

    union
    {
        char                    *pParmStr;
        int                     parmInt;
        long int                parmLongInt;
        float                   parmFloat;
    } parm;
} ArgDetails;


#define NUM_ELEMENTS(pArray, type)      (sizeof(pArray) / sizeof(type))

enum AppArgIndex
{
    AppArgIndex_InputFile  = 0,
    AppArgIndex_OutputFile = 1,
    AppArgIndex_HeaderFile = 2,
    AppArgIndex_HeaderPath = 3,
    AppArgIndex_EnumName   = 4
};


static ArgDetails               g_CmdArgs[] =
{
    { "-i", true,  true, false, ArgParmType_ParmString },
    { "-o", true,  true, false, ArgParmType_ParmString },
    { "-h", false, true, false, ArgParmType_ParmString },
    { "-d", false, true, false, ArgParmType_ParmString },
    { "-n", false, true, false, ArgParmType_ParmString },
};


int ProcessArgs(int         argc,
                char        *argv[],
                ArgDetails  *pAppArgs,
                int         numAppArgs)
{
    int                         ret = 0;
    int                         i;
    int                         j;

    for (j = 0; j < numAppArgs; j++)
    {
        for (i = 1; i < argc; i++)
        {
            if (strcmp(argv[i], pAppArgs[j].pArgStr) == 0)
            {
                bool            isPresent = true;

                LOGD("ProcessArgs: found option \'%s\'\n", pAppArgs[j].pArgStr);

                if (pAppArgs[j].hasParm)
                {
                    ArgParmType curType = pAppArgs[j].parmType;
                    i++;

                    if (curType == ArgParmType_ParmString)
                    {
                        pAppArgs[j].parm.pParmStr = argv[i];
                        LOGD("ProcessArgs: found parameter \'%s\'\n",
                             pAppArgs[j].parm.pParmStr);
                    }
                    else if (curType == ArgParmType_ParmInteger)
                    {
                        pAppArgs[j].parm.parmInt =
                            static_cast<int>(strtol(argv[i], nullptr, 0));
                        LOGD("ProcessArgs: found parameter \'%d\'\n",
                            pAppArgs[j].parm.parmInt);
                    }
                    else if (curType == ArgParmType_ParmLongInteger)
                    {
                        pAppArgs[j].parm.parmLongInt =
                            strtol(argv[i], nullptr, 0);
                        LOGD("ProcessArgs: found parameter \'%ld\'\n",
                            pAppArgs[j].parm.parmLongInt);
                    }
                    else if (curType == ArgParmType_ParmFloat)
                    {
                        pAppArgs[j].parm.parmFloat = strtof(argv[i], nullptr);
                        LOGD("ProcessArgs: found parameter \'%f\'\n",
                            pAppArgs[j].parm.parmFloat);
                    }
                    else
                    {
                        LOGE("ParseArgs: arg \'%s\' does not take param type %d\n",
                             pAppArgs[j].pArgStr,
                             pAppArgs[j].parmType);
                        isPresent = false;
                    }
                }

                pAppArgs[j].isPresent = isPresent;
            }
        }

        if (pAppArgs[j].isRequired && !pAppArgs[j].isPresent)
        {
            LOGE("ParseArgs: arg \'%s\' is required\n", pAppArgs[j].pArgStr);
            ret = -1;
            goto errExit;
        }
    }

errExit:
    return ret;
}


int main(int argc, char *argv[])
{
    char                                *pDescrFileName = nullptr;
    char                                *pOutputFileName = nullptr;
    char                                *pEnumName = nullptr;
    FILE                                *pDescrFile = nullptr;
    FILE                                *pOutputFile = nullptr;
    FILE                                *pHeaderFile = nullptr;
    int                                 ret = -1;
    BuiltinDataDescrParser              *pParser = nullptr;
    map<uint32_t, BuiltinDataInfo *>    *pTrustMap = nullptr;
    char                                *pTmp = nullptr;
    char                                *pNintendoSdkRoot = nullptr;
    char                                *pHeaderOutName = nullptr;
    char                                hdrPathScratch[AppConstants::g_MaxPathLen + 1];
    int                                 status;

    //  Process input args, bail if we get a failure  */
    if (ProcessArgs(argc,
                    argv,
                    g_CmdArgs,
                    NUM_ELEMENTS(g_CmdArgs, ArgDetails)) < 0)
    {
        PrintUsage(argc, argv, AppConstants::g_pErrInvalidArgs);
        goto errExit;
    }

    //  Get the NINTENDO_SDK_ROOT env variable, it is required.
    pNintendoSdkRoot = getenv(AppConstants::g_NintendoSdkEnvVarName);
    if (pNintendoSdkRoot == nullptr)
    {
        LOGE("SDK env. var \'%s\' not defined\n",
             AppConstants::g_NintendoSdkEnvVarName);
        goto errExit;
    }

    if (g_CmdArgs[AppArgIndex_HeaderPath].isPresent)
    {
        pTmp = g_CmdArgs[AppArgIndex_HeaderPath].parm.pParmStr;
    }
    else
    {
        memset(hdrPathScratch, 0, sizeof(hdrPathScratch));
        nn_snprintf(hdrPathScratch,
                    sizeof(hdrPathScratch),
                    "%s%s%s",
                    pNintendoSdkRoot,
                    FILE_PATH_DELIM,
                    AppConstants::g_SslIncludeSubdir);
        pTmp = hdrPathScratch;
    }

    if (!g_CmdArgs[AppArgIndex_HeaderFile].isPresent)
    {
        g_CmdArgs[AppArgIndex_HeaderFile].parm.pParmStr =
            const_cast<char *>(AppConstants::g_DataIdHeaderName);
    }

    if (!g_CmdArgs[AppArgIndex_EnumName].isPresent)
    {
        g_CmdArgs[AppArgIndex_EnumName].parm.pParmStr =
            const_cast<char *>(AppConstants::g_DataEnumName);
    }

    pDescrFileName  = g_CmdArgs[AppArgIndex_InputFile].parm.pParmStr;
    pOutputFileName = g_CmdArgs[AppArgIndex_OutputFile].parm.pParmStr;
    pEnumName       = g_CmdArgs[AppArgIndex_EnumName].parm.pParmStr;

    pHeaderOutName =
        new char[strlen(pTmp) +
                 1 +
                 strlen(g_CmdArgs[AppArgIndex_HeaderFile].parm.pParmStr) +
                 1];
    if (pHeaderOutName == nullptr)
    {
        LOGE("Unable to alloc buf for header file name\n");
        goto errExit;
    }

    status = sprintf(pHeaderOutName,
                     "%s%s%s",
                     pTmp,
                     FILE_PATH_DELIM,
                     g_CmdArgs[AppArgIndex_HeaderFile].parm.pParmStr);
    pHeaderOutName[status] = '\0';

    //  Open the descriptor input file and the output file, make sure
    //  we can do something with them before proceeding.
    LOGI("opening descr file \'%s\'\n", pDescrFileName);
    pDescrFile = fopen(pDescrFileName, "rb");
    if (pDescrFile == nullptr)
    {
        LOGE("unable to open descr file, \'%s\': %d\n",
             pDescrFileName,
             errno);
        goto errExit;
    }

    LOGI("opening output file \'%s\'\n", pOutputFileName);
    pOutputFile = fopen(pOutputFileName, "wb+");
    if (pOutputFile == nullptr)
    {
        LOGE("unable to open output file, \'%s\': %d\n",
             pOutputFileName,
             errno);
        goto errExit;
    }

    LOGI("opening header file \'%s\'\n", pHeaderOutName);
    pHeaderFile = fopen(pHeaderOutName, "wb+");
    if (pHeaderFile == nullptr)
    {
        LOGE("unable to output header file, \'%s\': %d\n",
             pHeaderOutName,
             errno);
        goto errExit;
    }

    //  Create the XML parser, build the trusted cert list
    LOGI("parsing intput descr...\n");
    pParser = new BuiltinDataDescrParser(pNintendoSdkRoot, pDescrFile);
    if (pParser == nullptr)
    {
        LOGE("failed to create parser\n");
        goto errExit;
    }

    if (pParser->Initialize() < 0)
    {
        LOGE("failed to init parser\n");
        goto errExit;
    }

    pTrustMap = pParser->BuildBuiltinDataMap();
    if (pTrustMap == nullptr)
    {
        LOGE("failed to create trust map from descr file\n");
        goto errExit;
    }

    //  Write the header
    status = CppHeaderWriter::DoWrite(pHeaderFile, pEnumName, pTrustMap);
    if (status != 0)
    {
        LOGE("failed to create cert ID header\n");
        goto errExit;
    }

    //  Write the BDF.
    status = BdfWriter::DoWrite(pOutputFile, pTrustMap);
    if (status != 0)
    {
        LOGE("failed to write BDF\n");
        goto errExit;
    }

    //  Update success to 0, we are done!
    LOGI("done\n");
    ret = 0;

errExit:
    if (pHeaderOutName != nullptr)
    {
        delete[] pHeaderOutName;
        pHeaderOutName = nullptr;
    }

    if (pParser != nullptr)
    {
        if (pTrustMap != nullptr)
        {
            pParser->DestroyBuiltinDataMap(pTrustMap);
            pTrustMap = nullptr;
        }

        pParser->Finalize();
        delete pParser;
        pParser = nullptr;
    }

    if (pHeaderFile != nullptr)
    {
        fclose(pHeaderFile);
        pHeaderFile = nullptr;
    }

    if (pOutputFile != nullptr)
    {
        fclose(pOutputFile);
        pOutputFile = nullptr;
    }

    if (pDescrFile != nullptr)
    {
        fclose(pDescrFile);
        pDescrFile = nullptr;
    }

    return ret;
}    //  NOLINT(impl/function_size)
