﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "BdfWriter.h"
#include "util.h"


int BdfWriter::DoWrite(FILE *pOutFile, map<uint32_t, BuiltinDataInfo *> *pBdiMap)
{
    int                                         ret = -1;
    uint8_t                                     *pMetaBuf = nullptr;
    uint32_t                                    dataCount;
    uint32_t                                    curOffset;
    BdfMetaHdr                                  *pMetaHdr;
    BdfMetaDataEntry                            *pMetaDataEntries;
    uint32_t                                    i;
    size_t                                      metaBufSize;
    size_t                                      bytesWritten;
    uint8_t                                     *pData = nullptr;
    map<uint32_t, BuiltinDataInfo *>::iterator  it;

    //  Determine how many data blobs we have to deal with (including ones which
    //  have been remvoed or revoked) and alloc the meta data buffer.
    dataCount = static_cast<uint32_t>(pBdiMap->size());
    LOGD("BdfWriter::DoWrite: creating a file with %u data blobs in meta\n", dataCount);

    metaBufSize = sizeof(BdfMetaHdr) + (dataCount * sizeof(BdfMetaDataEntry));
    pMetaBuf = new uint8_t[metaBufSize];
    if (pMetaBuf == nullptr)
    {
        LOGE("BdfWriter::DoWrite: unable to alloc scratch buffer\n");
        goto errExit;
    }

    memset(pMetaBuf, 0, metaBufSize);
    pMetaHdr = reinterpret_cast<BdfMetaHdr *>(pMetaBuf);
    pMetaDataEntries = reinterpret_cast<BdfMetaDataEntry *>(pMetaHdr + 1);

    //  Setup the header with the magic number and count
    pMetaHdr->magicNum = BuiltinDataInfo::g_BdfMagicNum;
    pMetaHdr->dataCount = dataCount;

    //  The offset in the entries is taken from the end of the meta header.
    //  So to begin with, our offset is past the data entries.
    curOffset = dataCount * sizeof(*pMetaDataEntries);
    LOGD("BdfWriter::DoWrite: starting offset for data is: %u\n", curOffset);

    //  Walk through each entry in the map, build up the meta data
    for (i = 0, it = pBdiMap->begin(); it != pBdiMap->end(); ++it, i++)
    {
        BuiltinDataInfo                     *pCurInfo = it->second;
        BuiltinDataInfo::BuiltinDataStatus  curStatus;
        uint32_t                            curSize;

        //  Copy over the cert ID
        pMetaDataEntries[i].dataId = pCurInfo->GetId();

        //  Only trusted certs get actual data in their entries
        curSize = pCurInfo->GetDataSize();
        curStatus = pCurInfo->GetStatus();

        pMetaDataEntries[i].dataStatus = curStatus;

        //  The size and offset get adjusted based on status.  We do not
        //  include revoked or removed certs in the TCF.
        if ((curStatus == BuiltinDataInfo::BuiltinDataStatus_Enabled) ||
            (curStatus == BuiltinDataInfo::BuiltinDataStatus_Disabled))
        {
            pMetaDataEntries[i].dataOffset = curOffset;
        }
        else
        {
            curSize = 0;
            pMetaDataEntries[i].dataOffset = 0;
        }

        pMetaDataEntries[i].dataSize   = curSize;

        LOGD("BdfWriter::DoWrite: data \'%s\' info:\n\tID:     %8.8X\n\tStatus: %X\n\tSize:   %u\n\tOffset: %u\n",
             pCurInfo->GetName(),
             pMetaDataEntries[i].dataId,
             pMetaDataEntries[i].dataStatus,
             pMetaDataEntries[i].dataSize,
             pMetaDataEntries[i].dataOffset);

        //  Shift the offset by the size, but round up to the nearest
        //  uint32_t.
        curSize = NN_DETAIL_SSL_CALC_ALIGN(curSize, sizeof(uint32_t));
        curOffset += curSize;
    }

    //  Write out the header and meta data
    LOGD("BdfWriter::DoWrite: writing out header and meta data (%u bytes)\n",
         sizeof(*pMetaHdr) + (dataCount * sizeof(*pMetaDataEntries)));
    bytesWritten = fwrite(pMetaHdr, 1, sizeof(*pMetaHdr), pOutFile);
    if (bytesWritten != sizeof(*pMetaHdr))
    {
        ret = ferror(pOutFile);
        LOGE("BdfWriter::DoWrite: failed to write BDF header: %d\n", ret);
        goto errExit;
    }

    bytesWritten = fwrite(pMetaDataEntries,
                          1,
                          dataCount * sizeof(*pMetaDataEntries),
                          pOutFile);
    if (bytesWritten != (dataCount * sizeof(*pMetaDataEntries)))
    {
        ret = ferror(pOutFile);
        LOGE("BdfWriter::DoWrite: failed to write BDF meta data: %d\n", ret);
        goto errExit;
    }

    //  Now walk the map and write each cert out to the file
    for (i = 0, it = pBdiMap->begin(); it != pBdiMap->end(); ++it, i++)
    {
        BuiltinDataInfo                     *pCurInfo = it->second;
        BuiltinDataInfo::BuiltinDataStatus  curStatus;
        uint32_t                            curSize;
        uint32_t                            withPadSize;
        uint32_t                            tmp;
        uint32_t                            zero = 0;
        int                                 status;

        //  Only trusted certs get actual data in their entries
        curStatus = pCurInfo->GetStatus();
        if ((curStatus != BuiltinDataInfo::BuiltinDataStatus_Enabled) &&
            (curStatus != BuiltinDataInfo::BuiltinDataStatus_Disabled))
        {
            continue;
        }

        curSize = pCurInfo->GetDataSize();
        if (curSize == 0)
        {
            LOGW("BdfWriter::DoWrite: enabled data with no data! id 0x%8.8X\n",
                 pCurInfo->GetId());
            continue;
        }

        pData = new uint8_t[curSize];
        if (pData == nullptr)
        {
            LOGE("BdfWriter::DoWrite: failed to alloc data buf %d bytes for data id 0x%8.8X\n",
                 curSize,
                 pCurInfo->GetId());
            goto errExit;
        }

        status = pCurInfo->GetData(pData, curSize, &tmp);
        if (status != 0)
        {
            LOGE("BdfWriter::DoWrite: failed to get data for data \'%s\'\n",
                 pCurInfo->GetName());
            delete[] pData;
            pData = nullptr;
            goto errExit;
        }

        bytesWritten = fwrite(pData, 1, curSize, pOutFile);
        if (bytesWritten != curSize)
        {
            ret = ferror(pOutFile);
            LOGE("BdfWriter::DoWrite: failed to write data for \'%s\': %d\n",
                 pCurInfo->GetName(),
                 ret);
            delete[] pData;
            pData = nullptr;
            goto errExit;
        }

        delete[] pData;
        pData = nullptr;

        withPadSize = NN_DETAIL_SSL_CALC_ALIGN(curSize, sizeof(uint32_t));
        if (withPadSize != curSize)
        {
            withPadSize = withPadSize - curSize;
            LOGI("BdfWriter::DoWrite: data \'%s\' gets %u pad bytes on disk\n",
                 pCurInfo->GetName(),
                 withPadSize);
            bytesWritten = fwrite(&zero, 1, withPadSize, pOutFile);
            if (bytesWritten != withPadSize)
            {
                ret = ferror(pOutFile);
                LOGE("BdfWriter::DoWrite: failed to write %u pad bytes: %d\n",
                     withPadSize,
                     ret);
                goto errExit;
            }
        }
    }

    ret = 0;

errExit:
    //  Release the cert data buf if it is non-null
    if (pData != nullptr)
    {
        delete[] pData;
        pData = nullptr;
    }

    //  Release the header and meta data buffer
    if (pMetaBuf != nullptr)
    {
        delete[] pMetaBuf;
        pMetaBuf = nullptr;
    }

    return ret;
}    //  NOLINT(impl/function_size)
