﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SettingsManager
{
    using Properties;

    /// <summary>
    /// エラーコードを解析します。
    /// </summary>
    internal class ErrorCodeParser
    {
        private const string Prefix = "ErrorCode:";

        private readonly char[] Separators = new char[] { ':' };

        /// <summary>
        /// ErrorCodeParser クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal ErrorCodeParser()
        {
            this.Message = string.Empty;
        }

        /// <summary>
        /// エラーコードを検出したか否かを表す値を返します。
        /// </summary>
        internal bool Detects { get; private set; }

        /// <summary>
        /// 解析の結果として得られたエラーメッセージを返します。
        /// </summary>
        internal string Message { get; private set; }

        /// <summary>
        /// 指定されたメッセージを解析します。
        /// </summary>
        /// <param name="message">メッセージです。</param>
        internal void Parse(string message)
        {
            if (this.Detects)
            {
                return;
            }

            if (!message.StartsWith(Prefix))
            {
                return;
            }

            string[] args = message.Split(Separators, 3);

            var code = (ErrorCode)int.Parse(args[1]);

            var operand = args.Length < 3 ? string.Empty : args[2];

            this.Dispatch(code, operand);

            this.Detects = true;
        }

        private void Dispatch(ErrorCode code, string operand)
        {
            switch (code)
            {
                case ErrorCode.CommandSpecificationMissing:
                    this.SetCommandSpecificationMissingMessage();
                    break;
                case ErrorCode.CommandNotFound:
                    this.SetCommandNotFoundMessage(operand);
                    break;
                case ErrorCode.CommandNotSupported:
                    this.SetCommandNotSupportedMessage();
                    break;
                case ErrorCode.CommandResetModeInvalid:
                    this.SetResetModeInvalidMessage(operand);
                    break;
                case ErrorCode.FileSpecificationMissing:
                    this.SetFileSpecificationMissingMessage();
                    break;
                case ErrorCode.FileOpenFailure:
                    this.SetFileOpenFailureMessage(operand);
                    break;
                case ErrorCode.FileSizeChangeFailure:
                    this.SetFileSizeChangeFailureMessage(operand);
                    break;
                case ErrorCode.FileSizeAcquisitionFailure:
                    this.SetFileSizeAcquisitionFailureMessage(operand);
                    break;
                case ErrorCode.FileReadingFailure:
                    this.SetFileReadingFailureMessage(operand);
                    break;
                case ErrorCode.FileWritingFailure:
                    this.SetFileWritingFailureMessage(operand);
                    break;
                case ErrorCode.FileFlushingFailure:
                    this.SetFileFlushingFailureMessage(operand);
                    break;
                case ErrorCode.NodeKeyMissing:
                    this.SetNodeKeyMissingMessage(operand);
                    break;
                case ErrorCode.NodeKeyInvalid:
                    this.SetNodeKeyInvalidMessage(operand);
                    break;
                case ErrorCode.NodeTypeUnsupported:
                    this.SetNodeTypeUnsupportedMessage(operand);
                    break;
                case ErrorCode.NodeTypeMismatch:
                    this.SetNodeTypeMismatchMessage(operand);
                    break;
                case ErrorCode.NodeTypeShortArray:
                    this.SetNodeTypeShortArrayMessage(operand);
                    break;
                case ErrorCode.NodeValueInvalid:
                    this.SetNodeValueInvalidMessage(operand);
                    break;
                case ErrorCode.NodeValueTooLong:
                    this.SetNodeValueTooLongMessage(operand);
                    break;
                case ErrorCode.NodeValueOverLimitation:
                    this.SetNodeValueOverLimitationMessage(operand);
                    break;
                case ErrorCode.MemoryAllocationFailure:
                    this.SetMemoryAllocationFailureMessage(operand);
                    break;
                case ErrorCode.DataSizeInvalid:
                    this.SetDataSizeInvalidMessage(operand);
                    break;
                case ErrorCode.JsonParsingFailure:
                    this.SetJsonParsingFailureMessage(operand);
                    break;
                default:
                    this.SetUnknownErrorCodeMessage(code);
                    break;
            }
        }

        private void SetCommandSpecificationMissingMessage()
        {
            this.Message = Resources.ErrorCommandSpecificationMissing;
        }

        private void SetCommandNotFoundMessage(string command)
        {
            this.Message = string.Format(
                Resources.ErrorCommandNotFound, command);
        }

        private void SetCommandNotSupportedMessage()
        {
            this.Message = Resources.ErrorCommandNotSupported;
        }

        private void SetResetModeInvalidMessage(string mode)
        {
            this.Message = string.Format(
                Resources.ErrorCommandResetModeInvalid, mode);
        }

        private void SetFileSpecificationMissingMessage()
        {
            this.Message = Resources.ErrorFileSpecificationMissing;
        }

        private void SetFileOpenFailureMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorFileOpenFailure, args[1], args[0]);
        }

        private void SetFileSizeChangeFailureMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorFileSizeChangeFailure, args[1], args[0]);
        }

        private void SetFileSizeAcquisitionFailureMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorFileSizeAcquisitionFailure, args[1], args[0]);
        }

        private void SetFileReadingFailureMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorFileReadingFailure, args[1], args[0]);
        }

        private void SetFileWritingFailureMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorFileWritingFailure, args[1], args[0]);
        }

        private void SetFileFlushingFailureMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorFileFlushingFailure, args[1], args[0]);
        }

        private void SetNodeKeyMissingMessage(string key)
        {
            this.Message = string.Format(
                Resources.ErrorNodeKeyMissing, key);
        }

        private void SetNodeKeyInvalidMessage(string key)
        {
            this.Message = string.Format(
                Resources.ErrorNodeKeyInvalid, key);
        }

        private void SetNodeTypeUnsupportedMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorNodeTypeUnsupported, args[0], args[1]);
        }

        private void SetNodeTypeMismatchMessage(string operand)
        {
            string[] args = operand.Split(Separators, 3);

            this.Message = string.Format(
                Resources.ErrorNodeTypeMismatch, args[0], args[1], args[2]);
        }

        private void SetNodeTypeShortArrayMessage(string operand)
        {
            string[] args = operand.Split(Separators, 3);

            this.Message = string.Format(
                Resources.ErrorNodeTypeShortArray, args[0], args[1], args[2]);
        }

        private void SetNodeValueInvalidMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorNodeValueInvalid, args[0], args[1]);
        }

        private void SetNodeValueTooLongMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorNodeValueTooLong, args[0], args[1]);
        }

        private void SetNodeValueOverLimitationMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorNodeValueOverLimitation, args[0], args[1]);
        }

        private void SetMemoryAllocationFailureMessage(string size)
        {
            this.Message = string.Format(
                Resources.ErrorMemoryAllocationFailure, size);
        }

        private void SetDataSizeInvalidMessage(string size)
        {
            this.Message = string.Format(
                Resources.ErrorDataSizeInvalid, size);
        }

        private void SetJsonParsingFailureMessage(string operand)
        {
            string[] args = operand.Split(Separators, 2);

            this.Message = string.Format(
                Resources.ErrorJsonParsingFailure, args[0], args[1]);
        }

        private void SetUnknownErrorCodeMessage(ErrorCode code)
        {
            this.Message = string.Format(
                Resources.ErrorUnknownErrorCode, (int)code);
        }
    }
}
