﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SettingsManager.Command
{
    using System.IO;
    using Nintendo.ToolFoundation.CommandLine;
    using Properties;

    /// <summary>
    /// Export コマンドを提供します。
    /// </summary>
    internal sealed class Export : ICommand
    {
        private CommonOptions commonArgs = new CommonOptions();

        private string filePath = string.Empty;

        /// <summary>
        /// コマンドが選択されたか否かを表す値を返します。
        /// </summary>
        public bool IsSelected { get; private set; }

        /// <summary>
        /// コマンドラインパーサにコマンドを登録します。
        /// </summary>
        /// <param name="parser">コマンドラインパーサです。</param>
        public void RegiserWith(DelegateBasedCommandLineParser parser)
        {
            var context = parser
                .AddCommand("Export", () => this.IsSelected = true)
                .SetDescription(Resources.ExportCommand);

            this.commonArgs.RegiserWith(context);

            context
                .AddValue(AcceptFilePathOption, 1)
                .SetValueName("path")
                .SetDescription(Resources.ExportValueFilePath)
                .SetRequired();
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        public void Execute()
        {
            switch (this.commonArgs.Platform)
            {
                case Platform.Horizon:
                    this.ExecuteForHorizon();
                    break;

                case Platform.Windows:
                default:
                    this.ExecuteForWindows();
                    break;
            }
        }

        private static void DeleteFileQuietly(string path)
        {
            try
            {
                File.Delete(path);
            }
            catch
            {
            }
        }

        private void AcceptFilePathOption(string filePath)
        {
            filePath = Path.GetFullPath(filePath);

            if (string.IsNullOrEmpty(Path.GetFileName(filePath)))
            {
                throw new SettingsManagerException(
                    Resources.ErrorInvalidFilePath);
            }

            string directoryName = Path.GetDirectoryName(filePath);

            if (!Directory.Exists(directoryName))
            {
                throw new SettingsManagerException(
                    string.Format(
                        Resources.ErrorDirectoryNotFound,
                        Path.GetFileName(directoryName)));
            }

            this.filePath = filePath;
        }

        private void ExecuteForWindows()
        {
            var platform = Platform.Windows;

            var enginePath = ResourceManager.GetEnginePath(platform);

            var executer = new Executer(platform);

            executer.Ignite(enginePath, new[] { "Export", this.filePath });
        }

        private void ExecuteForHorizon()
        {
            string tempFilePath = Path.GetTempFileName();

            try
            {
                {
                    var platform = Platform.Horizon;

                    var enginePath = ResourceManager.GetEnginePath(platform);

                    var executer = new Executer(platform);

                    executer.Target = this.commonArgs.Target;

                    var args = new[] { "Dump", tempFilePath };

                    executer.Ignite(enginePath, args);
                }

                {
                    var platform = Platform.Windows;

                    var enginePath = ResourceManager.GetEnginePath(platform);

                    var executer = new Executer(platform);

                    var args = new[] { "Parse", tempFilePath, this.filePath };

                    executer.Ignite(enginePath, args);
                }
            }
            finally
            {
                DeleteFileQuietly(tempFilePath);
            }
        }
    }
}
