﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using CommandUtility;
using System.Text.RegularExpressions;
using System.Security.Cryptography;
using YamlDotNet.Serialization;

namespace MakeSignedBinary
{
    public class KeyConfig
    {
        public Int64 Version { get; set; }
        public Int64 KeyGeneration { get; set; }
        public string EncryptionKey { get; set; }
        public string Modulus { get; set; }
        public string PublicExponent { get; set; }
        public string PrivateExponent { get; set; }

        public static KeyConfig Load(string filename)
        {
            return CommandUtility.ConvertUtility.LoadYaml<KeyConfig>(File.ReadAllText(filename));
        }

        [YamlIgnore]
        public byte[] EncryptionKeyBytes
        {
            get
            {
                return ConvertFromHexArrayText(EncryptionKey, 16);
            }
        }

        [YamlIgnore]
        public byte[] PublicExponentBytes
        {
            get
            {
                return ConvertFromHexArrayText(PublicExponent, 4);
            }
        }

        [YamlIgnore]
        public byte[] PrivateExponentBytes
        {
            get
            {
                return ConvertFromHexArrayText(PrivateExponent, 256);
            }
        }

        [YamlIgnore]
        public byte[] ModulusBytes
        {
            get
            {
                return ConvertFromHexArrayText(Modulus, 256);
            }
        }

        private byte[] ConvertFromHexArrayText(string hexText, int expectedLength)
        {
            var bytes = BinaryUtility.FromHexString(Regex.Replace(hexText, "[: \n]", x => ""));

            // openssl の鍵の表示方法の都合で期待した長さより 1 バイト長ければ先頭を削る
            // 最上位ビットが 1 だった場合に負の数と解釈されないように 0x00 を先頭に追加するらしい
            if (bytes.Count() == expectedLength + 1 && bytes[0] == 0)
            {
                bytes = bytes.Skip(1).ToArray();
            }

            if (bytes.Count() != expectedLength)
            {
                throw new Exception(string.Format("Invalid Key Length: {0}", hexText));
            }

            return bytes;
        }

        public static KeyConfig Generate()
        {
            var config = new KeyConfig();

            var param = new RSACryptoServiceProvider(2048).ExportParameters(true);

            config.Version = 0;
            config.KeyGeneration = 0;
            config.EncryptionKey = BinaryUtility.ToPrettyHexString(CryptUtility.GenerateKey());
            config.PublicExponent = BinaryUtility.ToPrettyHexString(new byte[] { 0x00, 0x01, 0x00, 0x01 });
            config.Modulus = BinaryUtility.ToPrettyHexString(param.Modulus);
            config.PrivateExponent = BinaryUtility.ToPrettyHexString(param.D);

            return config;
        }

        public string DumpYaml()
        {
            return ConvertUtility.DumpYaml(this);
        }
    }
}
