﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;

namespace MakeInitialImage
{
    public class Utility
    {
        public static void WriteBinary<StructType>(Stream outputStream, StructType value)
        {
            var size = Marshal.SizeOf(typeof(StructType));
            var buffer = new byte[size];
            var handle = GCHandle.Alloc(buffer, GCHandleType.Pinned);

            try
            {
                Marshal.StructureToPtr(value, handle.AddrOfPinnedObject(), false);
            }
            finally
            {
                handle.Free();
            }

            outputStream.Write(buffer, 0, size);
        }

        public static StructType ReadBinary<StructType>(Stream inputStream)
        {
            var size = Marshal.SizeOf(typeof(StructType));
            var buffer = new byte[size];
            var handle = GCHandle.Alloc(buffer, GCHandleType.Pinned);

            try
            {
                inputStream.Read(buffer, 0, size);
                return (StructType)Marshal.PtrToStructure(handle.AddrOfPinnedObject(), typeof(StructType));
            }
            finally
            {
                handle.Free();
            }
        }

        public static void Enumerate<T>(IEnumerable<T> enumerable, Action<T, int> action)
        {
            var index = 0;
            foreach (var e in enumerable)
            {
                action(e, index);
                index += 1;
            }
        }

        public static string ReplaceVariables(string text, Dictionary<string, string> dictionary)
        {
            var patterns = from tuple in dictionary
                           select new Tuple<string, Regex>(tuple.Value, new Regex("(?<!\\\\)\\$\\{" + tuple.Key + "\\}"));
            return patterns.Aggregate<Tuple<string, Regex>, string>(
                text,
                (replaced, variable) => {
                    return variable.Item2.Replace(replaced, variable.Item1);
                });
        }

        public static long AlignUp(long a, long align)
        {
            return ((a + align - 1) / align) * align;
        }

        public static bool IsAligned(long a, long align)
        {
            return (((a + align - 1) / align) * align) == a;
        }

        public static void CopyDirectory(DirectoryInfo sourceDirectory, DirectoryInfo destinationDirectory)
        {
            if (!Directory.Exists(sourceDirectory.FullName))
            {
                throw new DirectoryNotFoundException(string.Format("Found no source directory: {0}", sourceDirectory));
            }

            if (!Directory.Exists(destinationDirectory.FullName))
            {
                destinationDirectory.Create();
            }

            foreach (FileInfo file in sourceDirectory.GetFiles())
            {
                string destinationFilePath = Path.Combine(destinationDirectory.FullName, file.Name);
                file.CopyTo(destinationFilePath, false);
            }

            foreach (DirectoryInfo subdir in sourceDirectory.GetDirectories())
            {
                string destinationSubDirectoryPath = Path.Combine(destinationDirectory.FullName, subdir.Name);
                CopyDirectory(subdir, new DirectoryInfo(destinationSubDirectoryPath));
            }
        }

        public static FileInfo EnsureExeFile(string inputPath, string defaultName)
        {
            if (inputPath == null)
            {
                var currentApplicationPosition = System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                var targetApplicationPath = Path.Combine(currentApplicationPosition, defaultName);

                if (File.Exists(targetApplicationPath))
                {
                    return new FileInfo(targetApplicationPath);
                }
                else
                {
                    throw new FileNotFoundException(string.Format("{0} is not found.", defaultName));
                }
            }
            else
            {
                if (File.Exists(inputPath))
                {
                    return new FileInfo(inputPath);
                }
                else
                {
                    throw new FileNotFoundException(string.Format("{0} is not found.", inputPath));
                }
            }
        }

        public static FileInfo EnsureExistFile(string filePath)
        {
            var fileInfo = new FileInfo(filePath);
            if (fileInfo.Exists)
            {
                return fileInfo;
            }
            else
            {
                throw new Exception(string.Format("File not found: {0}", fileInfo.FullName));
            }
        }
    }
}
