﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using Nintendo.Foundation.IO;
using CommandUtility;
using System.Text.RegularExpressions;

namespace CmacFileTool.Commands
{
    public class RemoveCmacCommand
    {
        [CommandLineOption('i', "input",
            Description = "set raw intial image or initial image with cmac",
            IsRequired = true)]
        public string InputFile { get; set; }

        [CommandLineOption('o', "output",
            Description = "set output as initial image without cmac",
            IsRequired = true)]
        public string OutputFile { get; set; }

        [CommandLineOption("key",
            Description = "set key.(binary-file-path or text-file-path or 32-hexchars)")]
        public string Key { get; set; }

        [CommandLineOption("disable-verify",
            Description = "remove cmac without verify.",
            DefaultValue = false)]
        public bool DisableVerify { get; set; }

        public void Run()
        {
            byte[] kek;
            if (DisableVerify)
            {
                kek = CryptUtility.GenerateKey();
            }
            else if (Key != null)
            {
                kek = AppendCmacCommand.LoadKeyFromArgument(Key);
            }
            else
            {
                throw new Exception("needs key for cmac file");
            }

            var inputFileInfo = new FileInfo(InputFile);
            var outputFileInfo = new FileInfo(OutputFile);

            using (var reader = inputFileInfo.OpenRead())
            using (var writer = outputFileInfo.OpenWrite())
            {
                // ヘッダ読む
                var header = CmacFileHeader.Load(reader, kek, DisableVerify);

                for (int blockIndex = 0; blockIndex < header.NumberOfBlocks; blockIndex++)
                {
                    // CMAC の読み込み
                    var cmac = new byte[16];
                    reader.Read(cmac, 0, cmac.Length);

                    // データの読み込み
                    var indexData = BinaryUtility.ToBinary<int>(blockIndex);
                    var buffer = new byte[indexData.Length + header.BlockSize];
                    indexData.CopyTo(buffer, 0);
                    reader.Read(buffer, indexData.Length, header.BlockSize);

                    // 検証
                    if (!DisableVerify)
                    {
                        CmacUtility.Verify(buffer, cmac, header.CmacKey);
                    }

                    // 書き込み
                    writer.Write(buffer, indexData.Length, header.BlockSize);
                }
            }
        }
    }
}
