﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Threading;
using System.Threading.Tasks.Dataflow;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Channels.Ipc;
using System.Runtime.Remoting;
using Nintendo.Htcs;

namespace Nintendo.Log
{
    public class LogServer : IDisposable
    {
        public LogServer(string portName)
        {
            PortName = portName;

            BroadcastBlock = new BroadcastBlock<Dictionary<string, object>>(log => log);

            ParseBlock.LinkTo(BroadcastBlock);

            ConcatBlock = CreateConcatBlock();
            ConcatBlock.LinkTo(ParseBlock);

            InitializeRemoteObject();
        }

        public void Dispose()
        {
            var doStop = false;
            lock (StartedLock)
            {
                doStop = Started;
            }
            if (doStop)
            {
                Stop();
            }
        }

        public void Start()
        {
            lock (StartedLock)
            {
                if (Started)
                {
                    throw new InvalidOperationException("LogServer is already started.");
                }

                ClientTable = new HashSet<PortMapItem>();
                HtcsProxy.OnPortMapUpdate += ConnectionFunc;
                HtcsProxy.Start();
                CancellationTokenSource = new CancellationTokenSource();
                Started = true;
            }
        }

        public void Stop()
        {
            lock (StartedLock)
            {
                if (!Started)
                {
                    throw new InvalidOperationException("LogServer is not started.");
                }

                HtcsProxy.OnPortMapUpdate -= ConnectionFunc;
                HtcsProxy.Stop();
                CancellationTokenSource.Cancel();
                lock(ClientTable)
                {
                    while (ClientTable.Count > 0)
                    {
                        Monitor.Wait(ClientTable);
                    }
                }
                Started = false;
            }
        }

        internal BufferBlock<Dictionary<string, object>> CreateLogBufferBlock()
        {
            var bufferBlock = new BufferBlock<Dictionary<string, object>>();
            BroadcastBlock.LinkTo(bufferBlock);
            return bufferBlock;
        }

        private void InitializeRemoteObject()
        {
            ChannelServices.RegisterChannel(new IpcServerChannel(LogServiceUrl.PortName), true);
            RemoteObject = new LogService(this);
            RemotingServices.Marshal(RemoteObject, LogServiceUrl.ObjectUri, typeof(LogService));
        }

        // シリアライズされた LogRecord を Key-Value 形式に変換するブロック
        private TransformBlock<LogRecord, Dictionary<string, object>> ParseBlock =
             new TransformBlock<LogRecord, Dictionary<string, object>>((logRecord) =>
        {
            var header = logRecord.Header;
            var payload = logRecord.Payload;
            var output = new Dictionary<string, object>();

            output["PeerName"] = header.PeerName;
            output["ProcessId"] = header.ProcessId;
            output["ThreadId"] = header.ThreadId;
            output["Severity"] = header.Severity;
            output["Verbosity"] = header.Verbosity;
            output["Timestamp"] = DateTime.Now; // タイムスタンプはログがすべて揃ったこの時点で追加する

            return
                output.Concat(
                    LogDataChunk.Parse(payload)).ToDictionary(
                        pair => pair.Key, pair => pair.Value);
        });

        // LogPacket を連結して、LogRecord を構築するブロックを生成する
        private IPropagatorBlock<LogPacket, LogRecord> CreateConcatBlock()
        {
            var logRecordBuffer = new Dictionary<LogRecordHeader, byte[]>();

            var inBlock = new BufferBlock<LogPacket>();
            var outBlock = new BufferBlock<LogRecord>();
            var actionBlock = new ActionBlock<LogPacket>(logPacket =>
            {
                var logRecordHeader = new LogRecordHeader();
                logRecordHeader.PeerName = logPacket.Header.PeerName;
                logRecordHeader.ProcessId = logPacket.Header.ProcessId;
                logRecordHeader.ThreadId = logPacket.Header.ThreadId;
                logRecordHeader.Severity = logPacket.Header.Severity;
                logRecordHeader.Verbosity = logPacket.Header.Verbosity;

                if (logPacket.Header.IsHead)
                {
                    logRecordBuffer[logRecordHeader] = logPacket.Payload;
                }
                else
                {
                    if (!logRecordBuffer.ContainsKey(logRecordHeader))
                    {
                        // 先頭パケットを受信していない場合はバッファにヘッダが見つからない
                        return; // 先頭パケットから受信できなかった場合は捨てる
                    }
                    logRecordBuffer[logRecordHeader] =
                        logRecordBuffer[logRecordHeader].Concat(logPacket.Payload).ToArray();
                }

                if (logPacket.Header.IsTail)
                {
                    var logRecord = new LogRecord();
                    logRecord.Header = logRecordHeader;
                    logRecord.Payload = logRecordBuffer[logRecordHeader];
                    logRecordBuffer.Remove(logRecordHeader);
                    outBlock.Post(logRecord);
                }
            });
            inBlock.LinkTo(actionBlock);

            return DataflowBlock.Encapsulate(inBlock, outBlock);
        }

        private void ClientFunc(TcpClient client, string peerName)
        {
            using (new LogPort(peerName, "iywys@$Log", BroadcastBlock))
            using (new LogPort(peerName, "@Log", BroadcastBlock))
            using (new JsonLogPort(peerName, "@JsonLog", BroadcastBlock))
            //using (new XmlLogPort(peerName, "@XmlLog", BroadcastBlock))
            using (var bs = new BufferedStream(client.GetStream()))
            using (CancellationTokenSource.Token.Register(() => bs.Close()))
            {
                while (true)
                {
                    LogPacket packet;
                    try
                    {
                        var headerBuffer = new byte[LogPacketHeader.HeaderSize];
                        bs.ReadSync(headerBuffer, 0, LogPacketHeader.HeaderSize);

                        var header = LogPacketHeader.Parse(headerBuffer);
                        header.PeerName = peerName;

                        var payloadBuffer = new byte[header.PayloadSize];
                        bs.ReadSync(payloadBuffer, 0, header.PayloadSize);

                        packet = new LogPacket();
                        packet.Header = header;
                        packet.Payload = payloadBuffer;
                    }
                    catch (Exception exception) when (
                        exception is IOException ||
                        exception is ObjectDisposedException)
                    {
                        break; // ターゲットとの接続が切れたので抜ける
                    }

                    ConcatBlock.Post(packet);
                }
            }
            Debug.Assert(!client.Connected);
            client.Close();
        }

        private void ConnectionFunc(List<PortMapItem> PortMap)
        {
            lock (ClientTable)
            {
                var newClientPortMapItem = PortMap.Where(
                    portMapItem =>
                        portMapItem.HtcsEndPoint.HtcsPortName == PortName &&
                        !ClientTable.Contains(portMapItem)).ToArray();

                foreach (var portMapItem in newClientPortMapItem)
                {
                    ClientTable.Add(portMapItem);
                    var thread = new Thread(() =>
                    {
                        try
                        {
                            var tcpClient = new TcpClient();

                            try
                            {
                                tcpClient.Connect(portMapItem.IPEndPoint);
                            }
                            catch
                            {
                                return;
                            }

                            ClientFunc(
                                tcpClient, portMapItem.HtcsEndPoint.HtcsPeerName);
                        }
                        finally
                        {
                            lock (ClientTable)
                            {
                                ClientTable.Remove(portMapItem);
                                if (!(ClientTable.Count > 0))
                                {
                                    Monitor.Pulse(ClientTable);
                                }
                            }
                        }
                    });
                    thread.Start();
                }
            }
        }

        private readonly string PortName;
        private object StartedLock = new object();
        private bool Started = false;
        private LogService RemoteObject;
        private HashSet<PortMapItem> ClientTable;
        private const int ConnectionPollingIntervalMilliseconds = 1000;
        private IPropagatorBlock<LogPacket, LogRecord> ConcatBlock;
        private BroadcastBlock<Dictionary<string, object>> BroadcastBlock;
        private CancellationTokenSource CancellationTokenSource;
    }

    internal static class BufferedStreamExtention
    {
        // size バイト受信するまでブロックする
        public static void ReadSync(this BufferedStream networkStream, byte[] buffer, int offset, int size)
        {
            var readSize = 0;
            while (readSize < size)
            {
                var result = networkStream.Read(buffer, offset + readSize, size - readSize);
                if (result <= 0)
                {
                    throw new IOException(); // ターゲットとの接続が切れた
                }
                readSize += result;
            }
            Debug.Assert(readSize == size);
        }
    }
}
