﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using DataChunkParser = System.Func<byte[], object>;

namespace Nintendo.Log
{
    internal class LogDataChunk
    {
        private static DataChunkParser StringParser = buffer =>
        {
            return Encoding.UTF8.GetString(buffer, 0, buffer.Length);
        };

        private static DataChunkParser IntParser = buffer =>
        {
            Debug.Assert(buffer.Length == sizeof(int));
            return BitConverter.ToInt32(buffer, 0);
        };

        private static readonly DateTime posixTimeBase = new DateTime(1970, 1, 1, 0, 0, 0);

        private static DataChunkParser PosixTimeParser = buffer =>
        {
            Debug.Assert(buffer.Length == sizeof(long));
            return posixTimeBase.AddSeconds(BitConverter.ToInt64(buffer, 0));
        };

        private static DataChunkParser BoolParser = buffer =>
        {
            Debug.Assert(buffer.Length == sizeof(bool));
            return BitConverter.ToBoolean(buffer, 0);
        };

        private static DataChunkParser UInt8Parser = buffer =>
        {
            Debug.Assert(buffer.Length == 1);
            return (uint)buffer[0];
        };

        private struct DataChunkTableItem
        {
            public DataChunkTableItem(string keyName, DataChunkParser parser)
            {
                KeyName = keyName;
                Parser = parser;
            }

            public string KeyName;
            public DataChunkParser Parser;
        }

        private static readonly Dictionary<int, DataChunkTableItem> ChunkInfoTable = new Dictionary<int, DataChunkTableItem>()
        {
            { 0, new DataChunkTableItem("LogSessionBegin",   BoolParser) },
            { 1, new DataChunkTableItem("LogSessionEnd",     BoolParser) },
            { 2, new DataChunkTableItem("TextLog",           StringParser) },
            { 3, new DataChunkTableItem("LineNumber",        IntParser) },
            { 4, new DataChunkTableItem("FileName",          StringParser) },
            { 5, new DataChunkTableItem("FunctionName",      StringParser) },
            { 6, new DataChunkTableItem("ModuleName",        StringParser) },
            { 7, new DataChunkTableItem("ThreadName",        StringParser) },
            { 9, new DataChunkTableItem("UserSystemClock",   PosixTimeParser) },
            { 10, new DataChunkTableItem("ProcessName",      StringParser) },
        };

        private static KeyValuePair<string, object> ParseDataChunk(int key, byte[] buffer)
        {
            return new KeyValuePair<string, object>(
                ChunkInfoTable[key].KeyName,
                ChunkInfoTable[key].Parser(buffer));
        }

        // buffer[offset] から ULEB128 フォーマットで整数を読み込み offset を進める
        private static int ReadULEB128(byte[] buffer, ref int offset)
        {
            var bitBase = 0;
            var value = 0;
            do
            {
                value = value | ((buffer[offset] & 0x7F) << bitBase);
                bitBase += 7;
            } while ((buffer[offset++] & 0x80) != 0);
            return value;
        }

        // ペイロードをデータチャンクに分割して、同一のキーを持つデータチャンクをすべて連結する
        private static Dictionary<int, byte[]> Concat(byte[] buffer)
        {
            var output = new Dictionary<int, byte[]>();

            var offset = 0;
            while (offset < buffer.Length)
            {
                var key = ReadULEB128(buffer, ref offset);
                var size = ReadULEB128(buffer, ref offset);

                if (output.ContainsKey(key))
                {
                    output[key] = output[key].Concat(buffer.Skip(offset).Take(size)).ToArray();
                }
                else
                {
                    output[key] = buffer.Skip(offset).Take(size).ToArray();
                }

                offset += size;
            }
            Debug.Assert(offset == buffer.Length);

            return output;
        }

        // データチャンクのバイト列をパースして string => object の辞書に変換する
        public static Dictionary<string, object> Parse(byte[] buffer)
        {
            return Concat(buffer).Where(
                pair => ChunkInfoTable.ContainsKey(pair.Key)).Select(
                    pair => ParseDataChunk(pair.Key, pair.Value)).ToDictionary(
                        pair => pair.Key, pair => pair.Value);
        }
    }
}
