﻿using System;
using System.Diagnostics;
using System.Drawing;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace Nintendo.Log
{
    internal class LogServerWindow : Form
    {
        private readonly TimeSpan TimeoutForShutdown = TimeSpan.FromSeconds(5);

        public LogServerWindow()
        {
            this.Text = "Nintendo Log Server";

            NotifyIcon = new NotifyIcon();
            NotifyIcon.Icon = GetIcon();
            NotifyIcon.Text = this.Text;
            NotifyIcon.Visible = true;

            var contextMenuStrip = new ContextMenuStrip();
            NotifyIcon.ContextMenuStrip = contextMenuStrip;

            var shutdownItem = new ToolStripMenuItem("Shutdown");
            shutdownItem.Click += (sender, e) => Application.Exit();
            contextMenuStrip.Items.Add(shutdownItem);

            this.ShowInTaskbar = false;
            this.WindowState = FormWindowState.Minimized;
            this.Load += (sender, e) => this.Hide();
            this.FormClosed += (sender, e) =>
            {
                NotifyIcon.Visible = false;
                TimeoutKill(Process.GetCurrentProcess(), TimeoutForShutdown);
            };
        }

        private static void TimeoutKill(Process process, TimeSpan timeout)
        {
            Task.Factory.StartNew(() =>
            {
                Thread.Sleep((int)timeout.TotalMilliseconds);
                process.Kill();
            });
        }

        private Icon GetIcon()
        {
            using (var iconBitmap = new Bitmap(16, 16))
            {
                using (var g = Graphics.FromImage(iconBitmap))
                {
                    g.FillRectangle(Brushes.Black, new Rectangle(3, 1, 10, 14));
                    g.FillRectangle(Brushes.Orange, new Rectangle(5, 3, 6, 2));
                    using (var b = new SolidBrush(Color.FromArgb(63, 0, 0, 0)))
                    {
                        g.FillRectangle(b, new Rectangle(13, 3, 1, 12));
                    }
                }
                return Icon.FromHandle(iconBitmap.GetHicon());
            }
        }

        private NotifyIcon NotifyIcon;
    }
}
