﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Runtime.Remoting;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Web.Script.Serialization;
using System.Windows.Forms;

namespace Nintendo.Log
{
    public class HtcsLogService : ILogService
    {
        Boolean KeepGoing = true;
        BufferBlock<Dictionary<string, object>> BufferBlock = new BufferBlock<Dictionary<string, object>>();
        CancellationTokenSource receiveCancellationTokenSource = new CancellationTokenSource();

        Dictionary<string, object> ILogService.ReceiveLog(int pid)
        {
            Dictionary<string, object> output = BufferBlock.Receive(receiveCancellationTokenSource.Token);
            return output;
        }

        Socket Listener = null;

        public void StartListening()
        {
            Application.ApplicationExit += OnApplicationExit;
            byte[] bytes = null;
            IPHostEntry ipHostInfo = Dns.GetHostEntry("localhost");
            IPAddress ipAddress = ipHostInfo.AddressList[0];
            foreach (IPAddress address in ipHostInfo.AddressList)
            {
                if (address.AddressFamily == AddressFamily.InterNetwork)
                {
                    ipAddress = address;
                    break;
                }
            }
            IPEndPoint localEndPoint = new IPEndPoint(ipAddress, 17102);

            Listener = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);

            try
            {
                Listener.Connect(localEndPoint);
                while(Listener.Connected)
                {
                    bytes = new byte[LogPacketHeader.HeaderSize];
                    Listener.ReceiveSync(bytes);
                    LogPacketHeader header = LogPacketHeader.Parse(bytes);
                    bytes = new byte[header.PayloadSize];
                    Listener.ReceiveSync(bytes);

                    LogPacket packet;
                    packet.Header = header;
                    packet.Payload = bytes;

                    bytes = new byte[sizeof(Int32)];
                    Listener.ReceiveSync(bytes);
                    bytes = new byte[BitConverter.ToInt32( bytes, 0 )];
                    Listener.ReceiveSync(bytes);

                    string combined = System.Text.Encoding.UTF8.GetString(bytes).Trim();
                    Int32 markerIndex = combined.LastIndexOf('|');
                    string peerName = combined.Substring(0, markerIndex);
                    DateTime timestamp = Convert.ToDateTime(combined.Substring(markerIndex + 1));

                    LogRecordHeader recordHeader = LogClient.GetLogRecordHeader(packet);
                    var logRecordBuffer = new Dictionary<LogRecordHeader, byte[]>();
                    logRecordBuffer[recordHeader] = packet.Payload;

                    Dictionary<string, object> output = LogClient.GetLogRecordDictionary(recordHeader, logRecordBuffer);
                    if (output.ContainsKey("TextLog"))
                    {
                        output["PeerName"] = peerName;
                        output["Timestamp"] = timestamp;
                        BufferBlock.Post(output);
                    }
                }
            }
            catch(Exception e)
            {
                Console.WriteLine(e.ToString());
            }

            if(KeepGoing)
            {
                Listener.Close();
                KeepGoing = false;
                receiveCancellationTokenSource.Cancel();
                LogClient.InitializeRemoteObject();
            }
        }

        private void OnApplicationExit(object sender, EventArgs e)
        {
            Listener.Close();
            KeepGoing = false;
            receiveCancellationTokenSource.Cancel();
        }
    }

    internal static class SocketExtensions
    {
        public static void ReceiveSync(this Socket socket, byte[] buffer)
        {
            int offset = 0;
            while (offset < buffer.Length)
            {
                var result = socket.Receive(buffer, offset, buffer.Length - offset, SocketFlags.None);
                if (!(result > 0))
                {
                    throw new SocketException();
                }
                offset += result;
            }
            Debug.Assert(offset == buffer.Length);
        }
    }

    public static class LogClient
    {
        private static void InitializeIfNeeded()
        {
            lock (InitializedLock)
            {
                if (Initialized)
                {
                    return;
                }

                InitializeRemoteObject();
                InitializeBroadcastBlock();
                Initialized = true;
            }
        }

        public static void InitializeRemoteObject()
        {
            RemoteObject = new HtcsLogService();
            new Thread(() => ((HtcsLogService)RemoteObject).StartListening() ).Start();
        }

        private static void InitializeBroadcastBlock()
        {
            BroadcastBlock = new BroadcastBlock<Dictionary<string, object>>(log => log);
            Task.Factory.StartNew(() =>
            {
                var pid = Process.GetCurrentProcess().Id;
                while (true) // TODO: キャンセルによる終了
                {
                    try
                    {
                        var log = RemoteObject.ReceiveLog(pid); // サーバーからの受信
                        if(log != null)
                            BroadcastBlock.Post(log); // 受信したログを配信
                    }
                    catch (Exception e) when (
                        e is RemotingException ||
                        e is OperationCanceledException)
                    {
                        Thread.Sleep(1000); // サーバーがいないときはスリープしながらポーリング
                        // TODO: サーバーの起動
                    }
                }
            });
        }

        public static IDisposable LinkToLogBroadcastBlock(
            ITargetBlock<Dictionary<string, object>> bufferBlock, Predicate<Dictionary<string, object>> filterFunction)
        {
            InitializeIfNeeded();

            return filterFunction == null
                ? BroadcastBlock.LinkTo(bufferBlock)
                : BroadcastBlock.LinkTo(bufferBlock, filterFunction);
        }

        private static IEnumerable<Dictionary<string, object>> GetLogEnumerator(
            Predicate<Dictionary<string, object>> filterFunction, CancellationToken token)
        {
            var bufferBlock = new BufferBlock<Dictionary<string, object>>();
            using (LinkToLogBroadcastBlock(bufferBlock, filterFunction))
            {
                while (true)
                {
                    Dictionary<string, object> log;
                    try
                    {
                        log = bufferBlock.Receive(token);
                    }
                    catch (OperationCanceledException)
                    {
                        yield break;
                    }
                    yield return log;
                }
            }
        }

        private static bool CheckLogManagerBinaryHeader(FileStream fs, string path)
        {
            var headerBuffer = new byte[LogManagerBinaryHeader.HeaderSize];
            var readSize = fs.Read(headerBuffer, 0, LogManagerBinaryHeader.HeaderSize);

            if (readSize != LogManagerBinaryHeader.HeaderSize)
            {
                MessageBox.Show($"The specified file is not supported.{Environment.NewLine}{path}");
                return false;
            }

            var header = LogManagerBinaryHeader.Parse(headerBuffer);
            if (header.MagicNumber != LogManagerBinaryHeader.SupposedMagicNumber)
            {
                MessageBox.Show($"The specified file is not supported.{Environment.NewLine}{path}");
                return false;
            }

            return true;
        }

        public static LogRecordHeader GetLogRecordHeader(LogPacket logPacket)
        {
            var logRecordHeader = new LogRecordHeader();
            logRecordHeader.PeerName = logPacket.Header.PeerName;
            logRecordHeader.ProcessId = logPacket.Header.ProcessId;
            logRecordHeader.ThreadId = logPacket.Header.ThreadId;
            logRecordHeader.Severity = logPacket.Header.Severity;
            logRecordHeader.Verbosity = logPacket.Header.Verbosity;

            return logRecordHeader;
        }

        public static Dictionary<string, object> GetLogRecordDictionary(
            LogRecordHeader logRecordHeader, Dictionary<LogRecordHeader, byte[]> logRecordBuffer)
        {
            var logRecord = new LogRecord();
            logRecord.Header = logRecordHeader;
            logRecord.Payload = logRecordBuffer[logRecordHeader];
            logRecordBuffer.Remove(logRecordHeader);
            var output = new Dictionary<string, object>();

            output["PeerName"] = logRecord.Header.PeerName;
            output["ProcessId"] = logRecord.Header.ProcessId;
            output["ThreadId"] = logRecord.Header.ThreadId;
            output["Severity"] = logRecord.Header.Severity;
            output["Verbosity"] = logRecord.Header.Verbosity;
            // タイムスタンプは、ログファイルに含まれない。

            return output.Concat(LogDataChunk.Parse(logRecord.Payload)).
                ToDictionary( pair => pair.Key, pair => pair.Value );
        }

        public static IEnumerable<Dictionary<string, object>> GetLogsFromNxbinlog(string path)
        {
            return GetLogsFromNxbinlog(path, null, null);
        }

        public static IEnumerable<Dictionary<string, object>> GetLogsFromNxbinlog(string path, CancellationToken? token, Action<long, long> progressCallback)
        {
            long totalReadSize = 0;
            long fileSize = new FileInfo(path).Length;

            List<Dictionary<string, object>> logList = new List<Dictionary<string, object>>();

            using (var fs = new FileStream(path, FileMode.Open, FileAccess.Read))
            {
                if (!CheckLogManagerBinaryHeader(fs, path))
                {
                    yield break;
                }

                var logRecordBuffer = new Dictionary<LogRecordHeader, byte[]>();
                while (!(token?.IsCancellationRequested ?? false))
                {
                    LogPacket logPacket;

                    var headerBuffer = new byte[LogPacketHeader.HeaderSize];
                    var readSize = fs.Read(headerBuffer, 0, LogPacketHeader.HeaderSize);

                    if (readSize != LogPacketHeader.HeaderSize)
                    {
                        yield break;
                    }
                    totalReadSize += readSize;

                    var header = LogPacketHeader.Parse(headerBuffer);

                    var payloadBuffer = new byte[header.PayloadSize];
                    readSize = fs.Read(payloadBuffer, 0, header.PayloadSize);

                    if (readSize != header.PayloadSize)
                    {
                        yield break;
                    }
                    totalReadSize += readSize;

                    logPacket = new LogPacket();
                    logPacket.Header = header;
                    logPacket.Payload = payloadBuffer;

                    var logRecordHeader = GetLogRecordHeader(logPacket);
                    if (logPacket.Header.IsHead)
                    {
                        logRecordBuffer[logRecordHeader] = logPacket.Payload;
                    }
                    else
                    {
                        if (!logRecordBuffer.ContainsKey(logRecordHeader))
                        {
                            // 先頭パケットを受信していない場合はバッファにヘッダが見つからない
                            continue; // 先頭パケットから受信できなかった場合は捨てる
                        }
                        logRecordBuffer[logRecordHeader] =
                            logRecordBuffer[logRecordHeader].Concat(logPacket.Payload).ToArray();
                    }

                    if (logPacket.Header.IsTail)
                    {
                        var dic = GetLogRecordDictionary(logRecordHeader, logRecordBuffer);
                        if (dic.ContainsKey("TextLog"))
                        {
                            yield return dic;
                            progressCallback?.Invoke(totalReadSize, fileSize);
                        }
                    }
                }
            }
        }

        public static IEnumerable<Dictionary<string, object>> GetLogsFromJson(string path)
        {
            return JsonLogParser.Parse(path);
        }

        private static string ToPlainText(Dictionary<string, object> log)
        {
            if (!log.ContainsKey("TextLog"))
            {
                throw new ArgumentException();
            }
            return (string)log["TextLog"];
        }

        private static string ToJsonText(Dictionary<string, object> log, JavaScriptSerializer serializer)
        {
            var logForJson = new Dictionary<string, object>();
            foreach (var pair in log)
            {
                logForJson[pair.Key] =
                    (pair.Key == "Timestamp")
                        ? ((DateTime)log["Timestamp"]).ToString()
                        : pair.Value;
            }
            return serializer.Serialize(logForJson);
        }

        private static void WriteAsPlainText(TextWriter writer, IEnumerable<Dictionary<string, object>> logs)
        {
            foreach (var log in logs)
            {
                writer.Write(log["TextLog"]);
            }
        }

        private static void WriteAsJson(TextWriter writer, IEnumerable<Dictionary<string, object>> logs)
        {
            var serializer = new JavaScriptSerializer();

            writer.WriteLine("[");

            var first = logs.FirstOrDefault();
            if (first != null)
            {
                writer.Write(ToJsonText(first, serializer));
                foreach (var log in logs.Skip(1))
                {
                    writer.WriteLine(",");
                    writer.Write(ToJsonText(log, serializer));
                }
                writer.WriteLine();
            }

            writer.WriteLine("]");
        }

        public static Task CreateNxbinlogToTextTask(string outPath, string path, CancellationToken token, Action<long, long> progressCallback)
        {
            return CreateSaveLogTask(outPath, GetLogsFromNxbinlog(path, token, progressCallback), WriteAsPlainText);
        }

        public static Task CreateNxbinlogToJsonTask(string outPath, string path, CancellationToken token, Action<long, long> progressCallback)
        {
            return CreateSaveLogTask(outPath, GetLogsFromNxbinlog(path, token, progressCallback), WriteAsJson);
        }

        public static Task CreateSavePlainTextLogTask(string path, CancellationToken token, Predicate<Dictionary<string, object>> predicate)
        {
            return CreateSaveLogTask(path, GetLogEnumerator(log => log.ContainsKey("TextLog") && predicate(log), token), WriteAsPlainText);
        }

        public static Task CreateSaveJsonLogTask(string path, CancellationToken token, Predicate<Dictionary<string, object>> predicate)
        {
            return CreateSaveLogTask(path, GetLogEnumerator(log => log.ContainsKey("TextLog") && predicate(log), token), WriteAsJson);
        }

        private static Task CreateSaveLogTask(string path, IEnumerable<Dictionary<string, object>> logs, Action<TextWriter, IEnumerable<Dictionary<string, object>>> formatter)
        {
            return
                new Task(() =>
                {
                    using (var sw = new StreamWriter(path, false, new UTF8Encoding(false)))
                    {
                        sw.AutoFlush = true;
                        formatter(sw, logs);
                    }
                });
        }

        private static object InitializedLock = new object();
        private static bool Initialized = false;
        private static ILogService RemoteObject;
        private static BroadcastBlock<Dictionary<string, object>> BroadcastBlock;
    }
}
