﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <limits>

#define NOMINMAX

#include <nn/htcfs/htcfs_ResultPrivate.h>

#include "HtcfsNativeLibrary.h"

namespace Nintendo { namespace HtcTools { namespace HtcfsNativeLibrary {

namespace {

template<class ReturnValueType>
const ReturnValueType& ReturnAndDeclareAlive(Object ^obj, const ReturnValueType& retValue)
{
    GC::KeepAlive(obj);
    return retValue;
}

}

int32_t File::Read([Runtime::InteropServices::OutAttribute] int64_t %  outValue, int64_t offset, array<Byte>^ buffer, int64_t size)
{
    if (size < 0 || static_cast<uint64_t>(size) > std::numeric_limits<size_t>::max())
    {
        return ReturnAndDeclareAlive(this, nn::htcfs::ResultInvalidRequest().GetInnerValueForDebug());
    }

    if (size == 0)
    {
        return ReturnAndDeclareAlive(this, nn::ResultSuccess().GetInnerValueForDebug());
    }

    pin_ptr<uint8_t> pinBuffer = &buffer[0];

    size_t value;
    auto result = m_pImpl->Read(&value, offset, pinBuffer, static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0));

    pinBuffer = nullptr;

    if (result.IsSuccess())
    {
        outValue = static_cast<int64_t>(value);
    }

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t File::Write(int64_t offset, array<Byte>^ buffer, int64_t size, int32_t option)
{
    if (size < 0 || static_cast<uint64_t>(size) > std::numeric_limits<size_t>::max())
    {
        return ReturnAndDeclareAlive(this, nn::htcfs::ResultInvalidRequest().GetInnerValueForDebug());
    }

    if (size == 0)
    {
        return ReturnAndDeclareAlive(this, nn::ResultSuccess().GetInnerValueForDebug());
    }

    pin_ptr<uint8_t> pinBuffer = &buffer[0];

    auto result = m_pImpl->Write(offset, pinBuffer, static_cast<size_t>(size), nn::fs::WriteOption::MakeValue(option));

    pinBuffer = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t File::Flush()
{
    auto result = m_pImpl->Flush();
    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t File::SetSize(int64_t size)
{
    auto result = m_pImpl->SetSize(size);
    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t File::GetSize([Runtime::InteropServices::OutAttribute] int64_t % outValue)
{
    int64_t value;
    auto result = m_pImpl->GetSize(&value);

    if (result.IsSuccess())
    {
        outValue = value;
    }

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t Directory::Read([Runtime::InteropServices::OutAttribute] int64_t % outValue, array<Byte>^ buffer, int64_t bufferSize)
{
    NN_ABORT_UNLESS(bufferSize % sizeof(nn::fs::DirectoryEntry) == 0);

    if (bufferSize == 0)
    {
        return ReturnAndDeclareAlive(this, nn::ResultSuccess().GetInnerValueForDebug());
    }

    pin_ptr<uint8_t> pinBuffer = &buffer[0];

    int64_t value;
    auto result = m_pImpl->Read(&value, reinterpret_cast<nn::fs::DirectoryEntry*>(pinBuffer), bufferSize / sizeof(nn::fs::DirectoryEntry));

    pinBuffer = nullptr;

    if (result.IsSuccess())
    {
        outValue = value;
    }

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t Directory::GetDirectoryEntrySize()
{
    return sizeof(nn::fs::DirectoryEntry);
}

int32_t Directory::GetEntryCount([Runtime::InteropServices::OutAttribute] int64_t % outValue)
{
    int64_t value;
    auto result = m_pImpl->GetEntryCount(&value);

    if (result.IsSuccess())
    {
        outValue = value;
    }

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoCreateFile(array<Byte>^ path, int64_t size)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    auto result = m_pImpl->DoCreateFile(reinterpret_cast<const char*>(pinPath), size, 0);

    pinPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoDeleteFile(array<Byte>^ path)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    auto result = m_pImpl->DoDeleteFile(reinterpret_cast<const char*>(pinPath));

    pinPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoCreateDirectory(array<Byte>^ path)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    auto result = m_pImpl->DoCreateDirectory(reinterpret_cast<const char*>(pinPath));

    pinPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoDeleteDirectory(array<Byte>^ path)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    auto result = m_pImpl->DoDeleteDirectory(reinterpret_cast<const char*>(pinPath));

    pinPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoDeleteDirectoryRecursively(array<Byte>^ path)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    auto result = m_pImpl->DoDeleteDirectoryRecursively(reinterpret_cast<const char*>(pinPath));
    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}
int32_t FileSystem::DoCleanDirectoryRecursively(array<Byte>^ path)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    auto result = m_pImpl->DoCleanDirectoryRecursively(reinterpret_cast<const char*>(pinPath));

    pinPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoRenameFile(array<Byte>^ currentPath, array<Byte>^ newPath)
{
    pin_ptr<uint8_t> pinCurrentPath = &currentPath[0];
    pin_ptr<uint8_t> pinNewPath = &newPath[0];

    auto result = m_pImpl->DoRenameFile(reinterpret_cast<const char*>(pinCurrentPath), reinterpret_cast<const char*>(pinNewPath));

    pinCurrentPath = nullptr;
    pinNewPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoRenameDirectory(array<Byte>^ currentPath, array<Byte>^ newPath)
{
    pin_ptr<uint8_t> pinCurrentPath = &currentPath[0];
    pin_ptr<uint8_t> pinNewPath = &newPath[0];

    auto result = m_pImpl->DoRenameDirectory(reinterpret_cast<const char*>(pinCurrentPath), reinterpret_cast<const char*>(pinNewPath));

    pinCurrentPath = nullptr;
    pinNewPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoGetEntryType([System::Runtime::InteropServices::OutAttribute] int32_t % directoryEntryType, array<Byte>^ path)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    nn::fs::DirectoryEntryType value;
    auto result = m_pImpl->DoGetEntryType(&value, reinterpret_cast<const char*>(pinPath));

    directoryEntryType = value;

    pinPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoOpenFile([System::Runtime::InteropServices::OutAttribute] File^ % file, array<Byte>^ path, int32_t mode)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    std::unique_ptr<nn::fs::fsa::IFile> pFile;
    auto result = m_pImpl->DoOpenFile(&pFile, reinterpret_cast<const char*>(pinPath), static_cast<nn::fs::OpenMode>(mode));

    if (result.IsSuccess())
    {
        file = gcnew File(pFile.release());
    }

    pinPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}


int32_t FileSystem::DoOpenDirectory([System::Runtime::InteropServices::OutAttribute] Directory^ % directory, array<Byte>^ path, int32_t mode)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    std::unique_ptr<nn::fs::fsa::IDirectory> pDirectory;
    auto result = m_pImpl->DoOpenDirectory(&pDirectory, reinterpret_cast<const char*>(pinPath), static_cast<nn::fs::OpenDirectoryMode>(mode));

    if (result.IsSuccess())
    {
        directory = gcnew Directory(pDirectory.release());
    }

    pinPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoGetFileTimeStampRaw(
    [System::Runtime::InteropServices::OutAttribute] int64_t % createTime,
    [System::Runtime::InteropServices::OutAttribute] int64_t % modifyTime,
    [System::Runtime::InteropServices::OutAttribute] int64_t % accessTime,
    array<Byte>^ path)
{
    pin_ptr<uint8_t> pinPath = &path[0];

    nn::fs::FileTimeStampRaw fileTimeStampRaw;

    auto result = m_pImpl->DoGetFileTimeStampRaw(&fileTimeStampRaw, reinterpret_cast<const char*>(pinPath));

    if (result.IsSuccess())
    {
        createTime = fileTimeStampRaw.create;
        modifyTime = fileTimeStampRaw.modify;
        accessTime = fileTimeStampRaw.access;
    }

    pinPath = nullptr;

    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

int32_t FileSystem::DoCommit()
{
    auto result = m_pImpl->DoCommit();
    return ReturnAndDeclareAlive(this, result.GetInnerValueForDebug());
}

}}}
