﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;
using System.Runtime.InteropServices;

using Nintendo.HtcTools.HtcfsLibrary;

namespace Nintendo.HtcTools
{
    public class HtcfsService : IDisposable
    {
        private const int MaxFileHandleCount = 512;
        private const int MaxDirectoryHandleCount = 64;

        // Read, Write の際 htclow に対して与える送受信バッファの最大値
        private const long MaxHtclowBufferSize = int.MaxValue;

        private HandleAllocator fileHandleAllocator = new HandleAllocator(MaxFileHandleCount);
        private HandleAllocator directoryHandleAllocator = new HandleAllocator(MaxDirectoryHandleCount);

        private HtcfsNativeLibrary.FileSystem fileSystem = new HtcfsNativeLibrary.FileSystem();
        private DirectoryEntryTable<HtcfsNativeLibrary.File> fileTable = new DirectoryEntryTable<HtcfsNativeLibrary.File>();
        private DirectoryEntryTable<HtcfsNativeLibrary.Directory> directoryTable = new DirectoryEntryTable<HtcfsNativeLibrary.Directory>();

        private HeaderFactory factory = new HeaderFactory();

        private Stream stream;

        private Task serviceTask;

        public HtcfsService(HtclowClient htclowClient)
            : this(htclowClient.OpenChannelStream(Constants.HtclowModuleId, Constants.HtclowChannelId))
        {
        }

        internal HtcfsService(Stream stream)
        {
            this.stream = stream;
            StartServiceTask();
        }

        #region Dispose
        private void Dispose(bool disposing)
        {
            if (disposing)
            {
                // TORIAEZU: Stream を Close することで serviceTask を終了させる
                // TODO: Htclow を CancellationToken に対応させたうえで、真面目なキャンセルを実装
                stream.Close();
                stream = null;

                serviceTask.Wait();
                serviceTask.Dispose();
                serviceTask = null;

                directoryTable.Dispose();
                fileTable.Dispose();
                fileSystem.Dispose();
            }
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }
        #endregion

        public void WaitServiceTask()
        {
            serviceTask.Wait();
        }

        private void StartServiceTask()
        {
            serviceTask = new Task(ServiceTaskBody, TaskCreationOptions.LongRunning);
            serviceTask.Start();
        }

        private void ServiceTaskBody()
        {
            try
            {
                while (true)
                {
                    // ヘッダ受信
                    var buffer = new byte[Header.Size];

                    var readSize = stream.Read(buffer, 0, Header.Size);

                    if (readSize != Header.Size)
                    {
                        throw new HtcfsException("Failed to receive header.");
                    }

                    var header = new Header(buffer);

                    if (header.Protocol != Constants.Protocol)
                    {
                        throw new HtcfsException($"Invalid protocol. ({header.Protocol}).");
                    }
                    if (header.Version > Constants.MaxVersion)
                    {
                        throw new HtcfsException($"Invalid version. (version={header.Version}, MaxVersion={Constants.MaxVersion})");
                    }
                    if (header.PacketCategory != PacketCategory.Request)
                    {
                        throw new HtcfsException($"Invalid packet category. ({header}).");
                    }

                    switch (header.PacketType)
                    {
                        case PacketType.GetMaxProtocolVersion:
                            ProcessGetMaxProtocolVersion(header);
                            break;
                        case PacketType.SetProtocolVersion:
                            ProcessSetProtocolVersion(header);
                            break;
                        case PacketType.GetEntryType:
                            ProcessGetEntryType(header);
                            break;
                        case PacketType.GetPriorityForFile:
                            ProcessGetPriorityForFile(header);
                            break;
                        case PacketType.SetPriorityForFile:
                            ProcessSetPriorityForFile(header);
                            break;
                        case PacketType.OpenFile:
                            ProcessOpenFile(header);
                            break;
                        case PacketType.CloseFile:
                            ProcessCloseFile(header);
                            break;
                        case PacketType.CreateFile:
                            ProcessCreateFile(header);
                            break;
                        case PacketType.DeleteFile:
                            ProcessDeleteFile(header);
                            break;
                        case PacketType.RenameFile:
                            ProcessRenameFile(header);
                            break;
                        case PacketType.FileExists:
                            ProcessFileExists(header);
                            break;
                        case PacketType.ReadFile:
                            ProcessReadFile(header);
                            break;
                        case PacketType.WriteFile:
                            ProcessWriteFile(header);
                            break;
                        case PacketType.FlushFile:
                            ProcessFlushFile(header);
                            break;
                        case PacketType.GetFileTimeStamp:
                            ProcessGetFileTimeStamp(header);
                            break;
                        case PacketType.GetFileSize:
                            ProcessGetFileSize(header);
                            break;
                        case PacketType.SetFileSize:
                            ProcessSetFileSize(header);
                            break;
                        case PacketType.GetPriorityForDirectory:
                            ProcessGetPriorityForDirectory(header);
                            break;
                        case PacketType.SetPriorityForDirectory:
                            ProcessSetPriorityForDirectory(header);
                            break;
                        case PacketType.OpenDirectory:
                            ProcessOpenDirectory(header);
                            break;
                        case PacketType.CloseDirectory:
                            ProcessCloseDirectory(header);
                            break;
                        case PacketType.CreateDirectory:
                            ProcessCreateDirectory(header);
                            break;
                        case PacketType.DeleteDirectory:
                            ProcessDeleteDirectory(header);
                            break;
                        case PacketType.RenameDirectory:
                            ProcessRenameDirectory(header);
                            break;
                        case PacketType.DirectoryExists:
                            ProcessDirectoryExists(header);
                            break;
                        case PacketType.ReadDirectory:
                            ProcessReadDirectory(header);
                            break;
                        case PacketType.GetEntryCount:
                            ProcessGetEntryCount(header);
                            break;
                        default:
                            throw new HtcfsException($"Unknown packet type. ({header})");
                    }
                }
            }
            catch (Exception e) when (e is HtclowException || e is HtcfsException || e is ObjectDisposedException)
            {
                // ObjectDisposedException は NetworkEmulationStream 用
                Console.WriteLine(e);
            }
            finally
            {
                Console.WriteLine("HostFsService task finished.");
            }
        }

        private void ProcessGetMaxProtocolVersion(Header request)
        {
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, Constants.MaxVersion).SendTo(stream);
        }

        private void ProcessSetProtocolVersion(Header request)
        {
            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            var version = request.Param0;

            if (version < 0)
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidRequest).SendTo(stream);
                return;
            }

            if (version > Constants.MaxVersion)
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.UnsupportedVersion).SendTo(stream);
                return;
            }

            // プロトコルバージョン設定
            factory.Version = (short)request.Param0;

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success).SendTo(stream);
        }

        private void ProcessGetEntryType(Header header)
        {
            // リクエストボディ受信
            var path = Util.ReceivePathFrom(stream, header.BodySize);

            // メイン処理
            var nativeResult = fileSystem.DoGetEntryType(out var entryType, path);

            // レスポンス送信
            factory.MakeResponseHeader(header.PacketType, HtcfsResult.Success, nativeResult, entryType).SendTo(stream);
        }

        private void ProcessGetPriorityForFile(Header request)
        {
            var handle = (int)request.Param0;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!fileTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }
            var priority = fileTable.GetPriority(handle);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, priority).SendTo(stream);
        }

        private void ProcessSetPriorityForFile(Header request)
        {
            var handle = (int)request.Param0;
            var priority = (int)request.Param1;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!fileTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }
            fileTable.SetPriority(handle, priority);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success).SendTo(stream);
        }

        private void ProcessOpenFile(Header request)
        {
            var mode = (int)request.Param0;

            // リクエストボディ受信
            var path = Util.ReceivePathFrom(stream, request.BodySize);

            // メイン処理
            int handle;
            try
            {
                handle = fileHandleAllocator.Allocate();
            }
            catch (OutOfHandleException)
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.OutOfHandle).SendTo(stream);
                return;
            }

            var nativeResult = fileSystem.DoOpenFile(out var file, path, mode);

            if (Util.IsNativeResultSuccess(nativeResult))
            {
                fileTable.Add(handle, file);
            }
            else
            {
                fileHandleAllocator.Free(handle);
            }

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult, handle).SendTo(stream);
        }

        private void ProcessCloseFile(Header request)
        {
            var handle = (int)request.Param0;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!fileTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }

            fileTable.Remove(handle);

            fileHandleAllocator.Free(handle);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success).SendTo(stream);
        }

        private void ProcessCreateFile(Header request)
        {
            var fileSize = request.Param0;

            // リクエストボディ受信
            var path = Util.ReceivePathFrom(stream, request.BodySize);

            // メイン処理
            if (fileSize < 0)
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidRequest).SendTo(stream);
                return;
            }

            var nativeResult = fileSystem.DoCreateFile(path, fileSize);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
        }

        private void ProcessDeleteFile(Header request)
        {
            // リクエストボディ受信
            var path = Util.ReceivePathFrom(stream, request.BodySize);

            // メイン処理
            var nativeResult = fileSystem.DoDeleteFile(path);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
        }

        private void ProcessRenameFile(Header request)
        {
            var fromPathSize = (int)request.Param0;
            var toPathSize = (int)request.Param1;

            // リクエストボディ受信
            var fromPath = Util.ReceivePathFrom(stream, fromPathSize);
            var toPath = Util.ReceivePathFrom(stream, toPathSize);

            // メイン処理
            var nativeResult = fileSystem.DoRenameFile(fromPath, toPath);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
        }

        private void ProcessFileExists(Header request)
        {
            // リクエスト検証
            if (request.BodySize > Util.MaxPathLength)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // リクエストボディ受信
            var buffer = new byte[request.BodySize];

            var readSize = stream.Read(buffer, 0, buffer.Length);
            if (readSize != buffer.Length)
            {
                throw new HtcfsException("Failed to read body");
            }

            var path = Encoding.UTF8.GetString(buffer);

            // メイン処理
            var info = new FileInfo(path);
            var exist = info.Exists ? 1 : 0;

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, exist).SendTo(stream);
        }

        private void ProcessReadFile(Header request)
        {
            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            var handle = (int)request.Param0;
            var offset = request.Param1;
            var size = request.Param2;

            if (!fileTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }

            // メイン処理
            // TORIAEZU: fs_HostFileSystem との互換性を保つため、読み出しサイズと同じサイズのバッファを用意する
            var buffer = new byte[size];

            var nativeResult = fileTable[handle].Read(out var actualSize, offset, buffer, size);

            // レスポンス送信
            if (Util.IsNativeResultSuccess(nativeResult))
            {
                factory.MakeResponseHeader(request.PacketType, actualSize, HtcfsResult.Success, nativeResult).SendTo(stream);

                long totalSendSize = 0;
                while (totalSendSize < actualSize)
                {
                    var sendSize = (int)Math.Min(actualSize - totalSendSize, MaxHtclowBufferSize);
                    var sendBuffer = new byte[sendSize];

                    // 2G を超えるオフセットを扱うため unsafe を使用
                    unsafe
                    {
                        fixed (byte* source = &buffer[totalSendSize])
                        {
                            Marshal.Copy(new IntPtr(source), sendBuffer, 0, sendSize);
                        }
                    }

                    stream.Write(sendBuffer, 0, sendBuffer.Length);

                    totalSendSize += sendSize;
                }
            }
            else
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
            }
        }

        private void ProcessWriteFile(Header request)
        {
            var size = request.BodySize;
            var handle = (int)request.Param0;
            var option = (int)request.Param1;
            var offset = request.Param2;

            // メイン処理
            // TORIAEZU: fs_HostFileSystem との互換性を保つため、書き込みサイズと同じサイズのバッファを用意する
            var buffer = new byte[size];

            long totalReceiveSize = 0;
            while (totalReceiveSize < size)
            {
                var receiveSize = (int)Math.Min(size - totalReceiveSize, MaxHtclowBufferSize);
                var receiveBuffer = new byte[receiveSize];

                var actualReceivedSize = stream.Read(receiveBuffer, 0, receiveBuffer.Length);

                if (receiveBuffer.Length != actualReceivedSize)
                {
                    throw new HtcfsException("Connection closed while receiving write data.");
                }

                // 2G を超えるオフセットを扱うため unsafe を使用
                unsafe
                {
                    fixed (byte* destination = &buffer[totalReceiveSize])
                    {
                        Marshal.Copy(receiveBuffer, 0, new IntPtr(destination), receiveSize);
                    }
                }

                totalReceiveSize += receiveSize;
            }

            if (!fileTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }

            var nativeResult = fileTable[handle].Write(offset, buffer, size, option);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
        }

        private void ProcessFlushFile(Header request)
        {
            var handle = (int)request.Param0;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!fileTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }

            var nativeResult = fileTable[handle].Flush();

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
        }

        private void ProcessGetFileTimeStamp(Header request)
        {
            // リクエストボディ受信
            var path = Util.ReceivePathFrom(stream, request.BodySize);

            // メイン処理
            var nativeResult = fileSystem.DoGetFileTimeStampRaw(out var createTime, out var modifyTime, out var accessTime, path);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult, createTime, accessTime, modifyTime).SendTo(stream);
        }

        private void ProcessGetFileSize(Header request)
        {
            var handle = (int)request.Param0;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!fileTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }

            var nativeResult = fileTable[handle].GetSize(out var fileSize);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult, fileSize).SendTo(stream);
        }

        private void ProcessSetFileSize(Header request)
        {
            var handle = (int)request.Param0;
            var fileSize = request.Param1;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            if (fileSize < 0)
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidRequest).SendTo(stream);
                return;
            }

            // メイン処理
            if (!fileTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }

            var nativeResult = fileTable[handle].SetSize(fileSize);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
        }

        private void ProcessGetPriorityForDirectory(Header request)
        {
            var handle = (int)request.Param0;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!directoryTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }
            var priority = directoryTable.GetPriority(handle);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, priority).SendTo(stream);
        }

        private void ProcessSetPriorityForDirectory(Header request)
        {
            var handle = (int)request.Param0;
            var priority = (int)request.Param1;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!directoryTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }
            directoryTable.SetPriority(handle, priority);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success).SendTo(stream);
        }

        private void ProcessOpenDirectory(Header request)
        {
            var mode = (int)request.Param0;

            // リクエストボディ受信
            var path = Util.ReceivePathFrom(stream, request.BodySize);

            // メイン処理
            int handle;
            try
            {
                handle = directoryHandleAllocator.Allocate();
            }
            catch (OutOfHandleException)
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.OutOfHandle).SendTo(stream);
                return;
            }

            var nativeResult = fileSystem.DoOpenDirectory(out var directory, path, mode);

            if (Util.IsNativeResultSuccess(nativeResult))
            {
                directoryTable.Add(handle, directory);
            }
            else
            {
                directoryHandleAllocator.Free(handle);
            }

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult, handle).SendTo(stream);
        }

        private void ProcessCloseDirectory(Header request)
        {
            var handle = (int)request.Param0;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!directoryTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }

            directoryTable.Remove(handle);

            directoryHandleAllocator.Free(handle);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success).SendTo(stream);
        }

        private void ProcessCreateDirectory(Header request)
        {
            // リクエストボディ受信
            var path = Util.ReceivePathFrom(stream, request.BodySize);

            // メイン処理
            var nativeResult = fileSystem.DoCreateDirectory(path);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
        }

        private void ProcessDeleteDirectory(Header request)
        {
            var recursively = request.Param0 != 0 ? true : false;

            // リクエストボディ受信
            var path = Util.ReceivePathFrom(stream, request.BodySize);

            // メイン処理
            int nativeResult;

            if (recursively)
            {
                nativeResult = fileSystem.DoDeleteDirectoryRecursively(path);
            }
            else
            {
                nativeResult = fileSystem.DoDeleteDirectory(path);
            }

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
        }

        private void ProcessRenameDirectory(Header request)
        {
            var fromPathSize = (int)request.Param0;
            var toPathSize = (int)request.Param1;

            // リクエストボディ受信
            var fromPath = Util.ReceivePathFrom(stream, fromPathSize);
            var toPath = Util.ReceivePathFrom(stream, toPathSize);

            // メイン処理
            var nativeResult = fileSystem.DoRenameDirectory(fromPath, toPath);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
        }

        private void ProcessDirectoryExists(Header request)
        {
            // リクエスト検証
            if (request.BodySize > Util.MaxPathLength)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // リクエストボディ受信
            var buffer = new byte[request.BodySize];

            var readSize = stream.Read(buffer, 0, buffer.Length);
            if (readSize != buffer.Length)
            {
                throw new HtcfsException("Failed to read body");
            }

            var path = Encoding.UTF8.GetString(buffer);

            // DirectoryInfo チェック
            var info = new DirectoryInfo(path);
            var exist = info.Exists ? 1 : 0;

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, exist).SendTo(stream);
        }

        private void ProcessReadDirectory(Header request)
        {
            var handle = (int)request.Param0;
            var entryBufferCount = request.Param1;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!directoryTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }

            // entryBufferCount が int の最大値を超える場合には対応しない
            if (entryBufferCount < 0 || entryBufferCount > int.MaxValue)
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidRequest).SendTo(stream);
                return;
            }

            var directoryEntrySize = HtcfsNativeLibrary.Directory.GetDirectoryEntrySize();
            var bufferSize = directoryEntrySize * entryBufferCount;
            var buffer = new byte[bufferSize];

            var nativeResult = directoryTable[handle].Read(out var entryCount, buffer, bufferSize);

            // レスポンス送信
            if (Util.IsNativeResultSuccess(nativeResult))
            {
                factory.MakeResponseHeader(request.PacketType, directoryEntrySize * entryCount, HtcfsResult.Success, nativeResult, entryCount).SendTo(stream);

                if (entryCount > 0)
                {
                    stream.Write(buffer, 0, directoryEntrySize * (int)entryCount);
                }
            }
            else
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult).SendTo(stream);
            }
        }

        private void ProcessGetEntryCount(Header request)
        {
            var handle = (int)request.Param0;

            // リクエスト検証
            if (request.BodySize != 0)
            {
                throw new HtcfsException($"Request body size ({request.BodySize}) is incorrect.");
            }

            // メイン処理
            if (!directoryTable.Handles.Contains(handle))
            {
                factory.MakeResponseHeader(request.PacketType, HtcfsResult.InvalidHandle).SendTo(stream);
                return;
            }

            var nativeResult = directoryTable[handle].GetEntryCount(out var count);

            // レスポンス送信
            factory.MakeResponseHeader(request.PacketType, HtcfsResult.Success, nativeResult, count).SendTo(stream);
        }
    }
}
