﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;
using System.Net;
using System.Net.Sockets;

namespace Nintendo.HtcTools.HtcTmProxy
{
    internal class HtcTmProxy : IDisposable
    {
        private const byte HtclowModuleId = 0;
        private const ushort HtclowChannelId = 0;

        private const int TmAutoConnectPort = 17111;
        private const int ProxyPort = 20181;

        private const int BufferSize = 1024;

        private const int udpPollIntervalMs = 500;

        private readonly Task m_TargetToHostTask;
        private readonly Task m_HostToTargetTask;

        private readonly TcpListener m_HostListener;

        private readonly HtclowChannel m_HtclowChannel;

        private TcpClient m_HostTcpClient;
        private NetworkStream m_HostStream;

        public HtcTmProxy(HtclowClient htclowClient)
        {
            m_HostToTargetTask = new Task(HostToTargetTaskBody, TaskCreationOptions.LongRunning);
            m_TargetToHostTask = new Task(TargetToHostTaskBody, TaskCreationOptions.LongRunning);

            m_HostListener = new TcpListener(IPAddress.Loopback, ProxyPort);

            m_HtclowChannel = htclowClient.OpenChannel(HtclowModuleId, HtclowChannelId);
        }

        public void Start()
        {
            // Connect to target htclow channel
            m_HtclowChannel.Connect();

            // TCP server for TM
            m_HostListener.Server.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.ReuseAddress, true);
            m_HostListener.Start();

            Task.Run(() =>
            {
                while (true)
                {
                    // Send UDP packet for auto connect
                    var autoConnectPacket = new TmAutoConnectHeader(ProxyPort);

                    var udpClient = new UdpClient();
                    udpClient.Connect(IPAddress.Loopback, TmAutoConnectPort);
                    udpClient.Send(autoConnectPacket.GetBytes(), autoConnectPacket.GetLength());

                    // Accept
                    if (m_HostListener.Pending())
                    {
                        break;
                    }

                    Task.Delay(udpPollIntervalMs).Wait();
                }
            }).Wait();

            m_HostTcpClient = m_HostListener.AcceptTcpClient();
            m_HostTcpClient.NoDelay = true;
            m_HostStream = m_HostTcpClient.GetStream();

            // Start proxy tasks
            m_HostToTargetTask.Start();
            m_TargetToHostTask.Start();
        }

        public void Wait()
        {
            m_TargetToHostTask.Wait();
            m_HostToTargetTask.Wait();
        }

        public void Dispose()
        {
            m_HostStream?.Close();
            m_HostTcpClient?.Close();

            m_HostListener.Stop();

            m_HtclowChannel.Dispose();

            m_TargetToHostTask.Dispose();
            m_HostToTargetTask.Dispose();
        }

        // ホストからターゲットへのデータ転送を行う
        private void HostToTargetTaskBody()
        {
            using (var htclowWriter = new HtclowWriter(m_HtclowChannel, true))
            using (var hostReader = new BinaryReader(m_HostStream))
            {
                try
                {
                    // Read version from TargetManager.exe
                    var version = hostReader.ReadInt32();

                    // Send version to Target
                    htclowWriter.Write(version);

                    while (true)
                    {
                        int readSize;

                        var header = new TmPacketHeader();

                        // Read packet header from TargetManager.exe
                        readSize = m_HostStream.Read(header.Buffer, 0, header.Buffer.Length);
                        if (readSize != header.Buffer.Length)
                        {
                            throw new IOException("Failed to read packet header from TM.");
                        }

                        // Send packet header to Target
                        htclowWriter.Write(header.Buffer);

                        if (header.GetDataLength() > 0)
                        {
                            var body = new byte[header.GetDataLength()];

                            // Read packet body from TargetManager.exe
                            readSize = m_HostStream.Read(body, 0, body.Length);
                            if (readSize != body.Length)
                            {
                                throw new IOException("Failed to read packet header from TM.");
                            }

                            // Send packet body to Target
                            htclowWriter.Write(body);
                        }
                    }
                }
                catch
                {
                    m_HtclowChannel.Shutdown();
                    m_HostStream.Close();
                }
            }
        }

        // ターゲットからホストへのデータ転送を行う
        private void TargetToHostTaskBody()
        {
            using (var htclowReader = new HtclowReader(m_HtclowChannel, true))
            {
                try
                {
                    while (true)
                    {
                        // サイズを受信
                        var size = htclowReader.ReadInt32();

                        // データを受信
                        var data = htclowReader.Read(size);
                        m_HostStream.Write(data, 0, size);
                    }
                }
                catch
                {
                    m_HtclowChannel.Shutdown();
                    m_HostStream.Close();
                }
            }
        }
    }
}
