﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Threading.Tasks;

namespace Nintendo.FsAccessLogAnalysis
{
    public class FsAccessLogList
    {
        internal List<FsAccessLog> List { get; private set; } = new List<FsAccessLog>();
        internal FsAccessLog Begin { get; private set; } = null;
        internal FsAccessLog End { get; private set; } = null;

        public int Count
        {
            get { return List.Count; }
        }

        public void Add(FsAccessLog log)
        {
            if ((Begin == null) || (Begin.Start > log.Start))
            {
                Begin = log;
            }
            if ((End == null) || (End.End < log.End))
            {
                End = log;
            }
            List.Add(log);
        }

        public bool Any()
        {
            return List.Any();
        }

        // relativeMilliseconds 以降のログリストを取得
        public void SubList(long relativeMilliseconds)
        {
            if (relativeMilliseconds <= 0)
            {
                return;
            }
            long baseTime = Begin.Start;
            long branchTime = baseTime + relativeMilliseconds;
            if (branchTime > End.End)
            {
                Clear();
                return;
            }
            FsAccessLog target = new FsAccessLog();
            target.Start = branchTime;
            target.End = branchTime;

            int branch = List.BinarySearchEqualOrRight(target);
            int count = List.Count;
            if (branch >= count)
            {
                Clear();
                return;
            }
            List = List.GetRange(branch, List.Count - branch);
            Begin = List.Min();
        }

        public long GetDistanceMilliseconds(FsAccessLog log)
        {
            if (Any())
            {
                return FsAccessLogAnalyzer.GetDistanceMilliseconds(End, log);
            }
            return 0;
        }

        public long GetTotalMilliseconds()
        {
            if (Any())
            {
#if DEBUG
                Debug.Assert(Begin.Start == List.Min(log => log.Start), "not match the cached log");
                Debug.Assert(End.End == List.Max(log => log.End), "not match the cached log");
#endif
                return End.End - Begin.Start;
            }
            return 0;
        }

        public void Clear()
        {
            List.Clear();
            Begin = null;
            End = null;
        }
    }

    public class FsAccessLogAnalyzer
    {
        public static readonly long AllowRewindMilliseconds = 10 * 1000;

        public FsAccessLogAnalyzer()
        {
            IsCompleteLog = true;
        }

        public bool IsEmpty() { return !LogList.Any(); }

        public bool HasSdkVersion()
        {
            return !string.IsNullOrEmpty(LogSdkVersion);
        }

        public bool IsSpecNX()
        {
            return LogSpec == "NX";
        }

        public bool IsExpectedSdkVersion(Version expectVersion)
        {
            if (HasSdkVersion())
            {
                try
                {
                    Version logSdkVersion = new Version(LogSdkVersion);
                    if (logSdkVersion >= expectVersion)
                    {
                        return true;
                    }
                }
                catch
                {
                    return false;
                }
            }
            return false;
        }

        public bool ForSystem { get; private set; } = false;

        public bool HasSystemAccess()
        {
            return LogList.Any(log => log.MountForSystem);
        }

        public List<FsAccessLog> FilterByMountTarget(List<FsAccessLog> list, FsMountTarget target)
        {
            return list.FindAll(log => { return HandleInfo.GetMountTarget(log) == target; });
        }

        public Dictionary<FsMountTarget, List<FsAccessLog>> SplitWithMountTarget(FsAccessLogList list)
        {
            return SplitWithMountTarget(list, (FsMountTarget[])Enum.GetValues(typeof(FsMountTarget)));
        }

        public Dictionary<FsMountTarget, List<FsAccessLog>> SplitWithMountTarget(FsAccessLogList list, FsMountTarget[] targets)
        {
            Dictionary<FsMountTarget, List<FsAccessLog>> dst = new Dictionary<FsMountTarget, List<FsAccessLog>>();
            foreach (FsMountTarget target in targets)
            {
                if (dst.ContainsKey(target))
                {
                    continue;
                }
                var splitList = FilterByMountTarget(list.List, target);
                if (splitList.Any())
                {
                    dst.Add(target, splitList);
                }
            }
            return dst;
        }

        public bool HasAnyAccess(List<FsAccessLog> list, FsMountTarget target)
        {
            return list.Any(log => HandleInfo.GetMountTarget(log) == target);
        }
        public bool HasAnyAccess(FsMountTarget target)
        {
            return HasAnyAccess(LogList, target);
        }

        public int GetMountCount(List<FsAccessLog> list)
        {
            return list.Where(log => FsFunction.IsMountFunction(log.GetFunctionFullName())).Count();
        }

        public long GetTotalMilliseconds()
        {
            return GetTotalMilliseconds(LogList);
        }

        public static long GetDistanceMilliseconds(FsAccessLog before, FsAccessLog after)
        {
            if (after.Start <= before.End)
            {
                return 0;
            }
            return after.Start - before.End;
        }
        public static long GetStartPointElapsedMilliseconds(FsAccessLog before, FsAccessLog after)
        {
            return Math.Abs(after.Start - before.Start);
        }
        public static long GetStartPointElapsedMilliseconds(List<FsAccessLog> list)
        {
            if (!list.Any())
            {
                return 0;
            }
            long start = list.Min(log => log.Start);
            long end = list.Max(log => log.Start);
            return end - start;
        }
        public static long GetTotalMilliseconds(List<FsAccessLog> list)
        {
            if (!list.Any())
            {
                return 0;
            }
            long start = list.Min(log => log.Start);
            long end = list.Max(log => log.End);
            return end - start;
        }

        public static ulong GetTotalWriteSize(List<FsAccessLog> list)
        {
            return list.Select(log => log.ActuallyWrittenSize).Aggregate((x, y) => x + y);
        }
        public static ulong GetTotalReadSize(List<FsAccessLog> list)
        {
            return list.Select(log => log.ActuallyReadedSize).Aggregate((x, y) => x + y);
        }

        public static bool HasWriteAccess(List<FsAccessLog> list)
        {
            return list.Any(log => log.AccessType.HasFlag(FsApiAccessType.Write));
        }
        public static bool HasReadAccess(List<FsAccessLog> list)
        {
            return list.Any(log => log.AccessType.HasFlag(FsApiAccessType.Read));
        }

        public List<FsAccessLog> LogList
        {
            get;
            set;
        } = new List<FsAccessLog>();

        public List<FsAccessLog> ErrorLogList { get; set; } = new List<FsAccessLog>();

        public HandleLogInfo HandleInfo
        {
            get;
        } = new HandleLogInfo();

        public string LogSpec { get; private set; }
        public string LogSdkVersion { get; set; }
        public bool IsCompleteLog { get; set; }

        internal void CheckPrecondition(FsAccessLog log)
        {
            if (LogList.Any())
            {
                // 前のログよりも古いログは無効
                FsAccessLog last = LogList.Last();
                if (!log.IsStartLog() && !log.IsOutputStateLog() && (last.End > log.End))
                {
                    // 期間が重なる場合は（スレッドを考慮して）許容する
                    var compare = new MillisecondsPeriod.OverlapCompare();
                    if (compare.Compare(last.Period, log.Period) != 0)
                    {
                        // 期間の差が許容範囲内かどうか
                        if (GetDistanceMilliseconds(log, last) >= AllowRewindMilliseconds)
                        {
                            log.AnalyzerError = "Invalid timestamp order.";
                        }
                    }
                }
            }
        }

        internal void RecordLog(FsAccessLog log)
        {
            if (!log.HasLogError())
            {
                try
                {
                    RecordLogImpl(log);
                }
                catch (AnalyzerError e)
                {
                    log.AnalyzerError = e.Message;
                }
                catch (Exception e)
                {
                    Debug.Print(e.Message);
                    log.AnalyzerError = "Failed analyze of access log.";
                }
            }
        }

        private void RecordLogImpl(FsAccessLog log)
        {
            CheckPrecondition(log);

            string functionFullName = log.GetFunctionFullName();
            if (FsFunction.IsWrite(functionFullName))
            {
                log.AccessType |= FsApiAccessType.Write;
            }
            if (FsFunction.IsRead(functionFullName))
            {
                log.AccessType |= FsApiAccessType.Read;
            }
            if (log.Spec != null)
            {
                LogSpec = log.Spec;
            }
            if (log.SdkVersion != null)
            {
                LogSdkVersion = log.SdkVersion;
            }
            if (log.IsStartLog() && log.ForSystem)
            {
                ForSystem = true;
            }
            if (log.IsAnalyzable())
            {
                HandleInfo.RecordLog(log);
                LogList.Add(log);
            }
        }
    }

    internal class AnalyzerError : Exception
    {
        public AnalyzerError(string message)
            : base(message)
        {
        }
    }
}
