﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Runtime.InteropServices;
using System.Diagnostics;

namespace ErrorMessageDatabaseConverter
{
    public static class Util
    {
        /// <summary>
        /// directoryPath 以下のファイルを再帰的に列挙します。
        /// </summary>
        /// <param name="directoryPath">ファイルを列挙するディレクトリ</param>
        /// <returns>diretrocyPath 以下のファイルのフルパスのリスト</returns>
        public static List<string> ListFilesRecursively(string directoryPath)
        {
            var files = new List<string>();
            var directoryInfo = new DirectoryInfo(directoryPath);
            foreach (var d in directoryInfo.EnumerateDirectories())
            {
                files.AddRange(ListFilesRecursively(d.FullName));
            }
            foreach (var f in directoryInfo.EnumerateFiles())
            {
                files.Add(f.FullName);
            }
            return files;
        }

        public static Type DeserializeXml<Type>(string xmlFilePath) where Type : class
        {
            try
            {
                string xmlText = System.IO.File.ReadAllText(xmlFilePath, Encoding.UTF8);
                System.Xml.Serialization.XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(typeof(Type));
                string cleanXml = Regex.Replace(xmlText, @"<[a-zA-Z].[^(><.)]+/>", new MatchEvaluator(m => { return ""; }));
                MemoryStream memoryStream = new MemoryStream((new UTF8Encoding()).GetBytes(cleanXml));
                return (Type)serializer.Deserialize(memoryStream);
            }
            catch (Exception ex) when (ex is FileNotFoundException || ex is DirectoryNotFoundException)
            {
                Util.PrintError($"指定された XMLファイル（{xmlFilePath}）が見つかりませんでした。");
                return null;
            }
            catch (Exception ex)
            {
                Util.PrintError($"指定された XMLファイル（{xmlFilePath}）の読み込みに失敗しました。{ex.ToString()}");
                return null;
            }
        }

        public static void PrintError(string msg)
        {
            Console.WriteLine("[エラー] : " + msg);
        }

        public static void PrintWarning(string msg)
        {
            Console.WriteLine("[警告] : " + msg);
        }

        public class ProgressPrinter
        {
            private DateTime LastPrintTime;
            private int MaxCount;
            private int PrintIntervalMilliSec;

            public ProgressPrinter(int max, int interval = 200)
            {
                MaxCount = max;
                PrintIntervalMilliSec = interval;
                LastPrintTime = DateTime.Now;
            }

            public void PrintProgress(int currentCount)
            {
                var ms = (DateTime.Now - LastPrintTime).TotalMilliseconds;
                if (ms > PrintIntervalMilliSec)
                {
                    Console.WriteLine($"{100.0 * currentCount / MaxCount,5:0.0}% ({currentCount}/{MaxCount})");
                    LastPrintTime = DateTime.Now;
                }
            }
        }
    }
}
