﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/crypto/crypto_Sha256Generator.h>

#include "CrashReport.h"
#include "CrashReportRaw.h"

namespace nn { namespace creport {

namespace {

    const Bit64 StackTraceEndMarker = 0x5A5A5A5AF0F0F0F0ull;
    const Bit64 CodeRegionEndMarker = 0xF0F0F0F0A5A5A5A5ull;

}

// 復号化したデータを解析 for v1 (2.0.0 NUP)
void CrashReport::ParseRawDataV1(CrashReportRawDataV1& raw)
{
    exceptionCode     = raw.exceptionCode;
    exceptionAddress  = raw.exceptionAddress;
    codeSpecificData  = raw.codeSpecificData;
    exceptionThreadId = raw.threadId;

    ThreadInfo ti;

    memcpy(&ti.reg, &raw.reg, sizeof(ti.reg));
    ti.threadId = raw.threadId;
    ti.fp = raw.fp;
    ti.lr = raw.lr;
    ti.sp = raw.sp;
    ti.pc = raw.pc;
    ti.pstate = static_cast<nn::Bit32>(raw.pstate);

    for (int i = 0; i < 32; i++)
    {
        if (raw.stackTrace[i] == 0 || raw.stackTrace[i] == 4)
        {
            break;
        }
        ti.stackTrace.push_back(raw.stackTrace[i]);
    }

    thread.push_back(ti);
}

// 復号化したデータを解析 for v2 (2.1.0 NUP)
void CrashReport::ParseRawDataV2(CrashReportRawDataV2& raw)
{
    exceptionCode     = raw.exceptionCode;
    exceptionAddress  = raw.exceptionAddress;
    codeSpecificData  = raw.codeSpecificData;
    exceptionThreadId = raw.threadId;

    // 例外を起こしたスレッド情報
    {
        ThreadInfo ti;

        memcpy(&ti.reg, &raw.reg, sizeof(ti.reg));
        ti.threadId = raw.threadId;
        ti.fp = raw.fp;
        ti.lr = raw.lr;
        ti.sp = raw.sp;
        ti.pc = raw.pc;
        ti.pstate = static_cast<nn::Bit32>(raw.pstate);

        for (int i = 0; i < 32; i++)
        {
            if (raw.variadic[i] == StackTraceEndMarker || raw.variadic[i] == 0 || raw.variadic[i] == 4)
            {
                break;
            }
            ti.stackTrace.push_back(raw.variadic[i]);
        }

        thread.push_back(ti);
    }

    // モジュール情報の先頭を探す
    int pos = 0;
    while (pos < NN_ARRAY_SIZE(raw.variadic) && raw.variadic[pos] != StackTraceEndMarker)
    {
        pos++;
    }
    pos += 3;

    if (pos >= NN_ARRAY_SIZE(raw.variadic))
    {
        // 32bit アプリの場合スタックトレースが残らないので終端が見つからない（終端だけ残すべきだった…）
        pos = 2;
    }

    // モジュール情報
    while (raw.variadic[pos] != CodeRegionEndMarker)
    {
        ModuleInfo mi;

        mi.codeBase = raw.variadic[pos++];
        mi.codeEnd  = mi.codeBase + raw.variadic[pos++];
        for (int i = 0; i < 4; i++)
        {
            *(reinterpret_cast<Bit64*>(mi.moduleName + i * sizeof(Bit64))) = raw.variadic[pos++];
        }

        if (mi.moduleName[0]=='\0')
        {
            memcpy(mi.moduleName, "noname", 7);
        }

        module.push_back(mi);
    }
}

// 復号化したデータを解析 for v3 (3.0.0 NUP)
void CrashReport::ParseRawDataV3(CrashReportRawDataV3& raw)
{
    exceptionCode = 0;

    raw.Reset();
    raw.Read(&exceptionCode, 1);
    raw.Read(&exceptionAddress, sizeof(exceptionAddress));
    raw.Read(&codeSpecificData, sizeof(codeSpecificData));

    // モジュール情報
    nn::Bit8 moduleCount;
    raw.Read(&moduleCount, sizeof(moduleCount));
    for (int i = 0; i < moduleCount; ++i)
    {
        ModuleInfo mi;
        raw.Read(mi.moduleName, sizeof(mi.moduleName));
        raw.Read(&mi.codeBase, sizeof(mi.codeBase));
        raw.Read(&mi.codeEnd, sizeof(mi.codeEnd));

        if (mi.moduleName[0] == '\0')
        {
            memcpy(mi.moduleName, "noname", 7);
        }

        module.push_back(mi);
    }

    // スレッド情報
    nn::Bit8 stackTraceCount;
    ThreadInfo ti;
    raw.Read(&ti.threadId, sizeof(ti.threadId));
    raw.Read(&ti.reg, sizeof(ti.reg));
    raw.Read(&ti.fp, sizeof(ti.fp));
    raw.Read(&ti.lr, sizeof(ti.lr));
    raw.Read(&ti.sp, sizeof(ti.sp));
    raw.Read(&ti.pc, sizeof(ti.pc));
    raw.Read(&ti.pstate, sizeof(ti.pstate));
    raw.Read(&ti.stackTop, sizeof(ti.stackTop));
    raw.Read(&ti.stackBottom, sizeof(ti.stackBottom));
    raw.Read(&stackTraceCount, sizeof(stackTraceCount));
    for (int i = 0; i < stackTraceCount; i++)
    {
        nn::Bit64 address;
        raw.Read(&address, sizeof(address));
        ti.stackTrace.push_back(address);
    }
    thread.push_back(ti);

    exceptionThreadId = ti.threadId;
}

nn::Bit64 CrashReport::GetModuleOffset(nn::Bit64 addr)
{
    for (auto&& mi : module)
    {
        if (addr >= mi.codeBase && addr < mi.codeEnd)
        {
            return addr - mi.codeBase;
        }
    }
    return addr & 0xFFF;
}

void CrashReport::ShowCodeAddress(nn::Bit64 addr)
{
    for (auto&& mi : module)
    {
        if (addr >= mi.codeBase && addr < mi.codeEnd)
        {
            NN_LOG("  0x%016llX: 0x%llx + 0x%llx (%s)\n", addr, mi.codeBase, addr - mi.codeBase, mi.moduleName);
            return;
        }
    }
    NN_LOG("  0x%016llX\n", addr);
}

void CrashReport::Dump()
{
    NN_LOG("Exception code:         0x%llx\n", exceptionCode);
    NN_LOG("Exception address:      0x%016llX\n", exceptionAddress);
    NN_LOG("Fault address register: 0x%llx\n", codeSpecificData);
    NN_LOG("Exception Thread Id:    %lld\n", exceptionThreadId);
    NN_LOG("\n");

    // モジュール情報を表示
    NN_LOG("Code region:\n");

    for (auto&& mi : module)
    {
        NN_LOG("  0x%llx - 0x%llx [%s]\n", mi.codeBase, mi.codeEnd, mi.moduleName);
    }
    NN_LOG("\n");

    // スレッド情報を表示
    for (auto&& ti : thread)
    {
        NN_LOG("----\n");
        NN_LOG("Thread Id: %lld\n", ti.threadId);
        NN_LOG("Stack    : 0x%016llX - 0x%016llX\n", ti.stackTop, ti.stackBottom);

        for (int i = 0; i < 29; i++)
        {
            NN_LOG("X%2d:  0x%016llX\n", i, ti.reg[i]);
        }
        NN_LOG("FP :  0x%016llX\n", ti.fp);
        NN_LOG("LR :");
        ShowCodeAddress(ti.lr);
        NN_LOG("SP :  0x%016llX %s\n", ti.sp, (version < 3 || ti.sp >= ti.stackTop && ti.sp <= ti.stackBottom)? "" : "(Stack overflow?)");
        NN_LOG("PC :");
        ShowCodeAddress(ti.pc);
        NN_LOG("PSTATE: 0x%08llX\n", ti.pstate);
        NN_LOG("\n");

        NN_LOG("Stack trace:\n");
        for (auto&& addr : ti.stackTrace)
        {
            ShowCodeAddress(addr);
        }
        NN_LOG("\n");
    }
}

void CrashReport::ShowStackTraceHash()
{
    Bit8  hash[32];
    Bit64 trace[128];

    std::memset(trace, 0, sizeof(trace));

    for (auto&& ti : thread)
    {
        if (ti.threadId == exceptionThreadId)
        {
            int count = 0;
            trace[count++] = GetModuleOffset(ti.pc);
            trace[count++] = GetModuleOffset(ti.lr);
            for (auto&& addr : ti.stackTrace)
            {
                trace[count++] = GetModuleOffset(addr);
            }
        }
    }

    nn::crypto::GenerateSha256Hash(hash, sizeof(hash), trace, sizeof(trace));

    NN_LOG("Hash: ");
    for (auto&& h : hash)
    {
        NN_LOG("%02X", h);
    }
    NN_LOG("\n");
}

}} // nn::creport
