﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include <cstring>

#include "OpusCodec.h"

namespace Nintendo {
namespace CodecTool {
namespace {

struct StreamInformation
{
    int totalStreamCount;
    int stereoStreamCount;
    uint8_t channelMapping[detail::ChannelCountMax]; // uint32_t のビット数分
};

const uint32_t SpeakerLocationFront = 0x1 | 0x2;
const uint32_t SpeakerLocationBack = 0x10 | 0x20;
const uint32_t SpeakerLocationFrontOfCenter = 0x40 | 0x80;
const uint32_t SpeakerLocationSide = 0x200 | 0x400;
const uint32_t SpeakerLocationTopFront = 0x1000 | 0x4000;
const uint32_t SpeakerLocationTopBack = 0x8000 | 0x20000;

const uint32_t coupledMasks[] =
{
    SpeakerLocationFront,
    SpeakerLocationBack,
    SpeakerLocationFrontOfCenter,
    SpeakerLocationSide,
    SpeakerLocationTopFront,
    SpeakerLocationTopBack,
};

uint32_t GetCoupledMask(uint32_t mask)
{
    for (const auto coupledMask : coupledMasks)
    {
        if (mask & coupledMask)
        {
            return coupledMask;
        }
    }
    return 0u;
}

struct StreamInformation ParseChannelMask(uint32_t channelMask)
{
    auto index = 0;
    auto totalStreamCount = 0;
    auto stereoStreamCount = 0;
    uint8_t stereoChannels[detail::ChannelCountMax >> 1];
    uint8_t monoChannels[detail::ChannelCountMax];

    for (auto k = 0u; channelMask != 0u; ++k)
    {
        const auto mask = 1u << k;
        const auto coupledMask = GetCoupledMask(mask);
        if (coupledMask != 0u
            && ((coupledMask & channelMask) == coupledMask))
        {
            stereoChannels[stereoStreamCount * 2] = index;
            stereoChannels[stereoStreamCount * 2 + 1] = index + 1;
            channelMask &= ~coupledMask;
            stereoStreamCount++;
            totalStreamCount++;
            index += 2;
        }
        else if (mask & channelMask)
        {
            const auto monoStreamCount = totalStreamCount - stereoStreamCount;
            monoChannels[monoStreamCount] = index;
            channelMask &= ~mask;
            totalStreamCount++;
            index += 1;
        }
    }
    const auto channelCount = index;
    const auto stereoChannelSize = stereoStreamCount * 2;
    const auto monoChannelSize = (totalStreamCount - stereoStreamCount);

    struct StreamInformation streamInfo;

    std::memcpy(streamInfo.channelMapping, stereoChannels, stereoChannelSize);
    std::memcpy(streamInfo.channelMapping + stereoChannelSize, monoChannels, monoChannelSize);
    streamInfo.totalStreamCount = totalStreamCount;
    streamInfo.stereoStreamCount = stereoStreamCount;
    return streamInfo;
}

}

OpusEncoder::OpusEncoder(int sampleRate, int channelCount, uint32_t channelMask)
    : m_pOpusEncoderImpl(nullptr)
{
    Initialize(sampleRate, channelCount, channelMask);
}

OpusEncoder::~OpusEncoder()
{
    this->!OpusEncoder();
}

OpusEncoder::!OpusEncoder()
{
    if (m_pOpusEncoderImpl)
    {
        delete m_pOpusEncoderImpl;
        m_pOpusEncoderImpl = nullptr;
    }
}

void OpusEncoder::Initialize(int sampleRate, int channelCount, uint32_t channelMask)
{
    m_pOpusEncoderImpl = new detail::OpusEncoderImpl();
    if (m_pOpusEncoderImpl == nullptr)
    {
        throw gcnew System::OutOfMemoryException();
    }
    const struct StreamInformation streamInfo = ParseChannelMask(channelMask);
    switch (m_pOpusEncoderImpl->Initialize(sampleRate, channelCount, streamInfo.totalStreamCount, streamInfo.stereoStreamCount, streamInfo.channelMapping))
    {
        case detail::OpusResult_InvalidSampleRate:
        case detail::OpusResult_InvalidChannelCount:
        {
            throw gcnew System::ArgumentException();
        }
        case detail::OpusResult_InternalError:
        {
            // TODO:
            break;
        }
        default:
        {
            break;
        }
    }
}

array<Byte>^ OpusEncoder::Encode(array<short>^ input, int sampleCount)
{
    pin_ptr<short> in = &input[0];
    const int MaxPayloadBytes = 1500;  // same upper boundary as opus_demo
    auto out = new unsigned char[MaxPayloadBytes];
    if (out == nullptr)
    {
        throw gcnew System::OutOfMemoryException();
    }

    int32_t size;
    switch (m_pOpusEncoderImpl->Encode(&size, out, MaxPayloadBytes, in, sampleCount))
    {
        case detail::OpusResult_InvalidPointer:
        case detail::OpusResult_InvalidSampleCount:
        case detail::OpusResult_InvalidSize:
            throw gcnew System::ArgumentException();
        case detail::OpusResult_UnexpectedCodingMode:
            throw gcnew UnexpectedCodingModeException();
        case detail::OpusResult_InternalError:
            throw gcnew UnexpectedInternalErrorException();
        default:
            break;
    }

    array<Byte>^ output = gcnew array<Byte>(size);
    for (int i = 0; i < size; ++i)
    {
        output[i] = out[i];
    }
    delete[] out;
    in = nullptr;

    return output;
}

OpusBasicInfo^ OpusEncoder::GetBasicInfo()
{
    auto info = gcnew OpusBasicInfo();
    info->channelCount = ChannelCount;
    info->sampleRate = SampleRate;
    return info;
}

OpusMultiStreamInfo^ OpusEncoder::GetMultiStreamInfo()
{
    const auto channelCount = this->m_pOpusEncoderImpl->GetChannelCount();
    auto info = gcnew OpusMultiStreamInfo();
    info->size = 4 + channelCount;
    info->channelCount = channelCount;
    info->totalStreamCount = this->m_pOpusEncoderImpl->GetTotalStreamCount();
    info->stereoStreamCount = this->m_pOpusEncoderImpl->GetStereoStreamCount();
    auto channelMapping = new uint8_t[channelCount];
    this->m_pOpusEncoderImpl->GetChannelMapping(channelMapping);
    info->channelMapping = gcnew array<uint8_t>(channelCount);
    for (int i = 0; i < channelCount; ++i)
    {
        info->channelMapping[i] = channelMapping[i];
    }
    delete[] channelMapping;
    return info;
}

}  // namespace CodecTool
}  // namespace Nintendo
