﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using System.IO;
using CommandUtility;
using MakeFirmwareArchive;

namespace ChangeFirmwareArchiveKey
{
    public class ChangeFirmwareArchiveKeyArguments
    {
        [CommandLineOption('o', "output", Description = "output nfa path", IsRequired = true)]
        public string OutputNspPath { get; set; }

        [CommandLineOption('i', "input", Description = "input nfa path", IsRequired = true)]
        public string InputNfaPath { get; set; }

        [CommandLineOption("keyconfig", Description = "keyconfig path", IsRequired = true)]
        public string KeyPath { get; set; }

        [CommandLineOption("result", Description = "path to save authoring result", IsRequired = true)]
        public string AuthoringResultPath { get; set; }

        [CommandLineOption('v', "verbose", Description = "output logs", IsRequired = false)]
        public bool Verbose { get; set; }
    }

    class Program
    {
        static void Main(string[] args)
        {
            ChangeFirmwareArchiveKeyArguments parameters = new ChangeFirmwareArchiveKeyArguments();
            if (CommandLineParser.Default.ParseArgs<ChangeFirmwareArchiveKeyArguments>(args, out parameters))
            {
                Log.Verbose = parameters.Verbose;
                ChangeFirmwareArchiveKey(new FileInfo(parameters.OutputNspPath),
                                         new FileInfo(parameters.InputNfaPath),
                                         new FileInfo(parameters.KeyPath),
                                         new DirectoryInfo(parameters.AuthoringResultPath));
            }
            else
            {
                return;
            }
        }

        private static void ChangeFirmwareArchiveKey(FileInfo outputNfa, FileInfo inputNfa, FileInfo keyConfig, DirectoryInfo authoringResultPath)
        {
            using (var tempHolder = new TemporaryFileHolder("ChangeFirmwareArchiveKey"))
            {
                var extractedNfa = tempHolder.CreateTemporaryDirectory("ExtractedNfa");

                var archiveDirectory = FirmwareArchiveDirectory.FromFirmwareArchive(extractedNfa, inputNfa);

                //対象のディレクトリが存在しているなら鍵を置換する
                if (archiveDirectory.SystemDirectory.Exists)
                {
                    ChangeNspKeyInPartition(archiveDirectory.GetSystemNspList(extractedNfa), archiveDirectory.SystemDirectory, keyConfig, authoringResultPath);
                }

                if (archiveDirectory.SafeDirectory.Exists)
                {
                    ChangeNspKeyInPartition(archiveDirectory.GetSafeNspList(extractedNfa), archiveDirectory.SafeDirectory, keyConfig, authoringResultPath);
                }

                archiveDirectory.ArchiveNfa(outputNfa);
            }
        }

        private static void ChangeNspKeyInPartition(List<FileInfo> nspInfos, DirectoryInfo replacePartitionName, FileInfo keyConfig, DirectoryInfo authoringResultPath)
        {
            Parallel.ForEach(
                Enumerable.Range(0, nspInfos.Count).Zip(nspInfos, (index, nsp) => Tuple.Create(index, nsp)),
                new ParallelOptions { MaxDegreeOfParallelism = 128 },
                (indexedNsp) =>
            {
                var replacedNsp = indexedNsp.Item1 + 1;
                var nsp = indexedNsp.Item2;

                Log.WriteLine("replace key ({0}/{1}), partitionName = {2}, nsp = {3}", replacedNsp, nspInfos.Count, replacePartitionName.Name, nsp.Name);

                if (nsp.Name.Contains("_prod."))
                {
                    Log.WriteLine("Skipped");
                }
                else
                {
                    //名前が _prod.nsp になる
                    ChangeSystemNspKey(replacePartitionName, nsp, keyConfig);
                    //置き換え元の nsp を削除
                    nsp.Delete();
                }
            });

            //AuthoringTool が出す結果の xml を保存する
            SaveAuthoringResultXml(replacePartitionName, authoringResultPath);

            //AuthoringTool が出す結果の xml を作業ディレクトリから消す
            RemoveFilesExceptNsp(replacePartitionName);
        }

        private static void ChangeSystemNspKey(DirectoryInfo replacedNspDir, FileInfo systemNsp, FileInfo keyConfig)
        {
            SdkTool.Execute(
               SdkPath.FindToolPath("AuthoringTool", "AuthoringTool/AuthoringTool.exe", "AuthoringTool/AuthoringTool.exe"),
                new string[] {
                    "prodencryption",
                    "--includes-cnmt",
                    "-o", replacedNspDir.FullName,
                    "--keyconfig", keyConfig.FullName,
                    systemNsp.FullName,
                });
        }

        private static void SaveAuthoringResultXml(DirectoryInfo systemDirectory, DirectoryInfo authoringResultDir)
        {
            DirectoryInfo outputDir = new DirectoryInfo(Path.Combine(authoringResultDir.FullName, systemDirectory.Name));
            if (!outputDir.Exists)
            {
                outputDir.Create();
            }
            foreach (var file in systemDirectory.EnumerateFiles())
            {
                if (file.Extension == ".xml")
                {
                    file.CopyTo(Path.Combine(outputDir.FullName, file.Name), true);
                }
            }
        }

        private static void RemoveFilesExceptNsp(DirectoryInfo systemDirectory)
        {
            foreach (var file in systemDirectory.EnumerateFiles())
            {
                if (file.Extension != ".nsp")
                {
                    file.Delete();
                }
            }
        }
    }
}
