﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Reactive.Linq;
using System.Xml.Serialization;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using Livet.Messaging.IO;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class ErrorListParamVm : ParamVm
    {
        public ReadOnlyReactiveCollection<ErrorParamVm> Errors { get; }

        public ReactiveCommand OpenAllCommand { get; }

        public ReactiveCommand CloseAllCommand { get; }

        public ReactiveCommand<string> ExportCommand { get; }

        public ErrorListParamVm(string captionTag, string commentTag, ObservableCollection<ApplicationMetaError> model) : base(captionTag, commentTag, null)
        {
            OpenAllCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            OpenAllCommand.Subscribe(_ =>
            {
                foreach (var e in Errors)
                    e.IsShowDetail.Value = true;
            }).AddTo(CompositeDisposable);

            CloseAllCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CloseAllCommand.Subscribe(_ =>
            {
                foreach (var e in Errors)
                    e.IsShowDetail.Value = false;
            }).AddTo(CompositeDisposable);

            ExportCommand = new ReactiveCommand<string>().AddTo(CompositeDisposable);

            ExportCommand
                .Select(_ =>
                {
                    var message = Messenger.GetResponse(
                        new SavingFileSelectionMessage(BezelEditor.Appearance.Properties.Resources.MessageKey_FileSave)
                        {
                            Filter = Resources.DialogFilter_Xml
                        });
                    return message?.Response?.FirstOrDefault();
                })
                .Where(x => x != null)
                .Subscribe(filePath =>
                {
                    var errors = new ApplicationMetaErrorContainer
                    {
                        // XML として保存する際は現在の設定言語に応じたメッセージを出力する
                        Errors = Errors.Select(x => new ApplicationMetaError
                        {
                            Id = x.Id,
                            Title = x.Title,
                            Description = x.Description,
                            ErrorSeverity = x.Severity
                        }).ToList()
                    };
                    File.WriteAllText(filePath, errors.ToXml());
                })
                .AddTo(CompositeDisposable);

            Errors =
                model.ToReadOnlyReactiveCollection(e => new ErrorParamVm(e)).AddTo(CompositeDisposable);
        }
    }

    public class ErrorParamVm : ViewModelBase
    {
        public ApplicationMetaError Model { get; }

        public ApplicationMetaErrorSeverity Severity { get; }

        public string Id { get; }

        public string Title
        {
            get
            {
                var culturizedTitle = Model.CulturizedTitle?[Resources.Culture.TwoLetterISOLanguageName];
                if (culturizedTitle != null)
                    return culturizedTitle;
                if (string.IsNullOrEmpty(Model.TitleTag))
                    return Model.Title;
                return Resources.ResourceManager.GetString(Model.TitleTag, Resources.Culture);
            }
        }

        public string Description
        {
            get
            {
                var culturizedDescription = Model.CulturizedDescription?[Resources.Culture.TwoLetterISOLanguageName];
                if (culturizedDescription != null)
                    return culturizedDescription;
                if (string.IsNullOrEmpty(Model.DescriptionTag))
                    return Model.Description;
                return Resources.ResourceManager.GetString(Model.DescriptionTag, Resources.Culture);
            }
        }

        public ReactiveProperty<bool> IsShowDetail { get; }

        public ErrorParamVm(ApplicationMetaError model)
        {
            Model = model;

            IsShowDetail = new ReactiveProperty<bool>().AddTo(CompositeDisposable);

            Id = model.Id;
            Severity = model.ErrorSeverity;

            ////
            CultureService.Instance.ObserveProperty(x => x.Resources, false)
                .Subscribe(_ =>
                {
                    // ReSharper disable ExplicitCallerInfoArgument
                    RaisePropertyChanged(nameof(Title));
                    RaisePropertyChanged(nameof(Description));
                    // ReSharper restore ExplicitCallerInfoArgument
                })
                .AddTo(CompositeDisposable);
        }
    }
}
