﻿using BezelEditor.Mvvm.Messages;
using Microsoft.WindowsAPICodePack.Dialogs;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Properties;
using System;
using System.Linq;
using System.Reactive.Linq;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using BezelEditor.Foundation.Utilities;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class AocContentsParamVm : ParamVm
    {
        public ReactiveProperty<AocContentVm> SelectedAocContent { get; }
        public ReadOnlyReactiveCollection<AocContentVm> AocContents { get; }

        public ReactiveProperty<string> ValidationMessage { get; }

        public ReactiveCommand AddContentCommand { get; }
        public ReactiveCommand RemoveContentCommand { get; }

        public bool IsEditable { get; set; }

        public AocContentsParamVm(Container diContainer, App app, Project project)
            : base(null, null, null)
        {
            SelectedAocContent = new ReactiveProperty<AocContentVm>().AddTo(CompositeDisposable);

            AocContents = app.Project.AocMeta.Contents
                .ToReadOnlyReactiveCollection(x =>
                    new AocContentVm(diContainer, x, app.Project)
                    {
                        Messenger = diContainer.GetInstance<MainWindowVm>().Messenger
                    })
                .AddTo(CompositeDisposable);

            SelectedAocContent.Value = AocContents.FirstOrDefault();
            IsEditable = diContainer.GetInstance<AppProfile>().AppMode.IsMeta();

            {
                Func<string> aocMetaContents = () =>
                    diContainer.GetInstance<PageValidations>().AocMetaContents(app.Project.AocMeta.Contents);

                ValidationMessage = app.Project.AocMeta.Contents.CollectionChangedAsObservable()
                    .Select(x => aocMetaContents())
                    .ToReactiveProperty()
                    .AddTo(CompositeDisposable);

                // ※作成直後に更新されないため
                ValidationMessage.Value = aocMetaContents();
            }

            AddContentCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            AddContentCommand
                .Subscribe(_ => AddContent(project))
                .AddTo(CompositeDisposable);

            RemoveContentCommand = SelectedAocContent
                .Select(x => x != null)
                .ToReactiveCommand()
                .AddTo(CompositeDisposable);
            RemoveContentCommand
                .Subscribe(_ => RemoveContent(SelectedAocContent.Value, project))
                .AddTo(CompositeDisposable);
        }

        private void AddContent(Project project)
        {
            project.AocMeta.AddContent();

            DispatcherUtility.LazyRunOnUiThread(() => SelectedAocContent.Value = AocContents.LastOrDefault());
        }

        private void RemoveContent(AocContentVm aocContent, Project project)
        {
            var r = Messenger.GetResponse(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Information,
                Caption = Resources.Confirmation,
                Text = string.Format(Resources.DialogMessage_Confirmation_DeleteAocContent, aocContent.Title),
                StandardButtons = DialogMessage.StandardButtonsType.OkCancel
            });

            if (r?.Response?.DialogResult == DialogMessage.DialogResultType.Ok)
            {
                project.AocMeta.DeleteContent(aocContent.Model);
                DispatcherUtility.LazyRunOnUiThread(() => SelectedAocContent.Value = AocContents.FirstOrDefault());
            }
        }

        protected override bool IsVisibled(string keyword)
        {
            return true;
        }
    }
}
