﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------uj

using System;
using System.Linq;
using System.Reactive.Linq;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using static Nintendo.Authoring.AuthoringEditor.Properties.Resources;
using CardSpec = Nintendo.Authoring.AuthoringEditor.Core.CardSpec;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public class CardSpecPageVm : PageVmBase
    {
        public ParamVm SizeParamVm { get; private set; }
        public ParamVm ClockParamVm { get; private set; }

        public CardSpecPageVm(Container diContainer, ApplicationMeta appMeta)
            : base(nameof(NxDevelopmentCard))
        {
            var appProfile = diContainer.GetInstance<AppProfile>();
            if (appProfile.CanEditCardSpec)
                InitializeEditable(diContainer, appMeta);
            else
                InitializeReadOnly(diContainer, appMeta);
        }

        private void InitializeEditable(Container diContainer, ApplicationMeta appMeta)
        {
            var isNspMode = diContainer.GetInstance<AppProfile>().AppMode == AppModeType.ApplicationNsp;

            /////////////////////////////////////////////////////////
            var sizeRp =
                appMeta.CardSpec
                    .ToReactivePropertyAsSynchronized(x => x.Size)
                    .SetValidateNotifyError(s => diContainer.GetInstance<PageValidations>().CardSize(appMeta));

            // SIGLO-81378 によって 1 GB のカードを使用するように選択されていなければ、
            // 1 GB のカードを指定できないようにする
            var cardSizeList = appMeta.CardSpec.Size == 1
                ? CardSpec.SizeRange
                : CardSpec.SizeRange.Where(x => x != 1);

            SizeParamVm = new SelectionParamVm(
                nameof(CardSpecSize_Caption),
                nameof(CardSpecSize_Comment),
                sizeRp,
                cardSizeList.Cast<object>(),
                i => $"{i} GB"
            );
            SizeParamVm.IsReadOnly.Value = isNspMode;
            SizeParamVm.CaptionIsUseTag = nameof(AutomaticallySet);

            SizeParamVm.IsUse = appMeta.CardSpec.ToReactivePropertyAsSynchronized(x => x.IsAutomaticSettingSize)
                .AddTo(CompositeDisposable);

            SizeParamVm.IsInputEnabled = SizeParamVm.IsUse.Inverse().ToReactiveProperty().AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var clockRp =
                appMeta.CardSpec
                    .ToReactivePropertyAsSynchronized(x => x.ClockRate)
                    .SetValidateNotifyError(s => diContainer.GetInstance<PageValidations>().CardClockRate(appMeta));

            ClockParamVm = new SelectionParamVm(
                nameof(CardSpecClockRate_Caption),
                nameof(CardSpecClockRate_Comment),
                clockRp,
                CardSpec.ClockRateRange.Cast<object>(),
                i => $"{i} Mhz"
            );
            ClockParamVm.IsReadOnly.Value = isNspMode;
            ClockParamVm.CaptionIsUseTag = nameof(AutomaticallySet);

            ClockParamVm.IsUse = appMeta.CardSpec.ToReactivePropertyAsSynchronized(x => x.IsAutomaticSettingClockRate)
                .AddTo(CompositeDisposable);

            ClockParamVm.IsInputEnabled = ClockParamVm.IsUse.Inverse().ToReactiveProperty().AddTo(CompositeDisposable);

            // バリデーション更新
            Observable
                .Merge(SizeParamVm.IsUse.ToUnit())
                .Merge(appMeta.CardSpec.ObserveProperty(x => x.Size).ToUnit())
                .Subscribe(_ => sizeRp.ForceValidate())
                .AddTo(CompositeDisposable);

            Observable
                .Merge(ClockParamVm.IsUse.ToUnit())
                .Merge(appMeta.CardSpec.ObserveProperty(x => x.ClockRate).ToUnit())
                .Subscribe(_ => clockRp.ForceValidate())
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            Params = new ParamVm[]
            {
                SizeParamVm,
                ClockParamVm
            };

            /////////////////////////////////////////////////////////
            Observable
                .Merge(appMeta.CardSpec.ObserveProperty(x => x.ValidationSize).ToUnit())
                .Merge(appMeta.CardSpec.ObserveProperty(x => x.ValidationClockRate).ToUnit())
                .Subscribe(i => HasErrors.Value =
                        appMeta.CardSpec.ValidationSize != CardSpec.SizeValidationType.Ok ||
                        appMeta.CardSpec.ValidationClockRate != CardSpec.ClockRateValidationType.Ok
                )
                .AddTo(CompositeDisposable);
        }

        private void InitializeReadOnly(Container diContainer, ApplicationMeta appMeta)
        {
            /////////////////////////////////////////////////////////
            var sizeRp = new ReactiveProperty<string>().AddTo(CompositeDisposable);

            Observable
                .Merge(appMeta.CardSpec.ObserveProperty(x => x.Size).ToUnit())
                .Merge(appMeta.CardSpec.ObserveProperty(x => x.IsAutomaticSettingSize).ToUnit())
                .Merge(CultureService.Instance.ObserveProperty(x => x.CultureName, false).ToUnit())
                .Subscribe(_ =>
                {
                    sizeRp.Value = diContainer.GetInstance<StringHelper>().GetCardSizeReadableString(appMeta.CardSpec);
                })
                .AddTo(CompositeDisposable);

            SizeParamVm = new StringParamVm(
                nameof(CardSpecSize_Caption),
                nameof(CardSpecSize_ReadOnly_Comment),
                sizeRp
            ).AddTo(CompositeDisposable);
            SizeParamVm.IsReadOnly.Value = true;
            SizeParamVm.Width = GuiConstants.ExtraSmallWidth;

            /////////////////////////////////////////////////////////
            var clockRp = new ReactiveProperty<string>().AddTo(CompositeDisposable);

            Observable
                .Merge(appMeta.CardSpec.ObserveProperty(x => x.ClockRate).ToUnit())
                .Merge(appMeta.CardSpec.ObserveProperty(x => x.IsAutomaticSettingClockRate).ToUnit())
                .Merge(CultureService.Instance.ObserveProperty(x => x.CultureName, false).ToUnit())
                .Subscribe(_ =>
                {
                    clockRp.Value =
                        diContainer.GetInstance<StringHelper>().GetCardClockRateReadableString(appMeta.CardSpec);
                })
                .AddTo(CompositeDisposable);

            ClockParamVm = new StringParamVm(
                nameof(CardSpecClockRate_Caption),
                nameof(CardSpecClockRate_ReadOnly_Comment),
                clockRp
            ).AddTo(CompositeDisposable);
            ClockParamVm.IsReadOnly.Value = true;
            ClockParamVm.Width = GuiConstants.ExtraSmallWidth;

            /////////////////////////////////////////////////////////
            Params = new ParamVm[]
            {
                SizeParamVm,
                ClockParamVm
            };
        }
    }
}
