﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Reactive.Concurrency;
using System.Reactive.Linq;
using System.Runtime.InteropServices;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Interop;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Foundation.Utilities;
using GongSolutions.Wpf.DragDrop;
using Livet.Messaging.Windows;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Reactive.Bindings;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow
{
    public partial class MainWindowVm
    {
        [DllImport("user32.dll")]
        private static extern bool IsWindowEnabled(IntPtr hwnd);

        public ReactiveProperty<DraggingTypes> DraggingType { get; private set; }
        public ReactiveProperty<string> DraggingFilePathWithMessage { get; private set; }
        public ReactiveProperty<bool> IsInDragging { get; private set; }
        public ReactiveProperty<string[]> DroppedFile { get; private set; }

        private void InitializeDragAndDrop()
        {
            DraggingType = new ReactiveProperty<DraggingTypes>().AddTo(CompositeDisposable);
            DraggingFilePathWithMessage = new ReactiveProperty<string>().AddTo(CompositeDisposable);
            IsInDragging = new ReactiveProperty<bool>().AddTo(CompositeDisposable);
            IsInDragging.Where(x => x)
                .Subscribe(_ => DraggingFilePathWithMessage.Value = null)
                .AddTo(CompositeDisposable);
            DroppedFile = new ReactiveProperty<string[]>().AddTo(CompositeDisposable);

            DroppedFile
                .ObserveOn(ThreadPoolScheduler.Instance)
                .Where(x => x != null && x.Length > 0)
                .Select(async files =>
                {
                    if (files.Length == 2) // nsp のペアの場合で、かつ patch と app なら,  app を originalNsp としてパッチを開く
                    {
                        if (files.All(x => TypeHelper.FilePathToImportableFileType(x) == ImportableFileType.Nsp) == false)
                            return null;

                        var fileTypes = await Task.WhenAll(files.Select(x => NspImporter.DetectContentMetaTypeAsync(new NspFile(x))));
                        if (fileTypes[0] == ContentMetaType.Patch &&
                            fileTypes[1] == ContentMetaType.Application)
                        {
                            return new { FileType = ImportableFileType.Nsp, Path = files[0], OriginalPath = files[1] };
                        }
                        if (fileTypes[1] == ContentMetaType.Patch &&
                            fileTypes[0] == ContentMetaType.Application)
                        {
                            return new { FileType = ImportableFileType.Nsp, Path = files[1], OriginalPath = files[0] };
                        }
                        return null;
                    }

                    var sourceFile = files.FirstOrDefault(x => TypeHelper.FilePathToImportableFileType(x) != null);
                    var sourceFileType = TypeHelper.FilePathToImportableFileType(sourceFile);
                    if (sourceFileType.HasValue == false)
                        return null;

                    return new { FileType = sourceFileType.Value, Path = sourceFile, OriginalPath = string.Empty };
                })
                .Where(x => x != null)
                .ObserveOn(UIDispatcherScheduler.Default)
                .Subscribe(async openFile =>
                {
                    var file = await openFile;

                    if (ProjectFileHelper.CheckTargetModified(
                            file.FileType, Messenger, _app, AppMode.Value,
                            _diContainer.GetInstance<StringHelper>().ImportFromType(file.FileType)) == false)
                    {
                        return;
                    }

                    ProjectFileHelper.Open(
                        _diContainer, Messenger, file.FileType, _config,
                        file.Path,
                        string.IsNullOrEmpty(file.OriginalPath) == false ? file.OriginalPath : null, // originalNspFilePath
                        (p, originalNspFilePath) => _app.Open(file.FileType, p, originalNspFilePath),
                        p =>
                        {
                            _diContainer.GetInstance<AppProfile>().AppMode = TypeHelper.ToAppMode(_app.Project);
                            CurrentPanelType = PanelType.ProjectEdit;
                        });
                })
                .AddTo(CompositeDisposable);
        }

        public void DragOver(IDropInfo dropInfo)
        {
            if (DragAndDropHelper.IsContainsFileObject(dropInfo) == false)
            {
                IsInDragging.Value = false;
                dropInfo.Effects = DragDropEffects.None;
                return;
            }

            if (PresentationSource.FromVisual(dropInfo.VisualTarget) is HwndSource source &&
                IsWindowEnabled(source.Handle) == false)
            {
                IsInDragging.Value = false;
                dropInfo.Effects = DragDropEffects.None;
                return;
            }

            var files = DragAndDropHelper.MakeDropedFilePaths(dropInfo);
            var sourceFile = files.FirstOrDefault(x => TypeHelper.FilePathToImportableFileType(x) != null);
            var sourceFileType = TypeHelper.FilePathToImportableFileType(sourceFile);

            if (sourceFileType.HasValue == false)
            {
                dropInfo.Effects = DragDropEffects.None;
                DraggingType.Value = DraggingTypes.NotSupportType;
                DraggingFilePathWithMessage.Value = Properties.Resources.NotSupportType;
                return;
            }

            // プロジェクト編集中はその対応タイプしか扱えない
            if (CurrentPanelType == PanelType.ProjectEdit)
            {
                if (_diContainer.GetInstance<AppProfile>().AppMode != TypeHelper.ToAppMode(sourceFile))
                {
                    dropInfo.Effects = DragDropEffects.None;

                    switch (_diContainer.GetInstance<AppProfile>().AppMode)
                    {
                        case AppModeType.Project:
                            DraggingType.Value = DraggingTypes.CannnotLoadOnlyProject;
                            DraggingFilePathWithMessage.Value = Properties.Resources.CannnotLoadOnlyProject;
                            break;

                        case AppModeType.AocMeta:
                        case AppModeType.ApplicationMeta:
                            DraggingType.Value = DraggingTypes.CannnotLoadOnlyMeta;
                            DraggingFilePathWithMessage.Value = Properties.Resources.CannnotLoadOnlyMeta;
                            break;

                        case AppModeType.PatchNsp:
                        case AppModeType.ApplicationNsp:
                        case AppModeType.AocNsp:
                            DraggingType.Value = DraggingTypes.CannnotLoadOnlyNsp;
                            DraggingFilePathWithMessage.Value = Properties.Resources.CannnotLoadOnlyNsp;
                            break;

                        default:
                            throw new ArgumentOutOfRangeException();
                    }

                    return;
                }
            }

            sourceFile = Path.GetFileName(sourceFile);

            DraggingType.Value = DraggingTypes.ImportFile;
            DraggingFilePathWithMessage.Value =
                sourceFile != null
                    ? string.Format(Properties.Resources.ImportFile, sourceFile)
                    : null;

            dropInfo.Effects =
                files.Any(x => TypeHelper.FilePathToImportableFileType(x) != null)
                    ? DragDropEffects.Copy
                    : DragDropEffects.None;
        }

        public void Drop(IDropInfo dropInfo)
        {
            Messenger.Raise(new WindowActionMessage(WindowAction.Active, "WindowAction"));
            IsInDragging.Value = false;
            WpfUtility.DoEvents();
            DroppedFile.Value = DragAndDropHelper.MakeDropedFilePaths(dropInfo);
        }
    }
}
