﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Reactive.Threading.Tasks;
using System.Threading.Tasks;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ComparisonPanel
{
    // ビューとモデル 直結
    public class Param : DisposableModelBase
    {
        public class ResultValue
        {
            public bool IsIdentical { get; set; }
            public bool IsNotRead { get; set; }
            public bool HasValue0 { get; set; }
            public bool HasValue1 { get; set; }

            public string GetValue0() => IsNotRead ? Resources.ParameterNotRead : $"{ToExistsString(HasValue0)}";
            public string GetValue1() => IsNotRead ? Resources.ParameterNotRead : $"{ToExistsString(HasValue1)}" + (IsIdentical ? string.Empty : ",not-equal");

            private static string ToExistsString(bool x) => x ? Resources.ItemExists : Resources.ItemNoExists;
        }

        #region IsExpanded

        private bool _IsExpanded = true;

        public bool IsExpanded
        {
            get { return _IsExpanded; }
            set { SetProperty(ref _IsExpanded, value); }
        }

        #endregion

        #region Name

        private string _Name;

        public string Name
        {
            get
            {
                if (_Name == null)
                    return null;

                return Resources.ResourceManager.GetString(_Name, Resources.Culture) ?? _Name;
            }
            set { SetProperty(ref _Name, value); }
        }

        #endregion

        #region Value0

        private string _Value0;

        public string Value0
        {
            get { return _Value0; }
            set { SetProperty(ref _Value0, value); }
        }

        #endregion

        #region Value1

        private string _Value1;

        public string Value1
        {
            get { return _Value1; }
            set { SetProperty(ref _Value1, value); }
        }

        #endregion

        #region DisplayValue0

        private string _DisplayValue0;

        public string DisplayValue0
        {
            get { return _DisplayValue0; }
            set { SetProperty(ref _DisplayValue0, value); }
        }

        #endregion

        #region DisplayValue1

        private string _DisplayValue1;

        public string DisplayValue1
        {
            get { return _DisplayValue1; }
            set { SetProperty(ref _DisplayValue1, value); }
        }

        #endregion

        #region IsExistsValue0

        private bool _IsExistsValue0 = true;

        public bool IsExistsValue0
        {
            get { return _IsExistsValue0; }
            set { SetProperty(ref _IsExistsValue0, value); }
        }

        #endregion

        #region IsExistsValue1

        private bool _IsExistsValue1 = true;

        public bool IsExistsValue1
        {
            get { return _IsExistsValue1; }
            set { SetProperty(ref _IsExistsValue1, value); }
        }

        #endregion

        #region ComparisonState

        private ValueComparisonStates _ComparisonState;

        public ValueComparisonStates ComparisonState
        {
            get { return _ComparisonState; }
            set { SetProperty(ref _ComparisonState, value); }
        }

        #endregion

        #region Children

        private ObservableCollection<Param> _Children = new ObservableCollection<Param>();

        public ObservableCollection<Param> Children
        {
            get { return _Children; }
            set { SetProperty(ref _Children, value); }
        }

        #endregion

        #region IsDisplay

        private bool _IsDisplay = true;

        public bool IsDisplay
        {
            get { return _IsDisplay; }
            set { SetProperty(ref _IsDisplay, value); }
        }

        #endregion

        #region IsHideValue

        private bool _IsHideValue;

        public bool IsHideValue
        {
            get { return _IsHideValue; }
            set { SetProperty(ref _IsHideValue, value); }
        }

        #endregion

        #region ValueUpdateProgress

        private string _ValueUpdateProgress;

        public string ValueUpdateProgress
        {
            get { return _ValueUpdateProgress; }
            set { SetProperty(ref _ValueUpdateProgress, value); }
        }

        #endregion

        public Func<string, string> ValueToDisplay { get; set; }

        public bool IsEqualValueWithChildren
        {
            get
            {
                if (Value0 != Value1)
                    return false;

                foreach (var child in Children)
                    if (child.IsEqualValueWithChildren == false)
                        return false;

                return true;
            }
        }

        public void UpdateResult()
        {
            if (_IsAsyncParamResult && _AsyncParamResultValue == null)
                return;

            var s = Value0 == Value1;

            if (s == false)
            {
                ComparisonState = ValueComparisonStates.Difference;
                return;
            }

            ComparisonState = IsEqualValueWithChildren
                ? ValueComparisonStates.Equal
                : ValueComparisonStates.DifferenceChild;
        }

        private bool _IsAsyncParamResult;
        private ResultValue _AsyncParamResultValue;

        public Param(Config config, string name,
            Func<string> updateValue0, bool isExists0,
            Func<string> updateValue1, bool isExists1,
            Func<string, string> valueToDisplay)
            : this(config, name, updateValue0, updateValue1, valueToDisplay)
        {
            IsExistsValue0 = isExists0;
            IsExistsValue1 = isExists1;
        }

        public Param(Config config, string name, Func<string> updateValue0, Func<string> updateValue1,
            Func<string, string> valueToDisplay)
            : this(config, name)
        {
            ValueToDisplay = valueToDisplay;

            config.ObserveProperty(x => x.Culture)
                .Subscribe(_ =>
                {
                    Value0 = updateValue0();
                    Value1 = updateValue1();

                    DisplayValue0 = ValueToDisplay?.Invoke(Value0) ?? Value0;
                    DisplayValue1 = ValueToDisplay?.Invoke(Value1) ?? Value1;
                })
                .AddTo(CompositeDisposable);
        }

        public Param(Config config, string name, ResultValue resultValue, bool isExists0, bool isExists1, Func<string, string> valueToDisplay)
            : this(config, name)
        {
            ValueToDisplay = valueToDisplay;

            IsExistsValue0 = isExists0;
            IsExistsValue1 = isExists1;

            config.ObserveProperty(x => x.Culture)
                .Subscribe(_ =>
                {
                    Value0 = resultValue.GetValue0();
                    Value1 = resultValue.GetValue1();

                    DisplayValue0 = ValueToDisplay?.Invoke(Value0) ?? Value0;
                    DisplayValue1 = ValueToDisplay?.Invoke(Value1) ?? Value1;
                })
                .AddTo(CompositeDisposable);
        }

        public Param(Config config, string name, Func<Action<float>, Task<ResultValue>> updateValues,
            Func<string, string> valueToDisplay) : this(config, name)
        {
            ValueToDisplay = valueToDisplay;
            ComparisonState = ValueComparisonStates.Detecting;

            _IsAsyncParamResult = true;

            updateValues(p => ValueUpdateProgress = $"{p:0.00}".TrimEnd('0', '.') + "%")
                .ToObservable()
                .ObserveOnUIDispatcher()
                .Subscribe(r =>
                {
                    _AsyncParamResultValue = r;

                    Value0 = r.GetValue0();
                    Value1 = r.GetValue1();

                    DisplayValue0 = ValueToDisplay?.Invoke(Value0) ?? Value0;
                    DisplayValue1 = ValueToDisplay?.Invoke(Value1) ?? Value1;

                    UpdateResult();
                })
                .AddTo(CompositeDisposable);

            config.ObserveProperty(x => x.Culture)
                .Subscribe(_ =>
                {
                    if (_AsyncParamResultValue == null)
                        return;

                    Value0 = _AsyncParamResultValue.GetValue0();
                    Value1 = _AsyncParamResultValue.GetValue1();

                    DisplayValue0 = ValueToDisplay?.Invoke(Value0) ?? Value0;
                    DisplayValue1 = ValueToDisplay?.Invoke(Value1) ?? Value1;
                })
                .AddTo(CompositeDisposable);
        }

        public Param(Config config, string name)
        {
            Name = name;

            CompositeDisposable.Add(() => Children.ForEach(x => x.Dispose()));

            config.ObserveProperty(x => x.Culture)
                // ReSharper disable once ExplicitCallerInfoArgument
                .Subscribe(_ => RaisePropertyChanged(nameof(Name)))
                .AddTo(CompositeDisposable);
        }

        public Param(Config config, string name,
            Func<string> updateValue0, bool isExists0,
            Func<string> updateValue1, bool isExists1)
            : this(config, name, updateValue0, isExists0, updateValue1, isExists1, null)
        {

        }

        public Param(Config config, string name, Func<string> updateValue0, Func<string> updateValue1)
            : this(config, name, updateValue0, updateValue1, null)
        {

        }

        public void AddChild(Param child)
        {
            Debug.Assert(child != null);
            Children.Add(child);
        }
    }
}
