﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.ObjectModel;
using System.Linq;
using System.Reactive.Linq;
using System.Threading.Tasks;
using System.Windows.Threading;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Foundation.Utilities;
using BezelEditor.Mvvm;
using BezelEditor.Mvvm.Messages;
using Microsoft.WindowsAPICodePack.Dialogs;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ComparisonPanel.ParamTreeBuilder;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Nintendo.Authoring.AuthoringEditor.SelectTwoNspFilesWindow;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using Disposable = System.Reactive.Disposables.Disposable;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ComparisonPanel
{
    public class ComparisonPanelVm : ViewModelBase
    {
        public class Target
        {
            public Project Project { get; set; }
            public string Name { get; set; }
            public INspFile File { get; set; }
        }

        public ObservableCollection<Target> Targets { get; } = new ObservableCollection<Target>();
        public ReactiveProperty<bool> IsDisplayOnlyDifferences { get; }

        private ComparableFileType _TargetFileType;

        public ComparableFileType TargetFileType
        {
            get { return _TargetFileType; }

            set
            {
                if (_TargetFileType == value)
                    return;

                _TargetFileType = value;

                _paramTreeBuilder?.Dispose();

                switch (TargetFileType)
                {
                    case ComparableFileType.AppNsp:
                        _paramTreeBuilder = new AppNspParamTreeBuilder(_diContainer, Targets);
                        break;

                    case ComparableFileType.AocNsp:
                        _paramTreeBuilder = new AocNspParamTreeBuilder(_diContainer, Targets);
                        break;

                    case ComparableFileType.PatchNsp:
                    case ComparableFileType.PatchNspExtractedDirectory:
                        _paramTreeBuilder = new PatchNspParamTreeBuilder(_diContainer, Targets);
                        break;

                    case ComparableFileType.AppMeta:
                        _paramTreeBuilder = new AppMetaParamTreeBuilder(_diContainer, Targets);
                        break;

                    case ComparableFileType.AocMeta:
                        _paramTreeBuilder = new AocMetaParamTreeBuilder(_diContainer, Targets);
                        break;

                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
        }

        public ObservableCollection<Param> Params { get; set; }

        public Target Target0 => Targets.Count >= 1 ? Targets[0] : null;
        public Target Target1 => Targets.Count >= 2 ? Targets[1] : null;

        private readonly App _app;
        private readonly Container _diContainer;

        private ParamTreeBuilderBase _paramTreeBuilder;

        public ComparisonPanelVm(Container diContainer, Config config, App app)
        {
            _app = app;
            _diContainer = diContainer;

            IsDisplayOnlyDifferences = new ReactiveProperty<bool>(false, ReactivePropertyMode.DistinctUntilChanged ).AddTo(CompositeDisposable);
            IsDisplayOnlyDifferences.Subscribe(i => SetDisplayOnlyDifferences(i, Params))
                .AddTo(CompositeDisposable);

            Targets.CollectionChangedAsObservable()
                .Subscribe(_ =>
                {
                    if (Targets.Count < 2)
                        return;

                    Params = _paramTreeBuilder.UpdataParams(config);

                    UpdateParamsResult(Params);
                    SetDisplayOnlyDifferences(IsDisplayOnlyDifferences.Value, Params);

                    // ReSharper disable ExplicitCallerInfoArgument
                    RaisePropertyChanged(nameof(Target0));
                    RaisePropertyChanged(nameof(Target1));
                    // ReSharper restore ExplicitCallerInfoArgument

                }).AddTo(CompositeDisposable);

            CompositeDisposable.Add(() => Params?.ForEach(x => x.Dispose()));
            CompositeDisposable.Add(() =>
            {
                _paramTreeBuilder?.Dispose();
            });
        }

        public bool OpenAll(bool isExpanded)
        {
            if (Params == null)
                return false;

            OpenAllInternal(isExpanded, Params);

            return true;
        }

        public async Task ExtractFileAsync(INspFile nspFile)
        {
            var r =
                Messenger.GetResponse(new FolderSelectionMessage(GuiConstants.MessageKey_DirectoryOpen)
                {
                    Title = Resources.DestinationDirectory
                });

            if (r?.Response == null)
                return;

            var outputDir = r.Response;

            _app.CancelWorking();

            _app.OpenGeneratedMessageTag = nameof(Resources.OpenGeneratedObjectContainsFolder);
            _app.OpenGeneratedObjectAction = WindowsUtility.OpenFolderByExplorer;

            _app.WorkingKind = WorkingKind.Extract;
            _app.WorkingState = WorkingState.Working;

            using (Disposable.Create(() => _app.WorkingProgress = 0))
            {
                // 表示なし
                _app.WorkingProgress = -1;

                try
                {
                    await nspFile.ExtractAllAsync(
                        outputDir,
                        p => _app.WorkingProgress = p,
                        _app.WorkingCancellationTokenSource.Token);

                    _app.WorkingFilePath = outputDir;
                    _app.WorkingState = WorkingState.Succeeded;
                }
                catch (TaskCanceledException)
                {
                    _app.WorkingState = WorkingState.Canceled;
                }
            }
        }

        private static void UpdateParamsResult(ObservableCollection<Param> targets)
        {
            foreach (var t in targets)
            {
                t.UpdateResult();
                UpdateParamsResult(t.Children);
            }
        }

        private static void OpenAllInternal(bool isExpanded, ObservableCollection<Param> targets)
        {
            foreach (var t in targets)
            {
                t.IsExpanded = isExpanded;
                OpenAllInternal(isExpanded, t.Children);
            }
        }

        private static void SetDisplayOnlyDifferences(bool isDisplayOnlyDifferences,
            ObservableCollection<Param> targets)
        {
            if (targets == null)
                return;

            // ReSharper disable ConditionIsAlwaysTrueOrFalse
            if (isDisplayOnlyDifferences)
            {
                foreach (var t in targets)
                {
                    t.IsDisplay = t.IsEqualValueWithChildren == false;
                    SetDisplayOnlyDifferences(isDisplayOnlyDifferences, t.Children);
                }
            }
            else
            {
                foreach (var t in targets)
                {
                    t.IsDisplay = true;
                    SetDisplayOnlyDifferences(isDisplayOnlyDifferences, t.Children);
                }
            }
            // ReSharper restore ConditionIsAlwaysTrueOrFalse
        }

        public bool IsNoDifference => Params.All(x => x.IsEqualValueWithChildren);
    }
}
