﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Reactive.Concurrency;
using System.Reactive.Linq;
using System.Reflection;
using BezelEditor.Foundation;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using BezelEditor.Mvvm.Messages;
using Livet.Messaging;
using Livet.Messaging.Windows;
using Microsoft.WindowsAPICodePack.Dialogs;
using Nintendo.Authoring.AuthoringEditor.Controls;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.SettingWindow;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.AppMenu
{
    public class AppMenuVm : ViewModelBase
    {
        public ReactiveCommand CreateNewProjectCommand { get; }
        public ReactiveCommand<MruItemVm> OpenProjectCommand { get; }
        public ReactiveCommand<MruItemVm> ImportNspCommand { get; }
        public ReactiveCommand SaveCommand { get; }
        public ReactiveCommand SaveAsCommand { get; }

        public ReactiveCommand SettingAppCommand { get; }
        public ReactiveCommand ShowAboutCommand { get; }
        public ReactiveCommand CloseWindowCommand { get; }

        public ReactiveCommand BackToStartPageCommand { get; }

        public ReadOnlyReactiveCollection<MruItemVm> MruProjects { get; set; }
        public ReadOnlyReactiveCollection<MruItemVm> MruMetas { get; set; }
        public ReadOnlyReactiveCollection<MruItemVm> MruNsps { get; set; }

        public event EventHandler ProjectOpening;

        private readonly Container _diContainer;

        private ImportableFileType TargetFileType => _diContainer.GetInstance<AppProfile>().TargetFileType;
        private AppModeType AppMode => _diContainer.GetInstance<AppProfile>().AppMode;

        public AppMenuVm(Container diContainer, App app, Config config)
        {
            _diContainer = diContainer;

            var appProfile = _diContainer.GetInstance<AppProfile>();

            appProfile
                .ObserveProperty(x => x.AppMode)
                .Subscribe(_ =>
                {
                    // SaveCommand,SaveAsCommand再評価のため
                    app.RaisePropertyChanged(nameof(App.HasErrors));
                }).AddTo(CompositeDisposable);

            CreateNewProjectCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CreateNewProjectCommand
                .Subscribe(_ => CreateNewProject(app, appProfile.AppMode, ContentMetaType.Application))
                .AddTo(CompositeDisposable);

            OpenProjectCommand = new ReactiveCommand<MruItemVm>().AddTo(CompositeDisposable);
            OpenProjectCommand
                .Subscribe(mruItem => OpenProject(app, appProfile.AppMode, config, mruItem, TargetFileType))
                .AddTo(CompositeDisposable);

            ImportNspCommand = new ReactiveCommand<MruItemVm>().AddTo(CompositeDisposable);
            ImportNspCommand
                .Subscribe(mruItem => ImportNsp(app, appProfile.AppMode, config, mruItem))
                .AddTo(CompositeDisposable);

            SaveCommand =
                app.ObserveProperty(x => x.HasErrors)
                    .Select(CanSave)
                    .ToReactiveCommand()
                    .AddTo(CompositeDisposable);
            SaveCommand
                .Subscribe(_ => ProjectFileHelper.Save(TargetFileType.ToExportableFileType(AppMode), Messenger, app))
                .AddTo(CompositeDisposable);

            SaveAsCommand =
                app.ObserveProperty(x => x.HasErrors)
                    .Select(CanSave)
                    .ToReactiveCommand()
                    .AddTo(CompositeDisposable);
            SaveAsCommand
                .Subscribe(_ => ProjectFileHelper.SaveAs(TargetFileType.ToExportableFileType(AppMode), Messenger, app))
                .AddTo(CompositeDisposable);

            SettingAppCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            SettingAppCommand
                .Subscribe(_ => ShowSetting())
                .AddTo(CompositeDisposable);

            ShowAboutCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            ShowAboutCommand
                .Subscribe(_ => ShowAbout())
                .AddTo(CompositeDisposable);

            CloseWindowCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CloseWindowCommand
                .Subscribe(_ => CloseWindow()).AddTo(CompositeDisposable);

            BackToStartPageCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            BackToStartPageCommand
                .Subscribe(_ => BackToStartPage(app, appProfile.AppMode)).AddTo(CompositeDisposable);

            MruProjects =
                config.MruProjects.ToReadOnlyReactiveCollection(item => new MruItemVm(item),
                        ImmediateScheduler.Instance)
                    .AddTo(CompositeDisposable);

            MruMetas =
                config.MruMetas.ToReadOnlyReactiveCollection(item => new MruItemVm(item),
                        ImmediateScheduler.Instance)
                    .AddTo(CompositeDisposable);

            MruNsps =
                config.MruNsps.ToReadOnlyReactiveCollection(item => new MruItemVm(item),
                        ImmediateScheduler.Instance)
                    .AddTo(CompositeDisposable);
        }

        private bool CanSave(bool x)
        {
            // プロジェクト以外はエラーあると保存できない
            if (AppMode == AppModeType.Project)
                return true;

            return x == false;
        }

        private void BackToStartPage(App app, AppModeType appMode)
        {
            if (
                ProjectFileHelper.CheckTargetModified(TargetFileType, Messenger, app, appMode,
                    Properties.Resources.BackToStartPage) == false)
                return;

            _diContainer.GetInstance<MainWindowVm>().CurrentPanelType = MainWindowVm.PanelType.Startup;
        }

        private void CloseWindow()
        {
            Messenger.Raise(new WindowActionMessage(WindowAction.Close, "WindowAction"));
        }

        private void CreateNewProject(App app, AppModeType appMode, ContentMetaType type)
        {
            ProjectOpening?.Invoke(this, EventArgs.Empty);

            if (
                ProjectFileHelper.CheckTargetModified(TargetFileType, Messenger, app, appMode,
                    _diContainer.GetInstance<StringHelper>().CreateNew) == false)
                return;

            using (new InPreparationBlock())
                ProjectFileHelper.CreateNewProject(app, type);
        }

        private void OpenProject(App app, AppModeType appMode, Config config, MruItemVm mruItem, ImportableFileType type)
        {
            ProjectOpening?.Invoke(this, EventArgs.Empty);

            if (
                ProjectFileHelper.CheckTargetModified(TargetFileType, Messenger, app, appMode,
                    _diContainer.GetInstance<StringHelper>().CreateNew) == false)
                return;

            OpenProjectInternal(mruItem,
                m => ProjectFileHelper.Open(_diContainer, Messenger, type, config,
                    m?.FilePath.Value,
                    null, // originalNspFilePath
                    (p, originalNspFilePath) => app.Open(type, p, originalNspFilePath),
                    null));
        }

        private void ImportNsp(App app, AppModeType appMode, Config config, MruItemVm mruItem)
        {
            ProjectOpening?.Invoke(this, EventArgs.Empty);

            if (AppMode != AppModeType.ApplicationNsp &&
                ProjectFileHelper.CheckTargetModified(TargetFileType, Messenger, app, appMode,
                    _diContainer.GetInstance<StringHelper>().CreateNew) == false)
            {
                return;
            }

            const ImportableFileType type = ImportableFileType.Nsp;

            var helper = _diContainer.GetInstance<StringHelper>();
            if (ProjectFileHelper.CheckTargetModified(type, Messenger, app, appMode, helper.ImportFromType(type)) ==
                false)
                return;

            ProjectFileHelper.Open(_diContainer, Messenger, type, config,
                mruItem?.FilePath.Value,
                null, // originalNspFilePath
                (p, originalNspFilePath) => app.Open(type, p, originalNspFilePath),
                null);
        }

        private static void OpenProjectInternal(MruItemVm mruItem, Action<MruItemVm> openFile)
        {
            if (mruItem == null)
                openFile(null);
            else
            {
                using (new InPreparationBlock())
                    openFile(mruItem);
            }
        }

        private void ShowSetting()
        {
            using (var vm = _diContainer.GetInstance<SettingWindowVm>())
                Messenger.Raise(new TransitionMessage(vm, "ShowSettingWindow"));
        }

        private void ShowAbout()
        {
            var newLine = Environment.NewLine;
            var version = GetVersionString();
            var authoringToolVersion = AuthoringToolWrapper.Version();
            var copyright = Assembly.GetEntryAssembly() == null ? string.Empty : AssemblyConstants.Copyright;

            var aboutMessage =
                $"AuthoringEditor{newLine}" +
                $"{version}{newLine}" +
                $"{AssemblyConstants.BuildTime}{newLine}{newLine}" +
                $"AuthoringTool{newLine}" +
                $"{authoringToolVersion}{newLine}" +
                $"{AuthoringToolHelper.AuthoringToolExe}{newLine}{newLine}" +
                $"{copyright}";

            Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.None,
                Caption = Properties.Resources.AboutDialogTitle,
                Text = aboutMessage,
                StandardButtons = DialogMessage.StandardButtonsType.Ok
            });
        }

        private static string GetVersionString()
        {
            var nxAddonVersion = AuthoringToolWrapper.NxAddonVersion() ?? string.Empty;
            var assemblyVersion = AssemblyConstants.Version?.ToString() ?? string.Empty;
            return $"{nxAddonVersion.Replace('.', '_')}:{assemblyVersion.Replace('.', '_')}";
        }
    }
}
