﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringEditor.Helper
{
    public class FsUtil
    {
        private const int BufferSize = 1024 * 1024 * 16;

        public static void WriteFile(string filePath, IFileSystemArchiveReader fsReader, string fileName, long fileSize)
        {
            var directoryPath = Path.GetDirectoryName(filePath);
            if (string.IsNullOrEmpty(directoryPath) == false && Directory.Exists(directoryPath) == false)
                Directory.CreateDirectory(directoryPath);

            using (var writer = OpenWriteFileStream(filePath))
            {
                WriteFile(writer, fsReader, fileName, fileSize);
            }
        }

        public static Stream WriteFile(Stream writer, IFileSystemArchiveReader fsReader, string fileName, long fileSize)
        {
            long readSize = 0;
            while (readSize < fileSize)
            {
                uint bufferSize = (uint)Math.Min(BufferSize, fileSize - readSize);
                var bytes = fsReader.ReadFile(fileName, readSize, bufferSize);
                writer.Write(bytes, 0, bytes.Length);
                readSize += bytes.Length;
            }
            return writer;
        }

        public static bool CompareFile(IFileSystemArchiveReader sourceFsReader, IFileSystemArchiveReader targetFsReader,
            string fileName, long fileSize, Action<int> onReadBytes = null)
        {
            long readSize = 0;
            while (readSize < fileSize)
            {
                uint bufferSize = (uint)Math.Min(BufferSize, fileSize - readSize);

                var sourceBytes = sourceFsReader.ReadFile(fileName, readSize, bufferSize);
                var targetBytes = targetFsReader.ReadFile(fileName, readSize, bufferSize);

                if (sourceBytes.SequenceEqual(targetBytes) == false)
                    return false;

                readSize += sourceBytes.Length;

                onReadBytes?.Invoke(sourceBytes.Length);
            }
            return true;
        }

        public static FileStream OpenWriteFileStream(string extractedPath, FileOptions options = FileOptions.SequentialScan)
        {
            return new FileStream(extractedPath, FileMode.Create, FileAccess.ReadWrite, FileShare.None, 4096, options);
        }

        public static FileStream OpenReadOnlyFileStream(string path, FileOptions options = FileOptions.SequentialScan)
        {
            return new FileStream(path, FileMode.Open, FileAccess.Read, FileShare.Read, 4096, options);
        }
    }
}
