﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.Serialization;
using System.Xml.XPath;
using BezelEditor.Foundation;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using YamlDotNet.Serialization;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public class ApplicationIdMismatchedException : Exception
    {
        public ulong ExpectedApplicationId { get; }
        public ulong ActualApplicationId { get; }

        public ApplicationIdMismatchedException(ulong expected, ulong actual)
        {
            ExpectedApplicationId = expected;
            ActualApplicationId = actual;
        }
    }

    public class Project : DisposableModelBase
    {
        public const string CurrentFileVersion = "0.9.1";

        #region FileVersion

        private string _FileVersion = CurrentFileVersion;

        public string FileVersion
        {
            get { return _FileVersion; }
            set { SetProperty(ref _FileVersion, value); }
        }

        #endregion

        #region TargetFilePath

        private PathString _targetFilePath = string.Empty;

        [YamlIgnore]
        [XmlIgnore]
        public PathString TargetFilePath
        {
            get { return _targetFilePath; }
            set { SetProperty(ref _targetFilePath, value); }
        }

        #endregion

        #region ProjectDirectory

        private PathString _ProjectDirectory;

        [YamlIgnore]
        [XmlIgnore]
        public string ProjectDirectory
        {
            get { return _ProjectDirectory; }
            set { SetProperty(ref _ProjectDirectory, value); }
        }

        #endregion

        #region TemporaryDirPath

        private PathString _TemporaryDirPath;

        [YamlIgnore]
        [XmlIgnore]
        public PathString TemporaryDirPath
        {
            get { return _TemporaryDirPath; }
            set { SetProperty(ref _TemporaryDirPath, value); }
        }

        #endregion

        #region Meta

        private ApplicationMeta _meta = new ApplicationMeta();

        public ApplicationMeta Meta // ※プロパティ名とクラス名が対応していないところに注意。
        {
            // 　ファイルアップデータの準備が出来次第名前を変更する
            get { return _meta; }
            set
            {
                var old = _meta;
                if (SetProperty(ref _meta, value))
                    old?.Dispose();
            }
        }

        #endregion

        #region AocMeta

        private AocMeta _aocMeta = new AocMeta();

        public AocMeta AocMeta
        {
            get { return _aocMeta; }
            set
            {
                var old = _aocMeta;
                if (SetProperty(ref _aocMeta, value))
                    old?.Dispose();
            }
        }

        #endregion

        #region ApplicationContentMeta

        private ApplicationContentMeta _applicationContentMeta = new ApplicationContentMeta();

        [YamlIgnore]
        [XmlIgnore]
        public ApplicationContentMeta ApplicationContentMeta
        {
            get { return _applicationContentMeta; }
            set
            {
                var old = _applicationContentMeta;
                if (SetProperty(ref _applicationContentMeta, value))
                    old?.Dispose();
            }
        }

        #endregion

        #region PatchContentMeta

        private PatchContentMeta _patchContentMeta = new PatchContentMeta();

        [YamlIgnore]
        [XmlIgnore]
        public PatchContentMeta PatchContentMeta
        {
            get { return _patchContentMeta; }
            set
            {
                var old = _patchContentMeta;
                if (SetProperty(ref _patchContentMeta, value))
                    old?.Dispose();
            }
        }

        #endregion

        #region AocContentMetas

        private ObservableCollection<AocContentMeta> _aocContentMetas = new ObservableCollection<AocContentMeta>();

        [YamlIgnore]
        [XmlIgnore]
        public ObservableCollection<AocContentMeta> AocContentMetas
        {
            get { return _aocContentMetas; }
            set
            {
                var old = _aocContentMetas;
                if (SetProperty(ref _aocContentMetas, value))
                {
                    foreach (var e in old)
                        e.Dispose();
                }
            }
        }

        #endregion

        #region ProgramInfo

        private ProgramInfo _programInfo = new ProgramInfo();

        [YamlIgnore]
        [XmlIgnore]
        public ProgramInfo ProgramInfo
        {
            get { return _programInfo; }
            set
            {
                var old = _programInfo;
                if (SetProperty(ref _programInfo, value))
                    old?.Dispose();
            }
        }

        #endregion

        #region ContentMetaProperties

        private ContentMetaProperties _contentMetaProperties = new ContentMetaProperties();

        [YamlIgnore]
        [XmlIgnore]
        public ContentMetaProperties ContentMetaProperties
        {
            get { return _contentMetaProperties; }
            set
            {
                var old = _contentMetaProperties;
                if (SetProperty(ref _contentMetaProperties, value))
                    old?.Dispose();
            }
        }

        #endregion

        #region ProgramCodeDirectoryPath

        private string _ProgramCodeDirectoryPath = string.Empty;

        public string ProgramCodeDirectoryPath
        {
            get { return _ProgramCodeDirectoryPath; }
            set { SetProperty(ref _ProgramCodeDirectoryPath, value); }
        }

        #endregion

        #region ProgramDataDirectoryPath

        private string _ProgramDataDirectoryPath = string.Empty;

        public string ProgramDataDirectoryPath
        {
            get { return _ProgramDataDirectoryPath; }
            set { SetProperty(ref _ProgramDataDirectoryPath, value); }
        }

        #endregion

        #region Capability

        private ApplicationCapability _Capability;

        [YamlIgnore]
        [XmlIgnore]
        public ApplicationCapability Capability
        {
            get { return _Capability; }
            set { SetProperty(ref _Capability, value); }
        }

        #endregion

        #region AppCapability

        [YamlIgnore]
        [XmlIgnore]
        public ApplicationCapability AppCapability { get; } = new ApplicationCapability();

        #endregion

        #region NspFile

        private INspFile _NspFile;

        [YamlIgnore]
        [XmlIgnore]
        public INspFile NspFile
        {
            get { return _NspFile; }
            set { SetProperty(ref _NspFile, value); }
        }

        #endregion

        #region OriginalAppNspFilePath

        private string _OriginalAppNspFilePath;

        [YamlIgnore]
        [XmlIgnore]
        public string OriginalAppNspFilePath
        {
            get { return _OriginalAppNspFilePath; }
            set { SetProperty(ref _OriginalAppNspFilePath, value); }
        }

        #endregion

        #region XmlRootMetaKind

        private RootMetaKind _XmlRootMetaKind = RootMetaKind.NintendoSdkMeta;

        public RootMetaKind XmlRootMetaKind
        {
            get
            {
                if (AppCapability.IsSupportNintendoSdkMetaRoot == false)
                    return RootMetaKind.Meta;
                return _XmlRootMetaKind;
            }
            set { SetProperty(ref _XmlRootMetaKind, value); }
        }

        #endregion

        [YamlIgnore]
        [XmlIgnore]
        public INspExtraResourceImporter ExtraResourceImporter { get; private set; }

        // アプリケーション nsp , 追加コンテンツ nsp または製品化処理されていないパッチなら nsp の構成を確認可能
        [YamlIgnore]
        [XmlIgnore]
        public bool IsAvailableNspEntries => ApplicationContentMeta != null ||
                                             AocMeta.Contents.IsEmpty() == false ||
                                             (PatchContentMeta != null && PatchContentMeta.IsProduction == false);

        [YamlIgnore]
        [XmlIgnore]
        public Container DiContainer
        {
            get { return _DiContainer; }
            set
            {
                _DiContainer = value;
                Meta.DiContainer = value;
                AocMeta.DiContainer = value;

                // ReSharper disable once ExplicitCallerInfoArgument
                RaisePropertyChanged(nameof(DiContainer));
            }
        }

        private Container _DiContainer;

        [YamlIgnore]
        [XmlIgnore]
        public string Digest => PatchContentMeta != null ? PatchContentMeta.Digest : ApplicationContentMeta.Digest;

        public Project()
        {
            CompositeDisposable.Add(() => Meta?.Dispose());
            CompositeDisposable.Add(() => AocMeta?.Dispose());
            CompositeDisposable.Add(() => ApplicationContentMeta?.Dispose());
            CompositeDisposable.Add(() => ProgramInfo?.Dispose());
            CompositeDisposable.Add(() => ContentMetaProperties?.Dispose());

            // nsp 保存時など現在開いているファイル (TargetFilePath) に変更があった場合は、
            // 操作対象とする nsp ファイルのパスもそれに伴って変更する
            this.ObserveProperty(x => x.TargetFilePath)
                .Subscribe(x =>
                {
                    if (NspFile != null)
                        NspFile.NspPath = x;
                })
                .AddTo(CompositeDisposable);
        }

        public Project DeepClone()
        {
            var newProject = this.DeepCloneByYamlSerializer();
            var app = newProject.Meta?.Application;
            if (app != null)
                app.UnknownXmlElements = Meta?.Application.UnknownXmlElements;
            return newProject;
        }

        public static Project CreateNew(Container diContainer)
        {
            var project = new Project {DiContainer = diContainer};
            var app = project.Meta.Application;

            if (project.AppCapability.IsSupportVideoCapture)
            {
                app.VideoCapture = VideoCaptureType.Enable;
            }

            if (project.AppCapability.IsSupportRuntimeAddOnContentInstall)
            {
                app.RuntimeAddOnContentInstall = RuntimeAddOnContentInstallType.Deny;
            }

            if (project.AppCapability.IsSupportCrashReport)
            {
                app.CrashReport = CrashReportType.Deny;
            }

            if (project.AppCapability.IsSupportPlayLogs)
            {
                app.PlayLogQueryCapability = PlayLogQueryCapabilityType.None;
            }

            return project;
        }

        public static Project Import(Container diContainer, ImportableFileType type,
            PathString filePath, PathString originalAppNspFilePath = null)
        {
            var tempDirPath = diContainer.GetInstance<App>().AddTempDir();
            if (Directory.Exists(tempDirPath) == false)
                Directory.CreateDirectory(tempDirPath);

            Project project;

            switch (type)
            {
                case ImportableFileType.Project:
                    if (File.Exists(filePath) == false)
                        throw new FileNotFoundException();
                    project = FromProjectFile(filePath);
                    break;

                case ImportableFileType.Meta:
                    if (File.Exists(filePath) == false)
                        throw new FileNotFoundException();
                    project = FromMetaFile(diContainer, filePath);

                    // AuthoringEditor として編集できない値の誤りについては、読み込み時にチェックして違反があれば例外を飛ばす
                    // 編集可能な値については、読み込み時にエラーがあったとしても処理を継続する
                    if (project.Meta.HasErrorNonPublicProperties)
                        throw new FileFormatException(); // TODO: もっと分かりやすいエラーメッセージ
                    break;

                case ImportableFileType.Nsp:
                    if (File.Exists(filePath) == false && Directory.Exists(filePath) == false)
                        throw new FileNotFoundException();
                    project = FromNspFileOrDirectory(filePath, originalAppNspFilePath, tempDirPath, diContainer);
                    // 3PT からダウンロードしたファイルフォーマット (.xml) の場合はディレクトリが Nsp の構造 (部分的) になっているので、
                    // ファイルパスをディレクトリに再設定する
                    filePath = project.NspFile.NspPath;
                    break;

                default:
                    throw new ArgumentOutOfRangeException(nameof(type), type, null);
            }

            project.TargetFilePath = filePath;
            project.ProjectDirectory = filePath.Parent;
            project.TemporaryDirPath = tempDirPath;
            project.DiContainer = diContainer;
            project.Capability = ApplicationCapability.CreateCustomCapability(new Lazy<Version>(() =>
            {
                // プログラムを作成した SDK バージョンをプロジェクト固有のケイパビリティとする
                var sdkVersion = project.ProgramInfo?.SdkVersion;
                if (string.IsNullOrEmpty(sdkVersion))
                    return new Version(); // nmeta の場合は常にバージョンは空になる
                return Version.Parse(sdkVersion);
            }));

            project.FixupMeta();
            project.FixupFilePath();

            return project;
        }

        public void ImportPartial(PathString filePath)
        {
            if (File.Exists(filePath) == false)
                throw new FileNotFoundException();

            var metaXmlText = File.ReadAllText(filePath, Encoding.UTF8);
            var appMeta = ApplicationMetaFromMetaXml(metaXmlText);

            EnsureImportMetaApplicationId(appMeta);

            // nmeta の場合のみ (ApplicationContentMeta がない場合にのみ) Core と CardSpec をインポートする
            var isMetaEdit = ApplicationContentMeta == null;
            if (isMetaEdit)
            {
                Meta.Core.MergePropertyValues(appMeta.Core);
                Meta.CardSpec.MergePropertyValues(appMeta.CardSpec);
            }
            Meta.Application.MergePropertyValues(appMeta.Application);
        }

        public IEnumerable<string> ImportPartialDryRun(PathString filePath)
        {
            if (File.Exists(filePath) == false)
                throw new FileNotFoundException();

            var metaXmlText = File.ReadAllText(filePath, Encoding.UTF8);
            var appMeta = ApplicationMetaFromMetaXml(metaXmlText);
            var isMetaEdit = ApplicationContentMeta == null;

            EnsureImportMetaApplicationId(appMeta);

            if (isMetaEdit)
            {
                foreach (var prop in appMeta.Core.GetChangedPropertyNames().OrderBy(x => x))
                    yield return $"{nameof(Core)}/{prop}";
            }

            foreach (var prop in appMeta.Application.GetChangedPropertyNames().OrderBy(x => x))
                yield return $"{nameof(Application)}/{prop}";

            if (isMetaEdit)
            {
                foreach (var prop in appMeta.CardSpec.GetChangedPropertyNames().OrderBy(x => x))
                    yield return $"{nameof(CardSpec)}/{prop}";
            }
        }

        private void EnsureImportMetaApplicationId(ApplicationMeta appMeta)
        {
            if (appMeta.Core.IsPropertyChanged(nameof(Core.ApplicationId)) == false)
                return;
            if (Meta.Core.ApplicationId == appMeta.Core.ApplicationId)
                return;

            throw new ApplicationIdMismatchedException(Meta.Core.ApplicationId, appMeta.Core.ApplicationId);
        }

        public string MakeAppMetaXmlForAuthoringTool()
        {
            var serializer = new XmlSerializer(typeof(ApplicationMeta), new XmlRootAttribute
            {
                ElementName = XmlRootMetaKind.ToString()
            });

            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            using (var sw = new StringWriterUtf8())
            {
                var meta = PrepareMetaForOutputXml();
                serializer.Serialize(sw, meta, ns);
                return sw.ToString();
            }
        }

        public string MakeAocMetaXmlForAuthoringTool()
        {
            var data = new Serialization.AocMeta
            {
                AddOnContent = AocMeta.Contents
                    .Select(x => new Serialization.AddOnContent
                    {
                        Index = x.Index,
                        ReleaseVersion = x.ReleaseVersion,
                        ApplicationId = AocMeta.ApplicationId.ToHex(),
                        RequiredApplicationReleaseVersion = x.RequiredApplicationReleaseVersion,
                        Tag = x.Tag,
                        DataPath = new ExpandablePath
                        {
                            Path = x.DataPath.Path,
                            IsExpandEnvironmentVariable = x.DataPath.IsExpandEnvironmentVariable
                        }
                    }).ToArray()
            };

            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            using (var sw = new StringWriterUtf8())
            {
                new XmlSerializer(typeof(Serialization.AocMeta), new XmlRootAttribute
                {
                    ElementName = XmlRootMetaKind.ToString()
                }).Serialize(sw, data, ns);
                return sw.ToString();
            }
        }

        public bool OutputAppMetaXmlFileForAuthoringTool(PathString filePath)
        {
            try
            {
                var xml = MakeAppMetaXmlForAuthoringTool();
                File.WriteAllText(filePath, xml, Encoding.UTF8);
                return true;
            }
            catch
            {
                return false;
            }
        }

        public bool OutputAocMetaXmlFileForAuthoringTool(PathString filePath)
        {
            try
            {
                var xml = MakeAocMetaXmlForAuthoringTool();

                File.WriteAllText(filePath, xml, Encoding.UTF8);
                return true;
            }
            catch
            {
                return false;
            }
        }

        private static Project FromMetaFile(Container diContainer, PathString filePath)
        {
            var metaXmlText = File.ReadAllText(filePath, Encoding.UTF8);

            switch (TypeHelper.DetectMetaKind(filePath))
            {
                case MetaKind.Application:
                {
                    var project = new Project {Meta = ApplicationMetaFromMetaXml(metaXmlText)};
                    FixupMetaFromMetaFile(project.Meta, metaXmlText);
                    return project;
                }

                case MetaKind.AddOnContent:
                    return new Project {AocMeta = AocMetaFromMetaXml(diContainer, metaXmlText)};

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public bool IsUseCompatibleSdk()
        {
            var nspSdkVersion = ProgramInfo?.SdkVersion ?? string.Empty;
            var editorSdkVersion = AppCapability.NxAddonVersion?.ToString() ?? string.Empty;

            if (string.IsNullOrEmpty(nspSdkVersion) || string.IsNullOrEmpty(editorSdkVersion))
                return false;

            var nsp = new Version(nspSdkVersion);
            var editor = new Version(editorSdkVersion);

            // ビルドに使用した SDK のメジャーバージョンの一致しない NSP は "互換性がない" と判断して読み取り専用で開く
            return nsp.Major == editor.Major;
        }

        public void PrepareOnBeforeSave(PathString baseDirPath)
        {
            ProjectDirectory = baseDirPath;
            FixupFilePath();
        }

        public ExpandablePath ToRealtivePath(ExpandablePath sourceFilePath)
        {
            if (string.IsNullOrEmpty(sourceFilePath))
                return null;
            if (_ProjectDirectory == null)
                return sourceFilePath;
            // 環境変数を含む場合は相対パスへの変換を行わない
            if (sourceFilePath.IsExpandEnvironmentVariable)
                return sourceFilePath;
            try
            {
                var relativePath = sourceFilePath.Path.ToPathString().CreateRelativePath(_ProjectDirectory);
                return relativePath.ToString();
            }
            catch (Exception e) when (e is InvalidOperationException || e is ArgumentException)
            {
                // パス変換中に発生した例外は無視
                return sourceFilePath;
            }
        }

        public string ToAbsolutePath(string sourceFilePath)
        {
            if (string.IsNullOrEmpty(sourceFilePath))
                return null;
            try
            {
                if (_ProjectDirectory == null)
                    return Path.Combine(string.Empty, sourceFilePath);
                var path = sourceFilePath.ToPathString();
                if (path.IsAbsolute)
                    return path;
                return path.CreateAbsolutePath(_ProjectDirectory.Directory);
            }
            catch (Exception e) when (e is InvalidOperationException || e is ArgumentException)
            {
                return null;
            }
        }

        public static INspFile GetNspFile(PathString filePath)
        {
            if (File.Exists(filePath))
            {
                if (Path.GetExtension(filePath)?.Equals(".xml", StringComparison.OrdinalIgnoreCase) == false)
                {
                    return new NspFile(filePath)
                    {
                        EnumerationMode = NspFileEnumeration.RootOnly
                    };
                }
                // 指定されたファイルパスの拡張子が .xml の場合、3PT からダウンロードしてきた ROM の xml を含むディレクトリとみなす
                filePath = Path.GetDirectoryName(filePath).ToPathString();
            }
            if (Directory.Exists(filePath))
            {
                return new NspExtractedDirectory(filePath)
                {
                    EnumerationMode = NspFileEnumeration.RootOnly
                };
            }
            return null;
        }

        private ApplicationMeta PrepareMetaForOutputXml()
        {
            var meta = Meta.DeepCloneByYamlSerializer();

            meta.Application.UnknownXmlElements = Meta.Application.UnknownXmlElements;
            meta.Application.IconsFromMetaXml =
                meta.Application.Titles
                    .Where(x => x.IsReplaceIcon && x.IconFilePath.IsEmpty == false)
                    .Select(x => new Icon
                    {
                        IconPath = x.IconFilePath,
                        NxIconPath = x.IsReplaceNxIcon && x.NxIconFilePath.IsEmpty ? null : x.NxIconFilePath,
                        Language = x.Language
                    }).ToArray();

            if (AppCapability.IsSupportSeedForPseudoDeviceId == false ||
                meta.Core.ApplicationId == meta.Application.SeedForPseudoDeviceId)
            {
                // 要素が未サポートの SDK or ApplicationId と同値なので出力を省略
                meta.Application.IsUseSeedForPseudoDeviceAppId = true;
            }

            // 置き換えフラグが無効なら xml にパスが書き出されないように修正

            if (meta.Application.IsReplaceAccessibleUrlsFilePath == false)
                meta.Application.AccessibleUrlsFilePath = null;

            if (meta.Application.IsReplaceHtmlDocumentPath == false)
                meta.Application.HtmlDocumentPath = null;

            if (meta.Application.IsReplaceLegalInformationFilePath == false)
                meta.Application.LegalInformationFilePath = null;

            PrepareFsAccessControlData(meta.Core);

            // CrashReport をサポートしないなら書き出しをスキップ
            if (AppCapability.IsSupportCrashReport == false)
                meta.Application.CrashReport = null;

            PrepareSupportPlayLogs(meta);

            // RequiredNetworkServiceLicenseOnLaunch 非サポートの環境なら書き出しをスキップ
            if (AppCapability.IsSupportRequiredNetworkServiceLicenseOnLaunch == false)
                meta.Application.RequiredNetworkServiceLicenseOnLaunch = null;

            return meta;
        }

        private void PrepareSupportPlayLogs(ApplicationMeta meta)
        {
            if (!AppCapability.IsSupportPlayLogs)
            {
                // プレイ情報をサポートしないなら書き出しをスキップ
                meta.Application.PlayLogQueryCapability = null;
                meta.Application.PlayLogQueryableApplicationIds.Clear();
                return;
            }

            if (meta.Application.PlayLogQueryableApplicationIds.IsEmpty())
            {
                // プレイログ収集対象のアプリケーション ID が空の場合は None に変更
                meta.Application.PlayLogQueryCapability = PlayLogQueryCapabilityType.None;
                return;
            }

            // 次の条件すべて満たす場合 PlayLogQueryCapability を WhiteList に設定
            // * プレイ情報サポート環境
            // * プレイ情報収集対象のアプリ ID が設定済
            // * PlayLogQueryCapabilityType が null もしくは None
            if (meta.Application.PlayLogQueryCapability != PlayLogQueryCapabilityType.WhiteList &&
                meta.Application.PlayLogQueryCapability != PlayLogQueryCapabilityType.All)
            {
                meta.Application.PlayLogQueryCapability = PlayLogQueryCapabilityType.WhiteList;
            }
        }

        private void PrepareFsAccessControlData(Core core)
        {
            if (core.FsAccessControlData == null)
                return;

            if (AppCapability.IsSupportSaveDataOwnerIds == false)
                core.FsAccessControlData.SaveDataOwnerIds.Clear();

            // FlagPresets を使用する場合、その値を書き出すだけ
            if (core.FsAccessControlData.IsUseFlagPresets)
                return;

            if (core.FsAccessControlData.SaveDataOwnerIds.Any())
            {
                // FlagPresets は未指定だが SaveDataOwnerIds が存在するので、
                // FlagPresets の値を指定したことにしてデフォルト設定を使用
                core.FsAccessControlData.IsUseFlagPresets = true;
                core.FsAccessControlData.FlagPresets = FsAccessControlData.GetDefaultFlagPresets();
            }
            else
            {
                // FlagPresets も SaveDataOwnerIds も未指定なので XML 要素の出力を省略
                core.FsAccessControlData = null;
            }
        }

        private void FixupMeta()
        {
            var appMeta = Meta;
            var app = Meta.Application;

            app.IsUseApplicationErrorCode =
                !string.IsNullOrEmpty(app.ApplicationErrorCodeCategory);

            if (app.PresenceGroupId == 0ul)
                app.PresenceGroupId = appMeta.Core.ApplicationId;

            app.IsUsePresenceGroupIdAppId =
                app.PresenceGroupId == appMeta.Core.ApplicationId;

            app.IsUseSaveData =
                app.SaveDataSize > 0;

            app.IsSpecifiedSaveDataJournal =
                app.SaveDataJournalSize != app.SaveDataSize;

            app.IsSpecifiedDeviceSaveDataSize =
                app.DeviceSaveDataSize > 0;

            app.IsSpecifiedDeviceSaveDataJournalSize =
                app.DeviceSaveDataJournalSize > 0;

            if (AppCapability.IsSupportSeedForPseudoDeviceId)
            {
                if (app.SeedForPseudoDeviceId == 0ul)
                    app.SeedForPseudoDeviceId = appMeta.Core.ApplicationId;

                // SeedForPseudoDeviceId を指定する場合、値がアプリケーションと同値なら
                // ApplicationId と等しいことを表すフラグを立てる
                app.IsUseSeedForPseudoDeviceAppId = app.SeedForPseudoDeviceId == appMeta.Core.ApplicationId;
            }

            // SystemResourceSize は利用可能な場合にのみ読み書き可能
            appMeta.Core.IsUseSystemResourceSize = AppCapability.IsSupportSystemResourceSize;

            var icons = app.IconsFromMetaXml;

            foreach (var title in app.Titles)
            {
                var icon = icons?.FirstOrDefault(x => x.Language == title.Language);
                if (icon != null)
                {
                    title.IconFilePath = icon.IconPath;
                    title.NxIconFilePath = icon.NxIconPath;
                }
                // アイコンファイルが指定されていればデフォルトで置き換えフラグを立てる
                title.IsReplaceIcon = string.IsNullOrEmpty(title.IconFilePath) == false;
                title.IsReplaceNxIcon = false;
            }

            // アクセス可能な URL のリスト、オフライン HTML 、ソフトリーガル情報について、
            // それぞれ値が指定されていれば置き換えフラグを立てる
            if (string.IsNullOrEmpty(app.AccessibleUrlsFilePath) == false)
            {
                app.IsReplaceAccessibleUrlsFilePath = true;
            }
            if (string.IsNullOrEmpty(app.HtmlDocumentPath) == false)
            {
                app.IsReplaceHtmlDocumentPath = true;
            }
            if (string.IsNullOrEmpty(app.LegalInformationFilePath) == false)
            {
                app.IsReplaceLegalInformationFilePath = true;
            }

            if (AppCapability.IsSupportSaveDataOwnerIds == false)
            {
                appMeta.Core.FsAccessControlData.SaveDataOwnerIds.Clear();
            }

            {
                var defaultFlagPresets = FsAccessControlData.GetDefaultFlagPresets();
                var fs = appMeta.Core.FsAccessControlData;

                if (fs.FlagPresets == null)
                {
                    // FlagPresets の要素が存在しない場合は空文字列が指定されたとみなす
                    fs.IsUseFlagPresets = true;
                    fs.FlagPresets = string.Empty;
                }
                else
                {
                    fs.IsUseFlagPresets = fs.FlagPresets != defaultFlagPresets;
                }
            }

            if (AppCapability.IsSupportBcat)
            {
                // Bcat サポート環境なら、キャッシュサイズまたはパスフレーズが設定されていれば Bcat 使用フラグを立てる
                if (app.BcatDeliveryCacheStorageSize > 0 || string.IsNullOrEmpty(app.BcatPassphrase) == false)
                    app.IsUseBcat = true;
            }
            else
            {
                // Bcat 未サポートの環境なら Bcat の出力を常に省略する
                app.IsUseBcat = false;
            }

            // レーティング情報を団体名順に並び替え
            app.Ratings = app.Ratings.OrderBy(x => x.Organization).ToObservableCollection();

            // HDCP が利用できなければ値を強制的に空にする
            if (AppCapability.IsSupportHdcp == false)
                app.Hdcp = null;

            // AutohringEditor が属する SDK が VideoCapture をサポートするかどうか
            if (AppCapability.IsSupportVideoCapture)
            {
                // 設定値が null ならデフォルト値を設定
                if (app.VideoCapture.HasValue == false)
                    app.VideoCapture = VideoCaptureType.Enable;
            }

            if (AppCapability.IsSupportRuntimeAddOnContentInstall)
            {
                // 設定値が null ならデフォルト値を設定
                if (app.RuntimeAddOnContentInstall.HasValue == false)
                    app.RuntimeAddOnContentInstall = RuntimeAddOnContentInstallType.Deny;
            }
            else
            {
                // RuntmieAddOnContentInstall が利用できなければ値を強制的に空にする
                app.RuntimeAddOnContentInstall = null;
            }

            // 対応言語数に違いがあれば、AuthoringEditor 側が持つ言語一覧に合わせる
            FixupSupportedLanguages(app);

            // ファイルパスに対する環境変数の仕様の有無をチェック
            FixupExpandEnvironmentVariable(app);

            // クラッシュレポートの初期値を設定
            if (AppCapability.IsSupportCrashReport)
            {
                // 設定値が null ならデフォルト値 (Deny) を設定
                if (app.CrashReport.HasValue == false)
                    app.CrashReport = CrashReportType.Deny;
            }
            else
            {
                app.CrashReport = null;
            }

            // アプリケーション向けの一時セーブデータ領域のサポートの有無をチェック
            FixupSaveDataSizeExtend(app);
            FixupTempAndCacheStorage(app);

            // フィルタ記述ファイルの .nmeta 内への記載が非サポートなら要素をオミット
            if (AppCapability.IsSupportFilterDescriptionFileInMeta == false)
            {
                app.FilterDescriptionFilePath = null;
            }
            app.IsUseFilterDescriptionFilePath = !string.IsNullOrEmpty(app.FilterDescriptionFilePath);

            // プレイ情報の収集に関するサポート
            FixupPlayLogCapability(app);

            // レーティング情報
            FixupRating(app);

            // RequiredNetworkServiceLicenseOnLaunch 非サポートの環境なら要素をオミット
            if (AppCapability.IsSupportRequiredNetworkServiceLicenseOnLaunch == false)
            {
                app.RequiredNetworkServiceLicenseOnLaunch = null;
            }
        }

        private void FixupRating(Application app)
        {
            // nmeta に存在しないレーティング情報を IsUse = false で埋める
            var nmetaMissingOrganization = Constants.AllRatingOrganizations
                .Where(org => app.Ratings.Where(x => x.Organization == org).IsEmpty())
                .ToArray();

            foreach (var organization in nmetaMissingOrganization)
            {
                app.Ratings.Add(new Rating
                {
                    Organization = organization,
                    IsUse = false
                });
            }
        }

        private void FixupPlayLogCapability(Application app)
        {
            if (AppCapability.IsSupportPlayLogs)
            {
                // 設定値が null ならデフォルト値を設定
                if (app.PlayLogQueryCapability.HasValue == false)
                    app.PlayLogQueryCapability = PlayLogQueryCapabilityType.None;
                // アプリケーション ID のリストから重複を排除し、ID が 0 のものを除外したリストを保持
                app.PlayLogQueryableApplicationIds = app.PlayLogQueryableApplicationIds.Where(x => x.Id != 0x0ul).Distinct().ToObservableCollection();
            }
            else
            {
                // プレイ情報の設定が利用できなければ値を強制的に空にする
                app.PlayLogQueryCapability = null;
                app.PlayLogQueryableApplicationIds.Clear();
            }
        }

        private void FixupExpandEnvironmentVariable(Application app)
        {
            if (AppCapability.IsSupportEnvironmentVariableToPath)
                return;

            // AoC の DataPath を除いてファイルパスの仕様が許可されていないので、nmeta から読み込んだ結果を上書き
            // (環境変数を使用していないことにする)

            app.HtmlDocumentPath.IsExpandEnvironmentVariable = false;
            app.AccessibleUrlsFilePath.IsExpandEnvironmentVariable = false;
            app.LegalInformationFilePath.IsExpandEnvironmentVariable = false;

            foreach (var title in app.Titles)
            {
                title.IconFilePath.IsExpandEnvironmentVariable = false;
            }
        }

        private static void FixupSupportedLanguages(Application app)
        {
            var languageTypes = Enum.GetValues(typeof(LanguageType)).Cast<LanguageType>().ToArray();
            if (languageTypes.Length == app.SupportedLanguages.Count)
            {
                return;
            }

            var newSupportedLanguages = new List<SupportedLanguage>();
            foreach (var language in languageTypes)
            {
                var supported = app.SupportedLanguages.FirstOrDefault(x => x.Language == language);
                newSupportedLanguages.Add(new SupportedLanguage
                {
                    Language = language,
                    IsSupported = supported?.IsSupported ?? false,
                    HasTitle = supported?.HasTitle ?? false
                });
            }
            app.SupportedLanguages.Clear();
            newSupportedLanguages.AddRangeTo(app.SupportedLanguages);
        }

        private void FixupFilePath()
        {
            // ApplicationMeta
            {
                var app = Meta.Application;

                app.AccessibleUrlsFilePath = ToRealtivePath(app.AccessibleUrlsFilePath);
                app.HtmlDocumentPath = ToRealtivePath(app.HtmlDocumentPath);
                app.LegalInformationFilePath = ToRealtivePath(app.LegalInformationFilePath);

                foreach (var title in app.Titles)
                {
                    title.IconFilePath = ToRealtivePath(title.IconFilePath);
                    title.NxIconFilePath = ToRealtivePath(title.NxIconFilePath);
                }
            }

            // AocMeta
            {
                var aoc = AocMeta;
                foreach (var content in aoc.Contents)
                {
                    content.DataPath = ToRealtivePath(content.DataPath);
                }
            }
        }

        private static void FixupMetaFromMetaFile(ApplicationMeta appMeta, string xmlText)
        {
            using (var reader = new StringReader(xmlText))
            {
                var doc = XDocument.Load(reader);

                appMeta.CardSpec.IsAutomaticSettingSize =
                    doc.Root?.XPathSelectElements("CardSpec/Size").IsEmpty() == true;
                appMeta.CardSpec.IsAutomaticSettingClockRate =
                    doc.Root?.XPathSelectElements("CardSpec/ClockRate").IsEmpty() == true;
            }
        }

        private void FixupSaveDataSizeExtend(Application app)
        {
            if (AppCapability.IsSupportSaveDataSizeExtend == false)
                return;
            app.IsUseUserAccountSaveDataSizeMax = app.UserAccountSaveDataSizeMax > 0;
            app.IsUseUserAccountSaveDataJournalSizeMax = app.UserAccountSaveDataJournalSizeMax > 0;

            app.IsUseDeviceSaveDataSizeMax = app.DeviceSaveDataSizeMax > 0;
            app.IsUseDeviceSaveDataJournalSizeMax = app.DeviceSaveDataJournalSizeMax > 0;
        }

        private void FixupTempAndCacheStorage(Application app)
        {
            if (AppCapability.IsSupportTempAndCacheStorage == false)
                return;
            app.IsUseTemporaryStorageSize = app.TemporaryStorageSize > 0;
            app.IsUseCacheStorageSize = app.CacheStorageSize > 0;
            app.IsUseCacheStorageJournalSize = app.CacheStorageJournalSize > 0;
        }


        private static Project FromProjectFile(PathString filePath)
        {
            Project project = null;
            try
            {
                project = YamlHelper.LoadFromString<Project>(File.ReadAllText(filePath));
            }
            catch
            {
                // ignore
            }
            if (project == null)
                throw new FileFormatException();

            return project;
        }

        private static ApplicationMeta ApplicationMetaFromMetaXml(string xmlText)
        {
            var rootMetaKind = TypeHelper.DetectRootMetaKindFromXmlText(xmlText);

            ApplicationMeta appMeta;

            using (var reader = new StringReader(xmlText))
            {

                appMeta = (ApplicationMeta)new XmlSerializer(typeof(ApplicationMeta), new XmlRootAttribute
                {
                    ElementName = rootMetaKind.ToString(),
                }).Deserialize(reader);
            }

            return appMeta;
        }

        private static AocMeta AocMetaFromMetaXml(Container diContainer, string xmlText)
        {
            using (var reader = new StringReader(xmlText))
            {
                var rootMetaKind = TypeHelper.DetectRootMetaKindFromXmlText(xmlText);
                var loadedData =
                    (Serialization.AocMeta)new XmlSerializer(typeof(Serialization.AocMeta), new XmlRootAttribute
                    {
                        ElementName = rootMetaKind.ToString()
                    }).Deserialize(reader);

                var aocMeta = new AocMeta
                {
                    ApplicationId = loadedData.AddOnContent.Any()
                        ? loadedData.AddOnContent[0].ApplicationId.ToUlong()
                        : Constants.DefaultMetaProgramId
                };

                aocMeta.Contents =
                    loadedData.AddOnContent.Select(x => new AocContent(diContainer, aocMeta)
                    {
                        Index = x.Index,
                        ReleaseVersion = x.ReleaseVersion,
                        RequiredApplicationReleaseVersion = x.RequiredApplicationReleaseVersion,
                        DataPath = x.DataPath,
                        Tag = x.Tag
                    }).ToObservableCollection();

                return aocMeta;
            }
        }

        private static Project FromNspFileOrDirectory(PathString filePath, PathString originalAppNspFilePath, PathString tempDirPath, Container diContainer)
        {
            var nspFile = GetNspFile(filePath);
            if (nspFile == null)
                throw new FileNotFoundException();
            nspFile.OriginalNspFile = GetNspFile(originalAppNspFilePath);

            var importer = new NspImporter();
            var r = importer.ToMetaAsync(new NspImporter.Option
            {
                DiContainer = diContainer,
                InputNspFile = nspFile,
                TempNspExtractFilePath = tempDirPath,
            }).Result;

            if (r.Result != NspHandleResult.Ok)
                throw new NspHandleErrorException(r);

            var project = new Project
            {
                Meta = r.ApplicationMeta,
                AocMeta = r.AocMeta,
                ApplicationContentMeta = r.ApplicationContentMeta,
                PatchContentMeta = r.PatchContentMeta,
                AocContentMetas = r.AocContentMetas?.ToObservableCollection(),
                ProgramInfo = r.ProgramInfo,
                ContentMetaProperties = r.ContentMetaProperties,
                ExtraResourceImporter = r.ExtraResourceImporter,
                NspFile = nspFile
            };
            return project;
        }

    }
}
