﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Xml.Linq;
using BezelEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Foundation;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public static class Constants
    {
        public static readonly CultureSummry[] SupportedCultures;

        public static string ConfigFilePath => Path.Combine(ConfigDirPath, "config.yaml");

        public static string ConfigDirPath
        {
            get
            {
                var appData = Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData);
                return Path.Combine(appData, @"Nintendo\AuthoringEditor");
            }
        }

        public static string[] AllRatingOrganizations { get; }

        public static readonly Dictionary<string /*RatingOrganization*/, RatingData> AllRatingData
            = new Dictionary<string, RatingData>();

        public static readonly Dictionary<TargetSalesRegion, RatingData[]> PresetTargetSalesRegionRatingData;

        public static readonly ulong ProgramIdMinimum;
        public static readonly ulong ProgramIdMaximum;
        public static readonly ulong DefaultMetaProgramId;

        public const ulong AocIndexMinimum = 1;
        public const ulong AocIndexMaximum = 2000;

        static Constants()
        {
            #region カルチャー

            {
                var dir = Path.GetDirectoryName(Assembly.GetEntryAssembly()?.Location) ?? string.Empty;

                if (dir != string.Empty)
                {
                    var resFiles = Directory.EnumerateFiles(dir, "AuthoringEditor.resources.dll",
                        SearchOption.AllDirectories);

                    SupportedCultures = resFiles.Select(f =>
                        {
                            var name = Path.GetFileName(Path.GetDirectoryName(f));
                            if (name == null)
                                return null;

                            return new CultureSummry
                            {
                                Caption = CultureInfo.GetCultureInfo(name).NativeName,
                                CultureName = name
                            };
                        })
                        .Where(x => x != null)
                        .ToArray();

                    if (SupportedCultures.IsEmpty())
                    {
                        SupportedCultures = new[]
                        {
                            new CultureSummry
                            {
                                Caption = "Default",
                                CultureName = string.Empty
                            }
                        };
                    }
                }
                else
                {
                    SupportedCultures = new[]
                    {
                        new CultureSummry
                        {
                            Caption = "ja",
                            CultureName = "ja"
                        },
                        new CultureSummry
                        {
                            Caption = "en",
                            CultureName = "en"
                        }
                    };
                }
            }

            #endregion

            #region リージョン

            {
                string defaultRatingDataFile;
                try
                {
                    defaultRatingDataFile =
                        string.IsNullOrEmpty(AssemblyConstants.Directory)
                        ? string.Empty
                        : Path.Combine(AssemblyConstants.Directory, "RatingData.yml");
                }
                catch
                {
                    defaultRatingDataFile = string.Empty;
                }

                // データファイルがなければデフォルトのものを使う
                var ratingData = File.Exists(defaultRatingDataFile)
                    ? YamlHelper.Load<RatingData[]>(defaultRatingDataFile, () => null)
                    : YamlHelper.Load<RatingData[]>(Properties.Resources.RatingData, () => null);

                foreach (var r in ratingData)
                {
                    // RatingData.yml の年齢区分データから、同一の AgeValue の区分を集約したカテゴリのリストを作成する
                    r.AgeAggregatedCategories = GetAggregatedRatingCategoryData(r);
                    AllRatingData[r.RatingOrganization] = r;
                }

                AllRatingOrganizations = ratingData.Select(r => r.RatingOrganization).ToArray();

                // 仕向け地ごとのレーティング団体の一覧をグループ化
                PresetTargetSalesRegionRatingData =
                    AllRatingData
                        .GroupBy(x => x.Value.TargetSalesRegion, x => x.Value)
                        .ToDictionary(x => x.Key, x => x.ToArray());
            }

            #endregion

            #region ProgramIdMinimum, ProgramIdMaximum

            {
                ProgramIdMinimum = ulong.MinValue;
                ProgramIdMaximum = ulong.MaxValue;
                DefaultMetaProgramId = ProgramIdMaximum;

                try
                {
                    if (!string.IsNullOrEmpty(NintendoSdkHelper.ApplicationDescFilePath))
                    {
                        var doc = XDocument.Parse(File.ReadAllText(NintendoSdkHelper.ApplicationDescFilePath));

                        var min = doc.Descendants("ProgramIdMin").FirstOrDefault()?.Value;
                        var max = doc.Descendants("ProgramIdMax").FirstOrDefault()?.Value;

                        ProgramIdMinimum = min?.ToUlong() ?? ulong.MinValue;
                        ProgramIdMaximum = max?.ToUlong() ?? ulong.MaxValue;
                    }

                    if (!string.IsNullOrEmpty(NintendoSdkHelper.ApplicationMetaFilePath))
                    {
                        var doc = XDocument.Parse(File.ReadAllText(NintendoSdkHelper.ApplicationMetaFilePath));
                        var applicationId = doc.Descendants("ApplicationId").FirstOrDefault()?.Value;

                        if (applicationId?.IsUlong() == true)
                        {
                            DefaultMetaProgramId = applicationId.ToUlong();
                        }
                    }
                }
                catch
                {
                    // ignored
                }
            }

            #endregion
        }

        private static List<AggregatedRatingCategoryData> GetAggregatedRatingCategoryData(RatingData r)
        {
            var categories = new List<AggregatedRatingCategoryData>();
            foreach (var ageCategories in r.Categories.GroupBy(x => x.AgeValue))
            {
                categories.Add(new AggregatedRatingCategoryData
                {
                    AgeValue = ageCategories.Key,
                    Names = GetAggregatedNames(ageCategories),
                    RatingCategories = ageCategories.Select(x => x.RatingCategory).ToArray()
                });
            }
            return categories;
        }

        private static Dictionary<string, List<string>> GetAggregatedNames(IGrouping<int, RatingCategoryData> ageCategories)
        {
            var aggregatedNames = new Dictionary<string, List<string>>();
            foreach (var c in ageCategories)
            {
                foreach (var t in c.Names)
                {
                    List<string> names;
                    if (!aggregatedNames.TryGetValue(t.Key, out names))
                    {
                        names = aggregatedNames[t.Key] = new List<string>();
                    }
                    names.Add(t.Value);
                }
            }
            return aggregatedNames;
        }
    }
}
