﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using Reactive.Bindings.Notifiers;
using System;
using System.Linq;
using System.Reactive.Linq;
using System.Threading.Tasks;
using Microsoft.WindowsAPICodePack.Dialogs;
using System.IO;
using System.Timers;
using System.Text;
using System.ComponentModel;
using System.Collections.Generic;
using System.Windows.Controls;

namespace SystemAudioMonitor
{
    public class MainWindowViewModel : IDisposable
    {
        public ReactiveProperty<RecordingState> State { get; } = new ReactiveProperty<RecordingState>(RecordingState.NoConnection);

        public ReactiveProperty<string> DirectoryPath { get; } = new ReactiveProperty<string>();

        public ReactiveProperty<double> PlaybackVolume { get; } = new ReactiveProperty<double>(1.0);
        public double PlaybackVolumeMin { get; } = 0.0;
        public double PlaybackVolumeMax { get; } = 1.0;

        public ReactiveProperty<double> ShortTermLoudness { get; }
        public ReactiveProperty<double> MomentaryLoudness { get; }
        public ReactiveProperty<double> IntegratedLoudness { get; }

        public ReactiveProperty<double[]> FlPcmSamples { get; }
        public ReactiveProperty<double[]> FrPcmSamples { get; }
        public ReactiveProperty<double[]> FcPcmSamples { get; }
        public ReactiveProperty<double[]> LfePcmSamples { get; }
        public ReactiveProperty<double[]> RlPcmSamples { get; }
        public ReactiveProperty<double[]> RrPcmSamples { get; }
        public ReactiveProperty<string[]> AvailableTargets { get; } = new ReactiveProperty<string[]>();

        public ReactiveProperty<string> TargetName { get; } = new ReactiveProperty<string>();
        public ReactiveProperty<string> AdspLoad { get; }
        public ReactiveProperty<bool> IsAudioVolumeEnabled { get; } = new ReactiveProperty<bool>(false);
        public ReactiveProperty<bool> IsTargetListEnabled { get; } = new ReactiveProperty<bool>(false);

        public ReactiveCommand SelectDirectoryCommand { get; }
        public ReactiveCommand OpenDirectoryCommand { get; }
        public ReactiveCommand RecordCommand { get; }
        public ReactiveCommand StopCommand { get; }
        public ReactiveCommand ConnectCommand { get; }
        public ReactiveCommand DisconnectCommand { get; }
        public ReactiveCommand StartLoudnessLoggerCommand { get; }
        public ReactiveCommand ShowTargetListCommand { get; }
        public ReactiveCommand ShowAudioVolumeCommand { get; }
        public ReactiveCommand ConnectToTargetCommand { get; }

        private BusyNotifier Busy { get; } = new BusyNotifier();

        public Action<object, Exception> ConnectionError { get; set; }
        public Action<object, Exception> MonitoringError { get; set; }

        private readonly int WaveformSampleCount = 4096;
        private WaveformIndicator _waveformIndicator;

        private StreamWriter _loudnessLogger;
        private Timer _loudnessLoggerTimer;

        private WaveRecorder _recorder;
        private WaveReceiver _receiver;
        private WavePlayback _playback;
        private LoudnessIndicator _loudnessIndicator;

        private void OnPlaybackVolumeChanged(object sender, PropertyChangedEventArgs e)
        {
            if(_playback != null)
            {
                _playback.Volume = (float)PlaybackVolume.Value;
            }
        }

        public MainWindowViewModel()
        {
#if (DEBUG_CULTURE_EN)
            Resources.ExceptionMessages.Culture = System.Globalization.CultureInfo.GetCultureInfo("en");
            Resources.MainWindowResources.Culture = System.Globalization.CultureInfo.GetCultureInfo("en");
            Resources.RecordingStateMessages.Culture = System.Globalization.CultureInfo.GetCultureInfo("en");
#endif

            PlaybackVolume.PropertyChanged += OnPlaybackVolumeChanged;

            SelectDirectoryCommand = new[]
            {
                State.Select(x => x == RecordingState.Recording),
                State.Select(x => x == RecordingState.Stopping),
                Busy.ToReactiveProperty()
            }.CombineLatestValuesAreAllFalse().ToReactiveCommand();
            SelectDirectoryCommand.Subscribe(_ => SelectDirectory());

            OpenDirectoryCommand = DirectoryPath.Select(x => x != null && Directory.Exists(x)).ToReactiveCommand();
            OpenDirectoryCommand.Subscribe(_ => OpenDirectory());

            ConnectCommand = new[]
            {
                State.Select(x => x == RecordingState.NoConnection),
                Busy.Inverse().ToReactiveProperty()
            }.CombineLatestValuesAreAllTrue().ToReactiveCommand();
            ConnectCommand.Subscribe(async _ => await Connect());

            DisconnectCommand = new[]
            {
                State.Select(x => x == RecordingState.Connecting || x == RecordingState.Connected),
                Busy.Inverse().ToReactiveProperty()
            }.CombineLatestValuesAreAllTrue().ToReactiveCommand();
            DisconnectCommand.Subscribe(async _ =>
            {
                using (Busy.ProcessStart())
                {
                    await Disconnect();
                }
            });

            RecordCommand = new[]
            {
                State.Select(x => x == RecordingState.Connected),
                DirectoryPath.Select(x => !String.IsNullOrEmpty(x) && Path.IsPathRooted(x)),
                Busy.Inverse().ToReactiveProperty()
            }.CombineLatestValuesAreAllTrue().ToReactiveCommand();
            RecordCommand.Subscribe(_ =>
            {
                Record();
            });

            StopCommand = new[]
            {
                State.Select(x => x == RecordingState.Recording),
                Busy.Inverse().ToReactiveProperty()
            }.CombineLatestValuesAreAllTrue().ToReactiveCommand();
            StopCommand.Subscribe(async _ =>
            {
                using (Busy.ProcessStart())
                {
                    await Stop();
                }
            });

            StartLoudnessLoggerCommand = new ReactiveCommand();
            StartLoudnessLoggerCommand.Subscribe(_ => StartLoudnessLogger());

            ShowTargetListCommand = new[]
            {
                State.Select(x => x == RecordingState.NoConnection),
                Busy.Inverse().ToReactiveProperty()
            }.CombineLatestValuesAreAllTrue().ToReactiveCommand();
            ShowTargetListCommand.Subscribe(_ => ShowTargetList());

            ShowAudioVolumeCommand = new ReactiveCommand();
            ShowAudioVolumeCommand.Subscribe(_ => ShowAudioVolume());

            ConnectToTargetCommand = new ReactiveCommand();
            ConnectToTargetCommand.Subscribe(s => ConnectToTarget(s));

            ShortTermLoudness = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetShortTermLoudness()).ToReactiveProperty();
            MomentaryLoudness = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetMomentaryLoudness()).ToReactiveProperty();
            IntegratedLoudness = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetIntegratedLoudness()).ToReactiveProperty();

            AdspLoad = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetAdspLoad() >= 0 ? "(adsp load: " + GetAdspLoad() + "%)" : string.Empty).ToReactiveProperty();

            FlPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.FrontLeft)).ToReactiveProperty();
            FrPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.FrontRight)).ToReactiveProperty();
            FcPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.FrontCenter)).ToReactiveProperty();
            LfePcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.LowFrequency)).ToReactiveProperty();
            RlPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.RearLeft)).ToReactiveProperty();
            RrPcmSamples = Observable.Interval(TimeSpan.FromMilliseconds(50)).Select(_ => GetWaveform(WaveformSampleCount, WaveformIndicator.ChannelIndex_5_1.RearRight)).ToReactiveProperty();
        }

        double[] GetWaveform(int samples, WaveformIndicator.ChannelIndex_5_1 channelIndex)
        {
            if(_waveformIndicator == null)
            {
                return new double[samples];
            }
            return _waveformIndicator.GetWaveform(channelIndex);
        }

        private double GetShortTermLoudness()
        {
            if(_loudnessIndicator == null)
            {
                return double.NegativeInfinity;
            }
            return _loudnessIndicator.ShortTermLoudness;
        }

        private double GetMomentaryLoudness()
        {
            if (_loudnessIndicator == null)
            {
                return double.NegativeInfinity;
            }
            return _loudnessIndicator.MomentaryLoudness;
        }

        private double GetIntegratedLoudness()
        {
            if (_loudnessIndicator == null)
            {
                return double.NegativeInfinity;
            }
            return _loudnessIndicator.IntegratedLoudness;
        }

        private int GetAdspLoad()
        {
            return _receiver == null ? -1 : _receiver.AdspLoad;
        }

        private void SelectDirectory()
        {
            var dialog = new CommonOpenFileDialog();
            dialog.Title = Resources.MainWindowResources.DirectoryOpenDialogTitle;
            dialog.IsFolderPicker = true;
            dialog.EnsureReadOnly = false;
            dialog.AllowNonFileSystemItems = false;
            CommonFileDialogResult result = dialog.ShowDialog();
            if (result == CommonFileDialogResult.Ok)
            {
                DirectoryPath.Value = dialog.FileName;
            }
        }

        private void OpenDirectory()
        {
            if (DirectoryPath.Value != null && Directory.Exists(DirectoryPath.Value))
            {
                var explorer = new System.Diagnostics.Process();
                explorer.StartInfo.FileName = "explorer";
                explorer.StartInfo.Arguments = DirectoryPath.Value;
                explorer.Start();
            }
        }

        private void StartLoudnessLogger()
        {
            var dialog = new CommonOpenFileDialog();
            dialog.Title = Resources.MainWindowResources.DirectoryOpenDialogTitle;
            dialog.IsFolderPicker = false;
            dialog.EnsureReadOnly = false;
            dialog.AllowNonFileSystemItems = false;
            CommonFileDialogResult result = dialog.ShowDialog();
            if (result == CommonFileDialogResult.Ok)
            {
                _loudnessLoggerTimer?.Dispose();
                _loudnessLogger?.Dispose();

                _loudnessLogger = new StreamWriter(dialog.FileName, false, Encoding.ASCII);
                _loudnessLoggerTimer = new Timer(1000);  // 1000 msec
                _loudnessLoggerTimer.Elapsed += (sender, e) =>
                {
                    _loudnessLogger.WriteLine("[" + DateTime.Now.ToString() + "] " + ShortTermLoudness.Value.ToString() + ", " + MomentaryLoudness.Value.ToString() + ", " + IntegratedLoudness.Value.ToString());
                    _loudnessLogger.Flush();
                };
                _loudnessLoggerTimer.Start();
            }
        }

        private void ShowTargetList()
        {
            var names = new List<string>();
            names.Add(Resources.MainWindowResources.DefaultTargetString);
            try
            {
                var targets = Nintendo.Tm.TargetManager.GetTargets();
                foreach (var target in targets)
                {
                    names.Add(target.GetName());
                }
            }
            catch
            {
            }
            AvailableTargets.Value = names.ToArray();
            IsTargetListEnabled.Value = true;
        }

        private void ShowAudioVolume()
        {
            IsAudioVolumeEnabled.Value = true;
        }

        private async void ConnectToTarget(object targetName)
        {
            var name = targetName as string;
            if (name == Resources.MainWindowResources.DefaultTargetString)
            {
                await Connect();
            }
            else
            {
                await Connect(name);
            }
            IsTargetListEnabled.Value = false;
        }

        private void Shutdown()
        {
            if (_recorder != null)
            {
                Task.Run(async () => await Stop()).Wait();
            }
            if (_receiver != null)
            {
                Task.Run(async () => await Disconnect()).Wait();
            }
        }

        private void OnCaptureStopped(object sender, Exception e)
        {
            if(e != null)
            {
                MonitoringError?.Invoke(sender, e);
                Shutdown();
            }
        }

        private async Task<bool> Connect(string name = null)
        {
            // TODO: 接続時のエラーメッセージを受け取る
            State.Value = RecordingState.Connecting;

            _receiver = new WaveReceiver(name);

            try
            {
                Nintendo.Tm.Target connectTarget = null;
                if (name == null)
                {
                    var result = await Task.Run(() => Nintendo.Tm.TargetManager.TryGetDefaultTarget(out connectTarget));
                    if (!result)
                    {
                        State.Value = RecordingState.NoConnection;
                        throw new Exception(Resources.ExceptionMessages.DefaultTargetNotFoundException);
                    }
                }
                else
                {
                    var targets = await Task.Run(() => Nintendo.Tm.TargetManager.GetTargets());
                    connectTarget = targets.Where(x => x.GetName() == name).FirstOrDefault();
                    if (connectTarget == null)
                    {
                        State.Value = RecordingState.NoConnection;
                        throw new Exception(Resources.ExceptionMessages.DefaultTargetNotFoundException);
                    }
                }
                TargetName.Value = connectTarget.GetName();
                await Task.Run(() => { connectTarget.Connect(); });
            }
            catch(Exception e)
            {
                Console.WriteLine("{0}", e.Message);
                State.Value = RecordingState.NoConnection;
                ConnectionError?.Invoke(this, e);
                return false;
            }

            _loudnessIndicator = new LoudnessIndicator(_receiver.Format);
            _receiver.AddWaveProcessor(_loudnessIndicator);

            try
            {
                _waveformIndicator = new WaveformIndicator(_receiver.Format, WaveformSampleCount);
            }
            catch (Exception e)
            {
                Console.WriteLine("{0}", e.Message);
                Shutdown();
                ConnectionError?.Invoke(this, e);
                return false;
            }
            _receiver.AddWaveProcessor(_waveformIndicator);

            try
            {
                _playback = new WavePlayback(_receiver.Format);
            }
            catch (Exception e)
            {
                Console.WriteLine("{0}", e.Message);
                Shutdown();
                ConnectionError?.Invoke(this, e);
                return false;
            }
            _playback.Volume = (float) PlaybackVolume.Value;
            _receiver.AddWaveProcessor(_playback);

            _receiver.CaptureStopped += OnCaptureStopped;

            try
            {
                await _receiver.Start(name);
                State.Value = RecordingState.Connected;
            }
            catch (Exception e)
            {
                Console.WriteLine("{0}", e.Message);
                Shutdown();
                ConnectionError?.Invoke(this, e);
                return false;
            }

            return true;
        }

        private async Task<bool> Disconnect()
        {
            var result = await Task.Run(() =>
            {
                _receiver?.RemoveWaveProcessor(_waveformIndicator);
                _waveformIndicator = null;

                _receiver?.RemoveWaveProcessor(_playback);
                _playback?.Dispose();
                _playback = null;

                _receiver?.RemoveWaveProcessor(_loudnessIndicator);
                _loudnessIndicator = null;

                _receiver?.Stop();
                _receiver = null;

                State.Value = RecordingState.NoConnection;
                return true;
            });

            TargetName.Value = string.Empty;

            return result;
        }

        private bool Record()
        {
            try
            {
                _recorder = new WaveRecorder(_receiver.Format, DirectoryPath.Value);
            }
            catch(Exception e)
            {
                Console.WriteLine("{0}", e.Message);
                ConnectionError?.Invoke(this, e);
                return false;
            }
            _receiver?.AddWaveProcessor(_recorder);
            State.Value = RecordingState.Recording;
            return true;
        }

        private async Task<bool> Stop()
        {
            var result = await Task.Run(() =>
            {
                State.Value = RecordingState.Stopping;
                _receiver?.RemoveWaveProcessor(_recorder);
                _recorder?.Dispose();
                _recorder = null;
                return true;
            });

            State.Value = RecordingState.Connected;
            return result;
        }

        public void Dispose()
        {
            Shutdown();
        }
    }
}
