﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;

namespace SpatialAudioEvaluationTool
{
    class WaveformIndicator : IWaveProcessor
    {
        private double[][] _waveform;
        private WaveformFormat _format;
        private int _samples;

        public enum ChannelIndex_5_1
        {
            FrontLeft = 0,
            FrontRight = 1,
            FrontCenter = 2,
            LowFrequency = 3,
            RearLeft = 4,
            RearRight = 5
        }

        public WaveformIndicator(WaveformFormat format, int samples)
        {
            _format = format;
            _samples = samples;
            _waveform = new double[_format.ChannelCount][];
            for (int i = 0; i < _format.ChannelCount; ++i)
            {
                _waveform[i] = new double[samples];
            }
        }

        public double[] GetWaveform(ChannelIndex_5_1 channelIndex)
        {
            return _waveform[(int)channelIndex];
        }

        private double[][] AppendWaveform(double[][] data)
        {
            var result = new double[_format.ChannelCount][];
            for (int i = 0; i < _format.ChannelCount; ++i)
            {
                result[i] = new double[_samples];
            }
            for (var channelIndex = 0; channelIndex < _format.ChannelCount; ++channelIndex)
            {
                var channelData = data[channelIndex];
                var waveformData = _waveform[channelIndex];
                var resultChannelData = result[channelIndex];

                if (channelData.Length < waveformData.Length)
                {
                    Array.Copy(waveformData, channelData.Length, resultChannelData, 0, resultChannelData.Length - channelData.Length);
                    Array.Copy(channelData, 0, resultChannelData, resultChannelData.Length - channelData.Length, channelData.Length);
                }
                else
                {
                    Array.Copy(channelData, channelData.Length - resultChannelData.Length, resultChannelData, 0, resultChannelData.Length);
                }
            }

            return result;
        }

        private double[][] GetChannelInterleavedWaveform(byte[] buffer, int offset, int count)
        {
            int sampleBytes = _format.BitDepth / 8;
            var result = new double[_format.ChannelCount][];
            for (int i = 0; i < _format.ChannelCount; ++i)
            {
                result[i] = new double[(count - offset) / _format.ChannelCount / sampleBytes];
            }
            for (int i = 0; i < (count - offset) / sampleBytes; ++i)
            {
                int channelIndex = i % _format.ChannelCount;
                int sampleIndex = i / _format.ChannelCount;

                switch(_format.BitDepth)
                {
                    case 16:
                        result[channelIndex][sampleIndex] = (double)(BitConverter.ToInt16(buffer, offset + i * sizeof(Int16))) / Int16.MaxValue;
                        break;
                    default:
                        // 現状は 16 bit のみサポート
                        Debug.Assert(false);
                        break;
                }
            }
            return result;
        }

        public void AddSamples(byte[] buffer, int offset, int count)
        {
            var sampleData = GetChannelInterleavedWaveform(buffer, offset, count);
            _waveform = AppendWaveform(sampleData);
        }

        public void OnCaptureStopped(Exception e)
        {
        }
    }
}
